/*!*
 *
 *  Copyright (c) Highsoft AS. All rights reserved.
 *
 *!*/
import * as globals from "./globals";
/**
 * Formatter callback for the accessibility announcement.
 *
 * @param updatedSeries
 *        Array of all series that received updates. If an announcement is
 *        already queued, the series that received updates for that announcement
 *        are also included in this array.
 *
 * @param addedSeries
 *        This is provided if Highcharts.Chart#addSeries was called, and there
 *        is a new series. In that case, this argument is a reference to the new
 *        series.
 *
 * @param addedPoint
 *        This is provided if Highcharts.Series#addPoint was called, and there
 *        is a new point. In that case, this argument is a reference to the new
 *        point.
 *
 * @return The function should return a string with the text to announce to the
 *         user. Return empty string to not announce anything. Return `false` to
 *         use the default announcement format.
 */
export type AccessibilityAnnouncementFormatter = (updatedSeries: Array<Series>, addedSeries?: Series, addedPoint?: Point) => (false|string);
/**
 * The horizontal alignment of an element.
 */
export type AlignValue = ("center"|"left"|"right");
/**
 * Creates a frame for the animated SVG element.
 *
 * @param this
 *        The SVG element to animate.
 */
export type AnimationStepCallbackFunction = (this: SVGElement) => void;
/**
 * Callback to modify annotation's possitioner controls.
 */
export type AnnotationControlPointPositionerFunction = (this: AnnotationControlPoint, target: AnnotationControllable) => PositionObject;
/**
 * Possible directions for draggable annotations. An empty string (`''`) makes
 * the annotation undraggable.
 */
export type AnnotationDraggableValue = (''|'x'|'xy'|'y');
export type AxisEventCallbackFunction = (this: Axis) => void;
export type AxisExtremesTriggerValue = ("navigator"|"pan"|"scrollbar"|"zoom"|"rangeSelectorButton"|"rangeSelectorInput"|"traverseUpButton");
export type AxisLabelsFormatterCallbackFunction = (this: AxisLabelsFormatterContextObject<number>, that: AxisLabelsFormatterContextObject<string>) => string;
/**
 * Options for axes.
 */
export type AxisOptions = (XAxisOptions|YAxisOptions|ZAxisOptions);
/**
 * Options for plot band labels on axes.
 */
export type AxisPlotBandsLabelOptions = (XAxisPlotBandsLabelOptions|YAxisPlotBandsLabelOptions|ZAxisPlotBandsLabelOptions);
/**
 * Options for plot bands on axes.
 */
export type AxisPlotBandsOptions = (XAxisPlotBandsOptions|YAxisPlotBandsOptions|ZAxisPlotBandsOptions);
/**
 * Options for plot line labels on axes.
 */
export type AxisPlotLinesLabelOptions = (XAxisPlotLinesLabelOptions|YAxisPlotLinesLabelOptions|ZAxisPlotLinesLabelOptions);
/**
 * Options for plot lines on axes.
 */
export type AxisPlotLinesOptions = (XAxisPlotLinesOptions|YAxisPlotLinesOptions|ZAxisPlotLinesOptions);
export type AxisPointBreakEventCallbackFunction = (this: Axis, evt: AxisPointBreakEventObject) => void;
export type AxisSetExtremesEventCallbackFunction = (this: Axis, evt: AxisSetExtremesEventObject) => void;
export type AxisTickPositionerCallbackFunction = (this: Axis) => AxisTickPositionsArray;
export type AxisTitleAlignValue = ("high"|"low"|"middle");
export type AxisTitleOptions = (XAxisTitleOptions|YAxisTitleOptions|ZAxisTitleOptions);
export type AxisTypeValue = ("category"|"datetime"|"linear"|"logarithmic"|"treegrid");
export type BubbleSizeByValue = ("area"|"width");
export type ButtonRelativeToValue = ("plotBox"|"spacingBox");
/**
 * Gets fired when a series is added to the chart after load time, using the
 * `addSeries` method. Returning `false` prevents the series from being added.
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ChartAddSeriesCallbackFunction = (this: Chart, event: ChartAddSeriesEventObject) => void;
/**
 * Callback for chart constructors.
 *
 * @param chart
 *        Created chart.
 */
export type ChartCallbackFunction = (chart: Chart) => void;
/**
 * Gets fired when clicking on the plot background.
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ChartClickCallbackFunction = (this: Chart, event: PointerEventObject) => void;
/**
 * Gets fired when the chart is finished loading.
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ChartLoadCallbackFunction = (this: Chart, event: Event) => void;
/**
 * Fires when the chart is redrawn, either after a call to `chart.redraw()` or
 * after an axis, series or point is modified with the `redraw` option set to
 * `true`.
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ChartRedrawCallbackFunction = (this: Chart, event: Event) => void;
/**
 * Gets fired after initial load of the chart (directly after the `load` event),
 * and after each redraw (directly after the `redraw` event).
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ChartRenderCallbackFunction = (this: Chart, event: Event) => void;
/**
 * Gets fired when an area of the chart has been selected. The default action
 * for the selection event is to zoom the chart to the selected area. It can be
 * prevented by calling `event.preventDefault()` or return false.
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        Event informations
 *
 * @return Return false to prevent the default action, usually zoom.
 */
export type ChartSelectionCallbackFunction = (this: Chart, event: ChartSelectionContextObject) => (boolean|undefined);
/**
 * A clipping rectangle that can be applied to one or more SVGElement instances.
 * It is instanciated with the SVGRenderer#clipRect function and applied with
 * the SVGElement#clip function.
 */
export type ClipRectElement = SVGElement;
/**
 * Color axis types
 */
export type ColorAxisTypeValue = ("linear"|"logarithmic");
/**
 * A valid color to be parsed and handled by Highcharts. Highcharts internally
 * supports hex colors like `#ffffff`, rgb colors like `rgb(255,255,255)` and
 * rgba colors like `rgba(255,255,255,1)`. Other colors may be supported by the
 * browsers and displayed correctly, but Highcharts is not able to process them
 * and apply concepts like opacity and brightening.
 */
export type ColorString = string;
/**
 * A valid color type than can be parsed and handled by Highcharts. It can be a
 * color string, a gradient object, or a pattern object.
 */
export type ColorType = (ColorString|GradientColorObject|PatternObject);
/**
 * All possible cursor styles.
 */
export type CursorValue = ('alias'|'all-scroll'|'auto'|'cell'|'col-resize'|'context-menu'|'copy'|'crosshair'|'default'|'e-resize'|'ew-resize'|'grab'|'grabbing'|'help'|'move'|'n-resize'|'ne-resize'|
'nesw-resize'|'no-drop'|'none'|'not-allowed'|'ns-resize'|'nw-resize'|'nwse-resize'|'pointer'|'progress'|'row-resize'|'s-resize'|'se-resize'|'sw-resize'|'text'|'vertical-text'|'w-resize'|'wait'|
'zoom-in'|'zoom-out');
/**
 * All possible dash styles.
 */
export type DashStyleValue = ('Dash'|'DashDot'|'Dot'|'LongDash'|'LongDashDot'|'LongDashDotDot'|'ShortDash'|'ShortDashDot'|'ShortDashDotDot'|'ShortDot'|'Solid');
/**
 * Callback function to modify the CSV before parsing it by the data module.
 *
 * @param csv
 *        The CSV to modify.
 *
 * @return The CSV to parse.
 */
export type DataBeforeParseCallbackFunction = (csv: string) => string;
/**
 * Callback function that gets called after parsing data.
 *
 * @param chartOptions
 *        The chart options that were used.
 */
export type DataCompleteCallbackFunction = (chartOptions: Options) => void;
export type DataGroupingApproximationValue = ("average"|"averages"|"close"|"high"|"low"|"open"|"sum");
/**
 * Callback JavaScript function to format the data label as a string. Note that
 * if a `format` is defined, the format takes precedence and the formatter is
 * ignored.
 *
 * @param this
 *        Data label context to format
 *
 * @param options
 *        API options of the data label
 *
 * @return Formatted data label text
 */
export type DataLabelsFormatterCallbackFunction = (this: PointLabelObject, options: DataLabelsOptions) => (number|string|null|undefined);
/**
 * Values for handling data labels that flow outside the plot area.
 */
export type DataLabelsOverflowValue = ("allow"|"justify");
/**
 * Callback function to parse string representations of dates into JavaScript
 * timestamps (milliseconds since 1.1.1970).
 *
 * @return Timestamp (milliseconds since 1.1.1970) as integer for Date class.
 */
export type DataParseDateCallbackFunction = (dateValue: string) => number;
/**
 * Callback function to access the parsed columns, the two-dimentional input
 * data array directly, before they are interpreted into series data and
 * categories.
 *
 * @param columns
 *        The parsed columns by the data module.
 *
 * @return Return `false` to stop completion, or call `this.complete()` to
 *         continue async.
 */
export type DataParsedCallbackFunction = (columns: Array<Array<any>>) => (boolean|undefined);
/**
 * Possible types for a data item in a column or row.
 */
export type DataValueType = (number|string|null);
/**
 * Gets fired when a drilldown point is clicked, before the new series is added.
 * Note that when clicking a category label to trigger multiple series
 * drilldown, one `drilldown` event is triggered per point in the category.
 *
 * @param this
 *        The chart where the event occurs.
 *
 * @param e
 *        The drilldown event.
 */
export type DrilldownCallbackFunction = (this: Chart, e: DrilldownEventObject) => void;
/**
 * This gets fired after all the series have been drilled up. This is especially
 * usefull in a chart with multiple drilldown series.
 *
 * @param this
 *        The chart where the event occurs.
 *
 * @param e
 *        The final drillup event.
 */
export type DrillupAllCallbackFunction = (this: Chart, e: DrillupAllEventObject) => void;
/**
 * Gets fired when drilling up from a drilldown series.
 *
 * @param this
 *        The chart where the event occurs.
 *
 * @param e
 *        The drillup event.
 */
export type DrillupCallbackFunction = (this: Chart, e: DrillupEventObject) => void;
/**
 * The function callback to execute when the event is fired. The `this` context
 * contains the instance, that fired the event.
 *
 * @param eventArguments
 *        Event arguments.
 */
export type EventCallbackFunction<T> = (this: T, eventArguments?: (Event|Dictionary<any>)) => (boolean|void);
/**
 * Function callback to execute while data rows are processed for exporting.
 * This allows the modification of data rows before processed into the final
 * format.
 *
 * @param this
 *        Chart context where the event occured.
 *
 * @param event
 *        Event object with data rows that can be modified.
 */
export type ExportDataCallbackFunction = (this: Chart, event: ExportDataEventObject) => void;
/**
 * Gets fired after a chart is printed through the context menu item or the
 * Chart.print method.
 *
 * @param chart
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ExportingAfterPrintCallbackFunction = (chart: Chart, event: Event) => void;
/**
 * Gets fired before a chart is printed through the context menu item or the
 * Chart.print method.
 *
 * @param chart
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ExportingBeforePrintCallbackFunction = (chart: Chart, event: Event) => void;
/**
 * Function to call if the offline-exporting module fails to export a chart on
 * the client side.
 *
 * @param options
 *        The exporting options.
 *
 * @param err
 *        The error from the module.
 */
export type ExportingErrorCallbackFunction = (options: ExportingOptions, err: Error) => void;
/**
 * Possible MIME types for exporting.
 */
export type ExportingMimeTypeValue = ("application/pdf"|"image/jpeg"|"image/png"|"image/svg+xml");
export type FlagsShapeValue = ("circlepin"|"flag"|"squarepin");
/**
 * Formats data as a string. Usually the data is accessible throught the `this`
 * keyword.
 *
 * @param this
 *        Context to format
 *
 * @return Formatted text
 */
export type FormatterCallbackFunction<T> = (this: T) => string;
/**
 * An object of key-value pairs for HTML attributes.
 */
export type HTMLAttributes = Dictionary<(boolean|number|string|Function)>;
/**
 * An HTML DOM element. The type is a reference to the regular HTMLElement in
 * the global scope.
 */
export type HTMLDOMElement = HTMLElement;
/**
 * Function callback when a cluster is clicked.
 *
 * @param this
 *        The point where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type MarkerClusterDrillCallbackFunction = (this: Point, event: PointClickEventObject) => void;
/**
 * Format a number and return a string based on input settings.
 *
 * @param number
 *        The input number to format.
 *
 * @param decimals
 *        The amount of decimals. A value of -1 preserves the amount in the
 *        input number.
 *
 * @param decimalPoint
 *        The decimal point, defaults to the one given in the lang options, or a
 *        dot.
 *
 * @param thousandsSep
 *        The thousands separator, defaults to the one given in the lang
 *        options, or a space character.
 *
 * @return The formatted number.
 */
export type NumberFormatterCallbackFunction = (number: number, decimals: number, decimalPoint?: string, thousandsSep?: string) => string;
/**
 * The iterator callback.
 *
 * @param this
 *        The context.
 *
 * @param value
 *        The property value.
 *
 * @param key
 *        The property key.
 *
 * @param obj
 *        The object that objectEach is being applied to.
 */
export type ObjectEachCallbackFunction<T> = (this: T, value: any, key: string, obj: any) => void;
export type OptionsApproximationValue = ("barnes-hut"|"none");
export type OptionsBoostBlendingValue = ("add"|"darken"|"multiply");
export type OptionsDataClassColorValue = ("category"|"tween");
export type OptionsDateFormatValue = ("dd/mm/YY"|"dd/mm/YYYY"|"mm/dd/YY"|"mm/dd/YYYY"|"YYYY/mm/dd");
export type OptionsFindNearestPointByValue = ("x"|"xy");
export type OptionsGapUnitValue = ("relative"|"value");
export type OptionsGridLineInterpolationValue = ("circle"|"polygon");
export type OptionsHeaderShapeValue = ("callout"|"square");
export type OptionsIntegrationValue = ("euler"|"verlet");
export type OptionsLandmarkVerbosityValue = ("all"|"disabled"|"one");
export type OptionsLayoutAlgorithmValue = ("squarified"|"strip"|"stripes"|"sliceAndDice");
export type OptionsLayoutStartingDirectionValue = ("horizontal"|"vertical");
export type OptionsLayoutValue = ("horizontal"|"proximate"|"vertical");
export type OptionsLinecapValue = ("round"|"square");
export type OptionsMarkerEndValue = ("arrow"|"none");
export type OptionsMinorTickPositionValue = ("inside"|"outside");
export type OptionsModeValue = ("normal"|"serialize");
export type OptionsOnKeyValue = ("close"|"high"|"low"|"open"|"y");
export type OptionsOperatorValue = ("=="|"==="|">"|">="|"<"|"<=");
export type OptionsOverflowValue = ("allow"|"justify");
export type OptionsPanKeyValue = ("alt"|"ctrl"|"meta"|"shift");
export type OptionsPinchTypeValue = ("x"|"xy"|"y");
export type OptionsPointIntervalUnitValue = ("day"|"month"|"year");
export type OptionsPointValKeyValue = ("close"|"high"|"low"|"open");
export type OptionsPosition3dValue = ("chart"|"flap"|"offset"|"ortho");
export type OptionsRotationModeValue = ("auto"|"circular"|"parallel"|"perpendicular");
export type OptionsRotationOriginValue = ("center"|"end"|"start");
export type OptionsStackingValue = ("normal"|"overlap"|"percent"|"stream");
export type OptionsStepValue = ("center"|"left"|"right");
export type OptionsTextAlignValue = ("center"|"left"|"right");
export type OptionsTickmarkPlacementValue = ("between"|"on");
export type OptionsTickPositionValue = ("inside"|"outside");
export type OptionsTypeValue = ("x"|"xy"|"y");
export type OptionsUnitValue = ("percentage"|"pixels"|"weight");
export type OptionsZoomKeyValue = ("alt"|"ctrl"|"meta"|"shift");
export type OptionsZoomTypeValue = ("x"|"xy"|"y");
export type PaneBackgroundShapeValue = ("arc"|"circle"|"solid");
/**
 * The default pathfinder algorithm to use for a chart. It is possible to define
 * your own algorithms by adding them to the
 * `Highcharts.Pathfinder.prototype.algorithms` object before the chart has been
 * created.
 *
 * The default algorithms are as follows:
 *
 * `straight`: Draws a straight line between the connecting points. Does not
 * avoid other points when drawing.
 *
 * `simpleConnect`: Finds a path between the points using right angles only.
 * Takes only starting/ending points into account, and will not avoid other
 * points.
 *
 * `fastAvoid`: Finds a path between the points using right angles only. Will
 * attempt to avoid other points, but its focus is performance over accuracy.
 * Works well with less dense datasets.
 */
export type PathfinderTypeValue = ("straight"|"fastAvoid"|"simpleConnect"|string);
/**
 * Function callback when a series point is clicked. Return false to cancel the
 * action.
 *
 * @param this
 *        The point where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type PointClickCallbackFunction = (this: Point, event: PointClickEventObject) => void;
/**
 * Function callback to execute while series points are dragged. Return false to
 * stop the default drag action.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type PointDragCallbackFunction = (this: Point, event: PointDragEventObject) => void;
/**
 * Function callback to execute when a series point is dragged.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type PointDragStartCallbackFunction = (this: Point, event: PointDragStartEventObject) => void;
/**
 * Function callback to execute when series points are dropped.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type PointDropCallbackFunction = (this: Point, event: PointDropEventObject) => void;
/**
 * Gets fired when the legend item belonging to a point is clicked. The default
 * action is to toggle the visibility of the point. This can be prevented by
 * returning `false` or calling `event.preventDefault()`.
 *
 * @param this
 *        The point on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type PointLegendItemClickCallbackFunction = (this: Point, event: PointLegendItemClickEventObject) => void;
/**
 * Gets fired when the mouse leaves the area close to the point.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type PointMouseOutCallbackFunction = (this: Point, event: PointerEvent) => void;
/**
 * Gets fired when the mouse enters the area close to the point.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type PointMouseOverCallbackFunction = (this: Point, event: Event) => void;
/**
 * Possible option types for a data point. Use `null` to indicate a gap.
 */
export type PointOptionsType = (number|string|PointOptionsObject|Array<(number|string|null)>|null);
/**
 * Gets fired when the point is removed using the `.remove()` method.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type PointRemoveCallbackFunction = (this: Point, event: Event) => void;
/**
 * Gets fired when the point is selected either programmatically or following a
 * click on the point.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type PointSelectCallbackFunction = (this: Point, event: PointInteractionEventObject) => void;
/**
 * Possible key values for the point state options.
 */
export type PointStateValue = ("hover"|"inactive"|"normal"|"select");
/**
 * Fires when the point is unselected either programmatically or following a
 * click on the point.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type PointUnselectCallbackFunction = (this: Point, event: PointInteractionEventObject) => void;
/**
 * Gets fired when the point is updated programmatically through the `.update()`
 * method.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type PointUpdateCallbackFunction = (this: Point, event: PointUpdateEventObject) => void;
/**
 * Define the time span for the button
 */
export type RangeSelectorButtonTypeValue = ("all"|"day"|"hour"|"millisecond"|"minute"|"month"|"second"|"week"|"year"|"ytd");
/**
 * Callback function to react on button clicks.
 *
 * @param e
 *        Event arguments.
 *
 * @param Return
 *        false to cancel the default button event.
 */
export type RangeSelectorClickCallbackFunction = (e: Event, Return: (boolean|undefined)) => void;
/**
 * Callback function to parse values entered in the input boxes and return a
 * valid JavaScript time as milliseconds since 1970.
 *
 * @param value
 *        Input value to parse.
 *
 * @return Parsed JavaScript time value.
 */
export type RangeSelectorParseCallbackFunction = (value: string) => number;
/**
 * If a number is given, it defines the pixel length. If a percentage string is
 * given, like for example `'50%'`, the setting defines a length relative to a
 * base size, for example the size of a container.
 */
export type RelativeSize = (number|string);
/**
 * A callback function to gain complete control on when the responsive rule
 * applies.
 *
 * @param this
 *        Chart context.
 *
 * @return Return `true` if it applies.
 */
export type ResponsiveCallbackFunction = (this: Chart) => boolean;
/**
 * @param evt
 *        Mouse click event
 */
export type ScreenReaderClickCallbackFunction = (evt: MouseEvent) => void;
/**
 * Creates a formatted string for the screen reader module.
 *
 * @param context
 *        Context to format
 *
 * @return Formatted string for the screen reader module.
 */
export type ScreenReaderFormatterCallbackFunction<T> = (context: T) => string;
/**
 * Function callback when a series has been animated.
 *
 * @param this
 *        The series where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type SeriesAfterAnimateCallbackFunction = (this: Series, event: SeriesAfterAnimateEventObject) => void;
/**
 * Function callback when the checkbox next to the series' name in the legend is
 * clicked.
 *
 * @param this
 *        The series where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type SeriesCheckboxClickCallbackFunction = (this: Series, event: SeriesCheckboxClickEventObject) => void;
/**
 * Function callback when a series is clicked. Return false to cancel toogle
 * actions.
 *
 * @param this
 *        The series where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type SeriesClickCallbackFunction = (this: Series, event: SeriesClickEventObject) => void;
/**
 * Gets fired when the series is hidden after chart generation time, either by
 * clicking the legend item or by calling `.hide()`.
 *
 * @param this
 *        The series where the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type SeriesHideCallbackFunction = (this: Series, event: Event) => void;
/**
 * Gets fired when the legend item belonging to a series is clicked. The default
 * action is to toggle the visibility of the series. This can be prevented by
 * returning `false` or calling `event.preventDefault()`.
 *
 * @param this
 *        The series where the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type SeriesLegendItemClickCallbackFunction = (this: Series, event: SeriesLegendItemClickEventObject) => void;
/**
 * The SVG value used for the `stroke-linecap` and `stroke-linejoin` of a line
 * graph.
 */
export type SeriesLinecapValue = ("butt"|"round"|"square"|string);
/**
 * Gets fired when the mouse leaves the graph.
 *
 * @param this
 *        Series where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesMouseOutCallbackFunction = (this: Series, event: PointerEvent) => void;
/**
 * Gets fired when the mouse enters the graph.
 *
 * @param this
 *        Series where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesMouseOverCallbackFunction = (this: Series, event: PointerEvent) => void;
/**
 * Formatter callback function.
 *
 * @param this
 *        Data label context to format
 *
 * @return Formatted data label text
 */
export type SeriesNetworkgraphDataLabelsFormatterCallbackFunction = (this: (PointLabelObject|SeriesNetworkgraphDataLabelsFormatterContextObject)) => string;
/**
 * The possible types of series options.
 */
export type SeriesOptionsType = SeriesOptionsRegistry[keyof SeriesOptionsRegistry];
/**
 * Layout value for the child nodes in an organization chart. If `hanging`, this
 * node's children will hang below their parent, allowing a tighter packing of
 * nodes in the diagram.
 */
export type SeriesOrganizationNodesLayoutValue = ("hanging"|"normal");
/**
 * Formatter callback function.
 *
 * @param this
 *        Data label context to format
 *
 * @return Formatted data label text
 */
export type SeriesPackedBubbleDataLabelsFormatterCallbackFunction = (this: SeriesPackedBubbleDataLabelsFormatterContextObject) => string;
/**
 * Formatter callback function.
 *
 * @param this
 *        Data label context to format
 *
 * @return Formatted data label text
 */
export type SeriesSankeyDataLabelsFormatterCallbackFunction = (this: (PointLabelObject|SeriesSankeyDataLabelsFormatterContextObject)) => (string|undefined);
/**
 * Gets fired when the series is shown after chart generation time, either by
 * clicking the legend item or by calling `.show()`.
 *
 * @param this
 *        Series where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesShowCallbackFunction = (this: Series, event: Event) => void;
/**
 * Possible key values for the series state options.
 */
export type SeriesStateValue = ("hover"|"inactive"|"normal"|"select");
/**
 * An SVG DOM element. The type is a reference to the regular SVGElement in the
 * global scope.
 */
export type SVGDOMElement = globals.GlobalSVGElement;
/**
 * Array of path commands, that will go into the `d` attribute of an SVG
 * element.
 */
export type SVGPathArray = Array<(Array<SVGPathCommand>|[SVGPathCommand, number]|[SVGPathCommand, number, number]|[SVGPathCommand, number, number, number, number]|[SVGPathCommand, number, number,
number, number, number, number]|[SVGPathCommand, number, number, number, number, number, number, number])>;
/**
 * Possible path commands in an SVG path array. Valid values are `A`, `C`, `H`,
 * `L`, `M`, `Q`, `S`, `T`, `V`, `Z`.
 */
export type SVGPathCommand = ("a"|"c"|"h"|"l"|"m"|"q"|"s"|"t"|"v"|"z"|"A"|"C"|"H"|"L"|"M"|"Q"|"S"|"T"|"V"|"Z");
/**
 * Can be one of `arc`, `callout`, `circle`, `diamond`, `square`, `triangle`,
 * and `triangle-down`. Symbols are used internally for point markers, button
 * and label borders and backgrounds, or custom shapes. Extendable by adding to
 * SVGRenderer#symbols.
 */
export type SymbolKeyValue = ("arc"|"callout"|"circle"|"diamond"|"square"|"triangle"|"triangle-down");
export type TilemapShapeValue = ("circle"|"diamond"|"hexagon"|"square");
/**
 * Function of an additional date format specifier.
 *
 * @param timestamp
 *        The time to format.
 *
 * @return The formatted portion of the date.
 */
export type TimeFormatCallbackFunction = (timestamp: number) => string;
/**
 * Callback JavaScript function to format the data label as a string. Note that
 * if a `format` is defined, the format takes precedence and the formatter is
 * ignored.
 *
 * @param this
 *        Data label context to format
 *
 * @return Formatted data label text
 */
export type TimelineDataLabelsFormatterCallbackFunction = (this: (PointLabelObject|TimelineDataLabelsFormatterContextObject)) => (number|string|null|undefined);
/**
 * Time units used in `Time.get` and `Time.set`
 */
export type TimeUnitValue = ("Date"|"Day"|"FullYear"|"Hours"|"Milliseconds"|"Minutes"|"Month"|"Seconds");
/**
 * A callback to return the time zone offset for a given datetime. It takes the
 * timestamp in terms of milliseconds since January 1 1970, and returns the
 * timezone offset in minutes. This provides a hook for drawing time based
 * charts in specific time zones using their local DST crossover dates, with the
 * help of external libraries.
 *
 * @param timestamp
 *        Timestamp in terms of milliseconds since January 1 1970.
 *
 * @return Timezone offset in minutes.
 */
export type TimezoneOffsetCallbackFunction = (timestamp: number) => number;
/**
 * Callback function to format the text of the tooltip from scratch.
 *
 * In case of single or shared tooltips, a string should be be returned. In case
 * of splitted tooltips, it should return an array where the first item is the
 * header, and subsequent items are mapped to the points. Return `false` to
 * disable tooltip for a specific point on series.
 *
 * @param this
 *        Context to format
 *
 * @param tooltip
 *        The tooltip instance
 *
 * @return Formatted text or false
 */
export type TooltipFormatterCallbackFunction = (this: TooltipFormatterContextObject, tooltip: Tooltip) => (false|string|Array<(string|null|undefined)>|null|undefined);
/**
 * A callback function to place the tooltip in a specific position.
 *
 * @param this
 *        Tooltip context of the callback.
 *
 * @param labelWidth
 *        Width of the tooltip.
 *
 * @param labelHeight
 *        Height of the tooltip.
 *
 * @param point
 *        Point information for positioning a tooltip.
 *
 * @return New position for the tooltip.
 */
export type TooltipPositionerCallbackFunction = (this: Tooltip, labelWidth: number, labelHeight: number, point: TooltipPositionerPointObject) => PositionObject;
export type TooltipShapeValue = ("callout"|"circle"|"square");
export type VariablePieSizeByValue = ("area"|"radius");
/**
 * The vertical alignment of an element.
 */
export type VerticalAlignValue = ("bottom"|"middle"|"top");
/**
 * Proceed function to call original (wrapped) function.
 *
 * @param arg1
 *        Optional argument. Without any arguments defaults to first argument of
 *        the wrapping function.
 *
 * @param arg2
 *        Optional argument. Without any arguments defaults to second argument
 *        of the wrapping function.
 *
 * @param arg3
 *        Optional argument. Without any arguments defaults to third argument of
 *        the wrapping function.
 *
 * @return Return value of the original function.
 */
export type WrapProceedFunction = (arg1?: any, arg2?: any, arg3?: any) => any;
/**
 * Formatter function for the text of a crosshair label.
 *
 * @param this
 *        Axis context
 *
 * @param value
 *        Y value of the data point
 */
export type XAxisCrosshairLabelFormatterCallbackFunction = (this: Axis, value: number) => string;
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for announcing new data to
 * screen reader users. Useful for dynamic data applications and drilldown.
 *
 * Keep in mind that frequent announcements will not be useful to users, as they
 * won't have time to explore the new data. For these applications, consider
 * making snapshots of the data accessible, and do the announcements in batches.
 */
export interface AccessibilityAnnounceNewDataOptionsObject {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Optional formatter callback for
     * the announcement. Receives up to three arguments. The first argument is
     * always an array of all series that received updates. If an announcement
     * is already queued, the series that received updates for that announcement
     * are also included in this array. The second argument is provided if
     * `chart.addSeries` was called, and there is a new series. In that case,
     * this argument is a reference to the new series. The third argument,
     * similarly, is provided if `series.addPoint` was called, and there is a
     * new point. In that case, this argument is a reference to the new point.
     *
     * The function should return a string with the text to announce to the
     * user. Return empty string to not announce anything. Return `false` to use
     * the default announcement format.
     */
    announcementFormatter?: AccessibilityAnnouncementFormatter;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable announcing new data to
     * screen reader users
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Choose whether or not the
     * announcements should interrupt the screen reader. If not enabled, the
     * user will be notified once idle. It is recommended not to enable this
     * setting unless there is a specific reason to do so.
     */
    interruptUser?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Minimum interval between
     * announcements in milliseconds. If new data arrives before this amount of
     * time has passed, it is queued for announcement. If another new data event
     * happens while an announcement is queued, the queued announcement is
     * dropped, and the latest announcement is queued instead. Set to 0 to allow
     * all announcements, but be warned that frequent announcements are
     * disturbing to users.
     */
    minAnnounceInterval?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for configuring
 * accessibility for the chart. Requires the accessibility module to be loaded.
 * For a description of the module and information on its features, see
 * Highcharts Accessibility.
 */
export interface AccessibilityOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for announcing new data
     * to screen reader users. Useful for dynamic data applications and
     * drilldown.
     *
     * Keep in mind that frequent announcements will not be useful to users, as
     * they won't have time to explore the new data. For these applications,
     * consider making snapshots of the data accessible, and do the
     * announcements in batches.
     */
    announceNewData?: AccessibilityAnnounceNewDataOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A hook for adding custom
     * components to the accessibility module. Should be an object mapping
     * component names to instances of classes inheriting from the
     * Highcharts.AccessibilityComponent base class. Remember to add the
     * component to the keyboardNavigation.order for the keyboard navigation to
     * be usable.
     */
    customComponents?: any;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A text description of the chart.
     *
     * **Note: Prefer using linkedDescription or caption instead.**
     *
     * If the Accessibility module is loaded, this option is included by default
     * as a long description of the chart in the hidden screen reader
     * information region.
     *
     * Note: Since Highcharts now supports captions and linked descriptions, it
     * is preferred to define the description using those methods, as a visible
     * caption/description benefits all users. If the
     * `accessibility.description` option is defined, the linked description is
     * ignored, and the caption is hidden from screen reader users.
     */
    description?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable accessibility
     * functionality for the chart.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Theme to apply to the chart when
     * Windows High Contrast Mode is detected. By default, a high contrast theme
     * matching the high contrast system system colors is used.
     */
    highContrastTheme?: any;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for keyboard navigation.
     */
    keyboardNavigation?: KeyboardNavigationOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Amount of landmarks/regions to
     * create for screen reader users. More landmarks can make navigation with
     * screen readers easier, but can be distracting if there are lots of charts
     * on the page. Three modes are available:
     *
     * - `all`: Adds regions for all series, legend, menu, information region.
     *
     * - `one`: Adds a single landmark per chart.
     *
     * - `disabled`: No landmarks are added.
     */
    landmarkVerbosity?: OptionsLandmarkVerbosityValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Link the chart to an HTML
     * element describing the contents of the chart.
     *
     * It is always recommended to describe charts using visible text, to
     * improve SEO as well as accessibility for users with disabilities. This
     * option lets an HTML element with a description be linked to the chart, so
     * that screen reader users can connect the two.
     *
     * By setting this option to a string, Highcharts runs the string as an HTML
     * selector query on the entire document. If there is only a single match,
     * this element is linked to the chart. The content of the linked element
     * will be included in the chart description for screen reader users.
     *
     * By default, the chart looks for an adjacent sibling element with the
     * `highcharts-description` class.
     *
     * The feature can be disabled by setting the option to an empty string, or
     * overridden by providing the accessibility.description option.
     * Alternatively, the HTML element to link can be passed in directly as an
     * HTML node.
     *
     * If you need the description to be part of the exported image, consider
     * using the caption feature.
     *
     * If you need the description to be hidden visually, use the
     * accessibility.description option.
     */
    linkedDescription?: (string|HTMLDOMElement);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for descriptions of
     * individual data points.
     */
    point?: AccessibilityPointOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for the
     * screen reader information sections added before and after the chart.
     */
    screenReaderSection?: AccessibilityScreenReaderSectionOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options global to
     * all data series. Individual series can also have specific accessibility
     * options set.
     */
    series?: AccessibilitySeriesOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A text description of the chart
     * type.
     *
     * If the Accessibility module is loaded, this will be included in the
     * description of the chart in the screen reader information region.
     *
     * Highcharts will by default attempt to guess the chart type, but for more
     * complex charts it is recommended to specify this property for clarity.
     */
    typeDescription?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for descriptions of
 * individual data points.
 */
export interface AccessibilityPointOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Date format to use for points on
     * datetime axes when describing them to screen reader users.
     *
     * Defaults to the same format as in tooltip.
     *
     * For an overview of the replacement codes, see dateFormat.
     */
    dateFormat?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Formatter function to determine
     * the date/time format used with points on datetime axes when describing
     * them to screen reader users. Receives one argument, `point`, referring to
     * the point to describe. Should return a date format string compatible with
     * dateFormat.
     */
    dateFormatter?: ScreenReaderFormatterCallbackFunction<Point>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Formatter function to use
     * instead of the default for point descriptions.
     *
     * Receives one argument, `point`, referring to the point to describe.
     * Should return a string with the description of the point for a screen
     * reader user. If `false` is returned, the default formatter will be used
     * for that point.
     *
     * Note: Prefer using accessibility.point.valueDescriptionFormat instead if
     * possible, as default functionality such as describing annotations will be
     * preserved.
     */
    descriptionFormatter?: ScreenReaderFormatterCallbackFunction<Point>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Decimals to use for the values
     * in the point descriptions. Uses tooltip.valueDecimals if not defined.
     */
    valueDecimals?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format to use for describing the
     * values of data points to assistive technology - including screen readers.
     * The point context is available as `{point}`.
     *
     * Additionally, the series name, annotation info, and description added in
     * `point.accessibility.description` is added by default if relevant. To
     * override this, use the accessibility.point.descriptionFormatter option.
     */
    valueDescriptionFormat?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Prefix to add to the values in
     * the point descriptions. Uses tooltip.valuePrefix if not defined.
     */
    valuePrefix?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Suffix to add to the values in
     * the point descriptions. Uses tooltip.valueSuffix if not defined.
     */
    valueSuffix?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for the screen
 * reader information sections added before and after the chart.
 */
export interface AccessibilityScreenReaderSectionOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for the screen reader
     * information region after the chart. Analogous to beforeChartFormat.
     */
    afterChartFormat?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A formatter function to create
     * the HTML contents of the hidden screen reader information region after
     * the chart. Analogous to beforeChartFormatter.
     */
    afterChartFormatter?: ScreenReaderFormatterCallbackFunction<Chart>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Date format to use to describe
     * range of datetime axes.
     *
     * For an overview of the replacement codes, see dateFormat.
     */
    axisRangeDateFormat?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for the screen reader
     * information region before the chart. Supported HTML tags are `<h1-7>`,
     * `<p>`, `<div>`, `<a>`, `<ul>`, `<ol>`, `<li>`, and `<button>`. Attributes
     * are not supported, except for id on `<div>`, `<a>`, and `<button>`. Id is
     * required on `<a>` and `<button>` in the format `<tag id="abcd">`.
     * Numbers, lower- and uppercase letters, "-" and "#" are valid characters
     * in IDs.
     */
    beforeChartFormat?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A formatter function to create
     * the HTML contents of the hidden screen reader information region before
     * the chart. Receives one argument, `chart`, referring to the chart object.
     * Should return a string with the HTML content of the region. By default
     * this returns an automatic description of the chart based on
     * beforeChartFormat.
     */
    beforeChartFormatter?: ScreenReaderFormatterCallbackFunction<Chart>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Function to run upon clicking
     * the "Play as sound" button in the screen reader region.
     *
     * By default Highcharts will call the `chart.sonify` function.
     */
    onPlayAsSoundClick?: ScreenReaderClickCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Function to run upon clicking
     * the "View as Data Table" link in the screen reader region.
     *
     * By default Highcharts will insert and set focus to a data table
     * representation of the chart.
     */
    onViewDataTableClick?: ScreenReaderClickCallbackFunction;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options global to all
 * data series. Individual series can also have specific accessibility options
 * set.
 */
export interface AccessibilitySeriesOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether or not to add series
     * descriptions to charts with a single series.
     */
    describeSingleSeries?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Formatter function to use
     * instead of the default for series descriptions. Receives one argument,
     * `series`, referring to the series to describe. Should return a string
     * with the description of the series for a screen reader user. If `false`
     * is returned, the default formatter will be used for that series.
     */
    descriptionFormatter?: ScreenReaderFormatterCallbackFunction<Series>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When a series contains more
     * points than this, we no longer expose information about individual points
     * to screen readers.
     *
     * Set to `false` to disable.
     */
    pointDescriptionEnabledThreshold?: (boolean|number);
}
/**
 * Options to align the element relative to the chart or another box.
 */
export interface AlignObject {
    /**
     * Horizontal alignment. Can be one of `left`, `center` and `right`.
     */
    align?: AlignValue;
    /**
     * Use the `transform` attribute with translateX and translateY custom
     * attributes to align this elements rather than `x` and `y` attributes.
     */
    alignByTranslate?: boolean;
    /**
     * Vertical alignment. Can be one of `top`, `middle` and `bottom`.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * Horizontal pixel offset from alignment.
     */
    x?: number;
    /**
     * Vertical pixel offset from alignment.
     */
    y?: number;
}
/**
 * An animation configuration. Animation configurations can also be defined as
 * booleans, where `false` turns off animation and `true` defaults to a duration
 * of 500ms and defer of 0ms.
 */
export interface AnimationOptionsObject {
    /**
     * A callback function to exectute when the animation finishes.
     */
    complete?: Function;
    /**
     * The animation defer in milliseconds.
     */
    defer?: number;
    /**
     * The animation duration in milliseconds.
     */
    duration?: number;
    /**
     * The name of an easing function as defined on the `Math` object.
     */
    easing?: (string|Function);
    /**
     * A callback function to execute on each step of each attribute or CSS
     * property that's being animated. The first argument contains information
     * about the animation and progress.
     */
    step?: Function;
}
export interface AnnotationControllable {
    annotation: Annotation;
    chart: Chart;
    collection: string;
    points: Array<Point>;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's control
 * points. Each control point inherits options from controlPointOptions object.
 * Options from the controlPointOptions can be overwritten by options in a
 * specific control point.
 */
export interface AnnotationControlPointOptionsObject {
    positioner?: AnnotationControlPointPositionerFunction;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for an
 * annotation label.
 */
export interface AnnotationLabelAccessibilityOptionsObject {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Description of an annotation
     * label for screen readers and other assistive technology.
     */
    description?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) An array of points for the shape.
 * This option is available for shapes which can use multiple points such as
 * path. A point can be either a point object or a point's id.
 */
export interface AnnotationMockPointOptionsObject {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position of the point.
     * Units can be either in axis or chart pixel coordinates.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This number defines which xAxis
     * the point is connected to. It refers to either the axis id or the index
     * of the axis in the xAxis array. If the option is not configured or the
     * axis is not found the point's x coordinate refers to the chart pixels.
     */
    xAxis?: (number|string|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position of the point.
     * Units can be either in axis or chart pixel coordinates.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This number defines which yAxis
     * the point is connected to. It refers to either the axis id or the index
     * of the axis in the yAxis array. If the option is not configured or the
     * axis is not found the point's y coordinate refers to the chart pixels.
     */
    yAxis?: (number|string|null);
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `annotation`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface AnnotationsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders annotation immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
export interface AnnotationsCrookedLineControlPointOptions {
    events?: any;
}
/**
 * (Highstock) Options for annotation's labels. Each label inherits options from
 * the labelOptions object. An option from the labelOptions can be overwritten
 * by config for a specific label.
 */
export interface AnnotationsCrookedLineLabelOptions {
    /**
     * (Highstock) Accessibility options for an annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highstock) The alignment of the annotation's label. If right, the right
     * side of the label should be touching the point.
     */
    align?: AlignValue;
    /**
     * (Highstock) Whether to allow the annotation's labels to overlap. To make
     * the labels less sensitive for overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highstock) Whether to hide the annotation's label that is outside the
     * plot area.
     */
    crop?: boolean;
    /**
     * (Highstock) The label's pixel distance from the point.
     */
    distance?: number;
    /**
     * (Highstock) A format string for the data label.
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the annotation's
     * label. Note that if a `format` or `text` are defined, the format or text
     * take precedence and the formatter is ignored. `This` refers to a point
     * object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highstock) Whether the annotation is visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highstock) How to handle the annotation's label that flow outside the
     * plot area. The justify option aligns the label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highstock) When either the borderWidth or the backgroundColor is set,
     * this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highstock) The shadow of the box. The shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     */
    shape?: string;
    /**
     * (Highstock) Styles for the annotation's label.
     */
    style?: CSSObject;
    /**
     * (Highstock) Alias for the format option.
     */
    text?: string;
    /**
     * (Highstock) Whether to use HTML to render the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of the annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highstock) The x position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highstock) A crooked line annotation.
 */
export interface AnnotationsCrookedLineOptions {
    controlPointOptions?: AnnotationsCrookedLineControlPointOptions;
    /**
     * (Highstock) Options for annotation's labels. Each label inherits options
     * from the labelOptions object. An option from the labelOptions can be
     * overwritten by config for a specific label.
     */
    labelOptions?: AnnotationsCrookedLineLabelOptions;
    /**
     * (Highstock) Options for annotation's shapes. Each shape inherits options
     * from the shapeOptions object. An option from the shapeOptions can be
     * overwritten by config for a specific shape.
     */
    shapeOptions?: AnnotationsCrookedLineShapeOptions;
    /**
     * (Highstock) Additional options for an annotation with the type.
     */
    typeOptions?: AnnotationsCrookedLineTypeOptions;
}
/**
 * (Highstock) Options for annotation's shapes. Each shape inherits options from
 * the shapeOptions object. An option from the shapeOptions can be overwritten
 * by config for a specific shape.
 */
export interface AnnotationsCrookedLineShapeOptions {
    /**
     * (Highstock) Name of the dash style to use for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The height of the shape.
     */
    height?: number;
    /**
     * (Highstock) The radius of the shape.
     */
    r?: number;
    /**
     * (Highstock) Defines additional snapping area around an annotation making
     * this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highstock) The URL for an image to use as the annotation shape. Note,
     * type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highstock) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highstock) The pixel stroke width of the shape.
     */
    strokeWidth?: number;
    /**
     * (Highstock) The type of the shape, e.g. circle or rectangle.
     */
    type?: string;
    /**
     * (Highstock) The width of the shape.
     */
    width?: number;
}
/**
 * (Highstock) Line options.
 */
export interface AnnotationsCrookedLineTypeLineOptions {
    fill?: string;
}
/**
 * (Highstock) Additional options for an annotation with the type.
 */
export interface AnnotationsCrookedLineTypeOptions {
    /**
     * (Highstock) Line options.
     */
    line?: AnnotationsCrookedLineTypeLineOptions;
    points?: Array<AnnotationsCrookedLineTypePointsOptions>;
    /**
     * (Highstock) This number defines which xAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    xAxis?: number;
    /**
     * (Highstock) This number defines which yAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    yAxis?: number;
}
export interface AnnotationsCrookedLineTypePointsOptions {
    controlPoint?: number;
    /**
     * (Highstock) The x position of the point.
     */
    x?: number;
    /**
     * (Highstock) The y position of the point.
     */
    y?: number;
}
export interface AnnotationsElliottWaveControlPointOptions {
    events?: any;
}
/**
 * (Highstock) Options for annotation's labels. Each label inherits options from
 * the labelOptions object. An option from the labelOptions can be overwritten
 * by config for a specific label.
 */
export interface AnnotationsElliottWaveLabelOptions {
    /**
     * (Highstock) Accessibility options for an annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highstock) The alignment of the annotation's label. If right, the right
     * side of the label should be touching the point.
     */
    align?: string;
    /**
     * (Highstock) Whether to allow the annotation's labels to overlap. To make
     * the labels less sensitive for overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the annotation's label.
     */
    backgroundColor?: string;
    /**
     * (Highstock) The border color for the annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highstock) Whether to hide the annotation's label that is outside the
     * plot area.
     */
    crop?: boolean;
    /**
     * (Highstock) The label's pixel distance from the point.
     */
    distance?: number;
    /**
     * (Highstock) A format string for the data label.
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the annotation's
     * label. Note that if a `format` or `text` are defined, the format or text
     * take precedence and the formatter is ignored. `This` refers to a point
     * object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highstock) Whether the annotation is visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highstock) How to handle the annotation's label that flow outside the
     * plot area. The justify option aligns the label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highstock) When either the borderWidth or the backgroundColor is set,
     * this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highstock) The shadow of the box. The shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     */
    shape?: string;
    /**
     * (Highstock) Styles for the annotation's label.
     */
    style?: CSSObject;
    /**
     * (Highstock) Alias for the format option.
     */
    text?: string;
    type?: string;
    /**
     * (Highstock) Whether to use HTML to render the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of the annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highstock) The x position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highstock) An elliott wave annotation.
 */
export interface AnnotationsElliottWaveOptions {
    controlPointOptions?: AnnotationsElliottWaveControlPointOptions;
    /**
     * (Highstock) Options for annotation's labels. Each label inherits options
     * from the labelOptions object. An option from the labelOptions can be
     * overwritten by config for a specific label.
     */
    labelOptions?: AnnotationsElliottWaveLabelOptions;
    /**
     * (Highstock) Options for annotation's shapes. Each shape inherits options
     * from the shapeOptions object. An option from the shapeOptions can be
     * overwritten by config for a specific shape.
     */
    shapeOptions?: AnnotationsElliottWaveShapeOptions;
    /**
     * (Highstock) Additional options for an annotation with the type.
     */
    typeOptions?: AnnotationsElliottWaveTypeOptions;
}
/**
 * (Highstock) Options for annotation's shapes. Each shape inherits options from
 * the shapeOptions object. An option from the shapeOptions can be overwritten
 * by config for a specific shape.
 */
export interface AnnotationsElliottWaveShapeOptions {
    /**
     * (Highstock) Name of the dash style to use for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The height of the shape.
     */
    height?: number;
    /**
     * (Highstock) The radius of the shape.
     */
    r?: number;
    /**
     * (Highstock) Defines additional snapping area around an annotation making
     * this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highstock) The URL for an image to use as the annotation shape. Note,
     * type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highstock) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highstock) The pixel stroke width of the shape.
     */
    strokeWidth?: number;
    /**
     * (Highstock) The type of the shape, e.g. circle or rectangle.
     */
    type?: string;
    /**
     * (Highstock) The width of the shape.
     */
    width?: number;
}
/**
 * (Highstock) Line options.
 */
export interface AnnotationsElliottWaveTypeLineOptions {
    fill?: string;
    strokeWidth?: number;
}
/**
 * (Highstock) Additional options for an annotation with the type.
 */
export interface AnnotationsElliottWaveTypeOptions {
    /**
     * (Highstock) Line options.
     */
    line?: AnnotationsElliottWaveTypeLineOptions;
    points?: Array<AnnotationsElliottWaveTypePointsOptions>;
    /**
     * (Highstock) This number defines which xAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    xAxis?: number;
    /**
     * (Highstock) This number defines which yAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    yAxis?: number;
}
/**
 * (Highstock) Options for annotation's labels. Each label inherits options from
 * the labelOptions object. An option from the labelOptions can be overwritten
 * by config for a specific label.
 */
export interface AnnotationsElliottWaveTypePointsLabelOptions {
    /**
     * (Highstock) Accessibility options for an annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highstock) The alignment of the annotation's label. If right, the right
     * side of the label should be touching the point.
     */
    align?: AlignValue;
    /**
     * (Highstock) Whether to allow the annotation's labels to overlap. To make
     * the labels less sensitive for overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highstock) Whether to hide the annotation's label that is outside the
     * plot area.
     */
    crop?: boolean;
    /**
     * (Highstock) The label's pixel distance from the point.
     */
    distance?: number;
    /**
     * (Highstock) A format string for the data label.
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the annotation's
     * label. Note that if a `format` or `text` are defined, the format or text
     * take precedence and the formatter is ignored. `This` refers to a point
     * object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highstock) Whether the annotation is visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highstock) How to handle the annotation's label that flow outside the
     * plot area. The justify option aligns the label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highstock) When either the borderWidth or the backgroundColor is set,
     * this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highstock) The shadow of the box. The shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     */
    shape?: string;
    /**
     * (Highstock) Styles for the annotation's label.
     */
    style?: CSSObject;
    /**
     * (Highstock) Alias for the format option.
     */
    text?: string;
    /**
     * (Highstock) Whether to use HTML to render the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of the annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highstock) The x position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    y?: number;
}
export interface AnnotationsElliottWaveTypePointsOptions {
    controlPoint?: number;
    /**
     * (Highstock) Options for annotation's labels. Each label inherits options
     * from the labelOptions object. An option from the labelOptions can be
     * overwritten by config for a specific label.
     */
    label?: AnnotationsElliottWaveTypePointsLabelOptions;
    /**
     * (Highstock) The x position of the point.
     */
    x?: number;
    /**
     * (Highstock) The y position of the point.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Events available in annotations.
 */
export interface AnnotationsEventsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Event callback when annotation
     * is added to the chart.
     */
    add?: EventCallbackFunction<Annotation>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Event callback when annotation
     * is updated (e.g. drag and droppped or resized by control points).
     */
    afterUpdate?: EventCallbackFunction<Annotation>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Event callback when annotation
     * is removed from the chart.
     */
    remove?: EventCallbackFunction<Annotation>;
}
export interface AnnotationsFibonacciControlPointOptions {
    events?: any;
}
/**
 * (Highstock) Options for annotation's labels. Each label inherits options from
 * the labelOptions object. An option from the labelOptions can be overwritten
 * by config for a specific label.
 */
export interface AnnotationsFibonacciLabelOptions {
    /**
     * (Highstock) Accessibility options for an annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highstock) The alignment of the annotation's label. If right, the right
     * side of the label should be touching the point.
     */
    align?: string;
    /**
     * (Highstock) Whether to allow the annotation's labels to overlap. To make
     * the labels less sensitive for overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the annotation's label.
     */
    backgroundColor?: string;
    /**
     * (Highstock) The border color for the annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highstock) Whether to hide the annotation's label that is outside the
     * plot area.
     */
    crop?: boolean;
    /**
     * (Highstock) The label's pixel distance from the point.
     */
    distance?: number;
    /**
     * (Highstock) A format string for the data label.
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the annotation's
     * label. Note that if a `format` or `text` are defined, the format or text
     * take precedence and the formatter is ignored. `This` refers to a point
     * object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highstock) Whether the annotation is visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highstock) How to handle the annotation's label that flow outside the
     * plot area. The justify option aligns the label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highstock) When either the borderWidth or the backgroundColor is set,
     * this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highstock) The shadow of the box. The shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     */
    shape?: string;
    /**
     * (Highstock) Styles for the annotation's label.
     */
    style?: (AnnotationsFibonacciLabelStyleOptions|CSSObject);
    /**
     * (Highstock) Alias for the format option.
     */
    text?: string;
    /**
     * (Highstock) Whether to use HTML to render the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of the annotation's label.
     */
    verticalAlign?: string;
    /**
     * (Highstock) The x position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highstock) Styles for the annotation's label.
 */
export interface AnnotationsFibonacciLabelStyleOptions {
    color?: string;
}
/**
 * (Highstock) A fibonacci annotation.
 */
export interface AnnotationsFibonacciOptions {
    controlPointOptions?: AnnotationsFibonacciControlPointOptions;
    /**
     * (Highstock) Options for annotation's labels. Each label inherits options
     * from the labelOptions object. An option from the labelOptions can be
     * overwritten by config for a specific label.
     */
    labelOptions?: AnnotationsFibonacciLabelOptions;
    /**
     * (Highstock) Options for annotation's shapes. Each shape inherits options
     * from the shapeOptions object. An option from the shapeOptions can be
     * overwritten by config for a specific shape.
     */
    shapeOptions?: AnnotationsFibonacciShapeOptions;
    /**
     * (Highstock) Additional options for an annotation with the type.
     */
    typeOptions?: AnnotationsFibonacciTypeOptions;
}
/**
 * (Highstock) Options for annotation's shapes. Each shape inherits options from
 * the shapeOptions object. An option from the shapeOptions can be overwritten
 * by config for a specific shape.
 */
export interface AnnotationsFibonacciShapeOptions {
    /**
     * (Highstock) Name of the dash style to use for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The height of the shape.
     */
    height?: number;
    /**
     * (Highstock) The radius of the shape.
     */
    r?: number;
    /**
     * (Highstock) Defines additional snapping area around an annotation making
     * this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highstock) The URL for an image to use as the annotation shape. Note,
     * type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highstock) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highstock) The pixel stroke width of the shape.
     */
    strokeWidth?: number;
    /**
     * (Highstock) The type of the shape, e.g. circle or rectangle.
     */
    type?: string;
    /**
     * (Highstock) The width of the shape.
     */
    width?: number;
}
/**
 * (Highstock) An array with options for the labels.
 */
export interface AnnotationsFibonacciTypeLabelsOptions {
    /**
     * (Highstock) Accessibility options for an annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highstock) The alignment of the annotation's label. If right, the right
     * side of the label should be touching the point.
     */
    align?: AlignValue;
    /**
     * (Highstock) Whether to allow the annotation's labels to overlap. To make
     * the labels less sensitive for overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highstock) Whether to hide the annotation's label that is outside the
     * plot area.
     */
    crop?: boolean;
    /**
     * (Highstock) The label's pixel distance from the point.
     */
    distance?: number;
    /**
     * (Highstock) A format string for the data label.
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the annotation's
     * label. Note that if a `format` or `text` are defined, the format or text
     * take precedence and the formatter is ignored. `This` refers to a point
     * object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highstock) Whether the annotation is visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highstock) How to handle the annotation's label that flow outside the
     * plot area. The justify option aligns the label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highstock) When either the borderWidth or the backgroundColor is set,
     * this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highstock) The shadow of the box. The shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     */
    shape?: string;
    /**
     * (Highstock) Styles for the annotation's label.
     */
    style?: CSSObject;
    /**
     * (Highstock) Alias for the format option.
     */
    text?: string;
    /**
     * (Highstock) Whether to use HTML to render the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of the annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highstock) The x position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highstock) Line options.
 */
export interface AnnotationsFibonacciTypeLineOptions {
    fill?: string;
}
/**
 * (Highstock) Additional options for an annotation with the type.
 */
export interface AnnotationsFibonacciTypeOptions {
    /**
     * (Highstock) An array of background colors: Default to: (see online
     * documentation for example)
     */
    backgroundColors?: object;
    /**
     * (Highstock) The height of the fibonacci in terms of yAxis.
     */
    height?: number;
    /**
     * (Highstock) An array with options for the labels.
     */
    labels?: Array<AnnotationsFibonacciTypeLabelsOptions>;
    /**
     * (Highstock) Line options.
     */
    line?: AnnotationsFibonacciTypeLineOptions;
    /**
     * (Highstock) The color of line.
     */
    lineColor?: string;
    /**
     * (Highstock) An array of colors for the lines.
     */
    lineColors?: object;
    points?: Array<AnnotationsFibonacciTypePointsOptions>;
    /**
     * (Highstock) This number defines which xAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    xAxis?: number;
    /**
     * (Highstock) This number defines which yAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    yAxis?: number;
}
export interface AnnotationsFibonacciTypePointsOptions {
    controlPoint?: number;
    /**
     * (Highstock) The x position of the point.
     */
    x?: number;
    /**
     * (Highstock) The y position of the point.
     */
    y?: number;
}
export interface AnnotationsInfinityLineControlPointOptions {
    events?: any;
}
/**
 * (Highstock) Options for annotation's labels. Each label inherits options from
 * the labelOptions object. An option from the labelOptions can be overwritten
 * by config for a specific label.
 */
export interface AnnotationsInfinityLineLabelOptions {
    /**
     * (Highstock) Accessibility options for an annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highstock) The alignment of the annotation's label. If right, the right
     * side of the label should be touching the point.
     */
    align?: AlignValue;
    /**
     * (Highstock) Whether to allow the annotation's labels to overlap. To make
     * the labels less sensitive for overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highstock) Whether to hide the annotation's label that is outside the
     * plot area.
     */
    crop?: boolean;
    /**
     * (Highstock) The label's pixel distance from the point.
     */
    distance?: number;
    /**
     * (Highstock) A format string for the data label.
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the annotation's
     * label. Note that if a `format` or `text` are defined, the format or text
     * take precedence and the formatter is ignored. `This` refers to a point
     * object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highstock) Whether the annotation is visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highstock) How to handle the annotation's label that flow outside the
     * plot area. The justify option aligns the label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highstock) When either the borderWidth or the backgroundColor is set,
     * this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highstock) The shadow of the box. The shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     */
    shape?: string;
    /**
     * (Highstock) Styles for the annotation's label.
     */
    style?: CSSObject;
    /**
     * (Highstock) Alias for the format option.
     */
    text?: string;
    /**
     * (Highstock) Whether to use HTML to render the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of the annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highstock) The x position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highstock) An infinity line annotation.
 */
export interface AnnotationsInfinityLineOptions {
    controlPointOptions?: AnnotationsInfinityLineControlPointOptions;
    /**
     * (Highstock) Options for annotation's labels. Each label inherits options
     * from the labelOptions object. An option from the labelOptions can be
     * overwritten by config for a specific label.
     */
    labelOptions?: AnnotationsInfinityLineLabelOptions;
    /**
     * (Highstock) Options for annotation's shapes. Each shape inherits options
     * from the shapeOptions object. An option from the shapeOptions can be
     * overwritten by config for a specific shape.
     */
    shapeOptions?: AnnotationsInfinityLineShapeOptions;
    /**
     * (Highstock) Additional options for an annotation with the type.
     */
    typeOptions?: AnnotationsInfinityLineTypeOptions;
}
/**
 * (Highstock) Options for annotation's shapes. Each shape inherits options from
 * the shapeOptions object. An option from the shapeOptions can be overwritten
 * by config for a specific shape.
 */
export interface AnnotationsInfinityLineShapeOptions {
    /**
     * (Highstock) Name of the dash style to use for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The height of the shape.
     */
    height?: number;
    /**
     * (Highstock) The radius of the shape.
     */
    r?: number;
    /**
     * (Highstock) Defines additional snapping area around an annotation making
     * this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highstock) The URL for an image to use as the annotation shape. Note,
     * type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highstock) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highstock) The pixel stroke width of the shape.
     */
    strokeWidth?: number;
    /**
     * (Highstock) The type of the shape, e.g. circle or rectangle.
     */
    type?: string;
    /**
     * (Highstock) The width of the shape.
     */
    width?: number;
}
/**
 * (Highstock) Line options.
 */
export interface AnnotationsInfinityLineTypeLineOptions {
    fill?: string;
}
/**
 * (Highstock) Additional options for an annotation with the type.
 */
export interface AnnotationsInfinityLineTypeOptions {
    /**
     * (Highstock) Line options.
     */
    line?: AnnotationsInfinityLineTypeLineOptions;
    points?: Array<AnnotationsInfinityLineTypePointsOptions>;
    /**
     * (Highstock) This number defines which xAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    xAxis?: number;
    /**
     * (Highstock) This number defines which yAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    yAxis?: number;
}
export interface AnnotationsInfinityLineTypePointsOptions {
    controlPoint?: number;
    /**
     * (Highstock) The x position of the point.
     */
    x?: number;
    /**
     * (Highstock) The y position of the point.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's labels.
 * Each label inherits options from the labelOptions object. An option from the
 * labelOptions can be overwritten by config for a specific label.
 */
export interface AnnotationsLabelOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for an
     * annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the
     * annotation's label. If right, the right side of the label should be
     * touching the point.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow the
     * annotation's labels to overlap. To make the labels less sensitive for
     * overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the
     * annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide the annotation's
     * label that is outside the plot area.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The label's pixel distance from
     * the point.
     */
    distance?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the annotation's label. Note that if a `format` or `text` are
     * defined, the format or text take precedence and the formatter is ignored.
     * `This` refers to a point object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether the annotation is
     * visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle the annotation's
     * label that flow outside the plot area. The justify option aligns the
     * label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the borderWidth or
     * the backgroundColor is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. The
     * shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the annotation's
     * label.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Alias for the format option.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) An array of labels for the
 * annotation. For options that apply to multiple labels, they can be added to
 * the labelOptions.
 */
export interface AnnotationsLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for an
     * annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the
     * annotation's label. If right, the right side of the label should be
     * touching the point.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow the
     * annotation's labels to overlap. To make the labels less sensitive for
     * overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the
     * annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide the annotation's
     * label that is outside the plot area.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The label's pixel distance from
     * the point.
     */
    distance?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the annotation's label. Note that if a `format` or `text` are
     * defined, the format or text take precedence and the formatter is ignored.
     * `This` refers to a point object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether the annotation is
     * visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle the annotation's
     * label that flow outside the plot area. The justify option aligns the
     * label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the borderWidth or
     * the backgroundColor is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This option defines the point to
     * which the label will be connected. It can be either the point which
     * exists in the series - it is referenced by the point's id - or a new
     * point with defined x, y properties and optionally axes.
     */
    point?: (string|AnnotationMockPointOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. The
     * shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the annotation's
     * label.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Alias for the format option.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     */
    y?: number;
}
export interface AnnotationsMeasureControlPointOptions {
    events?: any;
}
/**
 * (Highstock) A measure annotation.
 */
export interface AnnotationsMeasureOptions {
    controlPointOptions?: AnnotationsMeasureControlPointOptions;
    /**
     * (Highstock) Additional options for an annotation with the type.
     */
    typeOptions?: AnnotationsMeasureTypeOptions;
}
export interface AnnotationsMeasureTypeBackgroundOptions {
    /**
     * (Highstock) The color of the rectangle.
     */
    fill?: string;
    /**
     * (Highstock) The color of border.
     */
    stroke?: string;
    /**
     * (Highstock) The width of border.
     */
    strokeWidth?: number;
}
/**
 * (Highstock) Configure a crosshair that is horizontally placed in middle of
 * rectangle.
 */
export interface AnnotationsMeasureTypeCrosshairXOptions {
    /**
     * (Highstock) The dash or dot style of the crosshair's line. For possible
     * values, see this demonstration.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Enable or disable the horizontal crosshair.
     */
    enabled?: boolean;
    /**
     * (Highstock) The marker-end defines the arrowhead that will be drawn at
     * the final vertex of the given crosshair's path.
     */
    markerEnd?: string;
    /**
     * (Highstock) The Z index of the crosshair in annotation.
     */
    zIndex?: number;
}
/**
 * (Highstock) Configure a crosshair that is vertically placed in middle of
 * rectangle.
 */
export interface AnnotationsMeasureTypeCrosshairYOptions {
    /**
     * (Highstock) The dash or dot style of the crosshair's line. For possible
     * values, see this demonstration.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Enable or disable the vertical crosshair.
     */
    enabled?: boolean;
    /**
     * (Highstock) The marker-end defines the arrowhead that will be drawn at
     * the final vertex of the given crosshair's path.
     */
    markerEnd?: OptionsMarkerEndValue;
    /**
     * (Highstock) The Z index of the crosshair in annotation.
     */
    zIndex?: number;
}
export interface AnnotationsMeasureTypeLabelOptions {
    /**
     * (Highstock) Enable or disable the label text (min, max, average, bins
     * values).
     *
     * Defaults to true.
     */
    enabled?: boolean;
    /**
     * (Highstock) Formatter function for the label text.
     *
     * Available data are:
     *
     *  (see online documentation for example)
     */
    formatter?: Function;
    /**
     * (Highstock) CSS styles for the measure label.
     */
    style?: (AnnotationsMeasureTypeLabelStyleOptions|CSSObject);
}
/**
 * (Highstock) CSS styles for the measure label.
 */
export interface AnnotationsMeasureTypeLabelStyleOptions {
    color?: string;
    fontSize?: string;
}
/**
 * (Highstock) Line options.
 */
export interface AnnotationsMeasureTypeLineOptions {
    fill?: string;
}
/**
 * (Highstock) Additional options for an annotation with the type.
 */
export interface AnnotationsMeasureTypeOptions {
    background?: AnnotationsMeasureTypeBackgroundOptions;
    /**
     * (Highstock) Configure a crosshair that is horizontally placed in middle
     * of rectangle.
     */
    crosshairX?: AnnotationsMeasureTypeCrosshairXOptions;
    /**
     * (Highstock) Configure a crosshair that is vertically placed in middle of
     * rectangle.
     */
    crosshairY?: AnnotationsMeasureTypeCrosshairYOptions;
    label?: AnnotationsMeasureTypeLabelOptions;
    /**
     * (Highstock) Line options.
     */
    line?: AnnotationsMeasureTypeLineOptions;
    points?: Array<AnnotationsMeasureTypePointsOptions>;
    /**
     * (Highstock) Decides in what dimensions the user can resize by dragging
     * the mouse. Can be one of x, y or xy.
     */
    selectType?: string;
    /**
     * (Highstock) This number defines which xAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    xAxis?: number;
    /**
     * (Highstock) This number defines which yAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the yAxis array.
     */
    yAxis?: number;
}
export interface AnnotationsMeasureTypePointsOptions {
    controlPoint?: number;
    /**
     * (Highstock) The x position of the point.
     */
    x?: number;
    /**
     * (Highstock) The y position of the point.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A basic type of an annotation. It
 * allows to add custom labels or shapes. The items can be tied to points, axis
 * coordinates or chart pixel coordinates.
 */
export interface AnnotationsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `annotation`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|AnnotationsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's control
     * points. Each control point inherits options from controlPointOptions
     * object. Options from the controlPointOptions can be overwritten by
     * options in a specific control point.
     */
    controlPointOptions?: AnnotationControlPointOptionsObject;
    /**
     * (Highstock) A crooked line annotation.
     */
    crookedLine?: AnnotationsCrookedLineOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Allow an annotation to be
     * draggable by a user. Possible values are `'x'`, `'xy'`, `'y'` and `''`
     * (disabled).
     */
    draggable?: AnnotationDraggableValue;
    /**
     * (Highstock) An elliott wave annotation.
     */
    elliottWave?: AnnotationsElliottWaveOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Events available in annotations.
     */
    events?: AnnotationsEventsOptions;
    /**
     * (Highstock) A fibonacci annotation.
     */
    fibonacci?: AnnotationsFibonacciOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Sets an ID for an annotation.
     * Can be user later when removing an annotation in
     * Chart.removeAnnotation(id) method.
     */
    id?: (number|string);
    /**
     * (Highstock) An infinity line annotation.
     */
    infinityLine?: AnnotationsInfinityLineOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's labels.
     * Each label inherits options from the labelOptions object. An option from
     * the labelOptions can be overwritten by config for a specific label.
     */
    labelOptions?: AnnotationsLabelOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array of labels for the
     * annotation. For options that apply to multiple labels, they can be added
     * to the labelOptions.
     */
    labels?: Array<AnnotationsLabelsOptions>;
    /**
     * (Highstock) A measure annotation.
     */
    measure?: AnnotationsMeasureOptions;
    /**
     * (Highstock) A pitchfork annotation.
     */
    pitchfork?: AnnotationsPitchforkOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's shapes.
     * Each shape inherits options from the shapeOptions object. An option from
     * the shapeOptions can be overwritten by config for a specific shape.
     */
    shapeOptions?: AnnotationsShapeOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array of shapes for the
     * annotation. For options that apply to multiple shapes, then can be added
     * to the shapeOptions.
     */
    shapes?: Array<AnnotationsShapesOptions>;
    /**
     * (Highstock) A tunnel annotation.
     */
    tunnel?: AnnotationsTunnelOptions;
    /**
     * (Highstock) A vertical line annotation.
     */
    verticalLine?: AnnotationsVerticalLineOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether the annotation is
     * visible.
     */
    visible?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the annotation.
     */
    zIndex?: number;
}
export interface AnnotationsPitchforkControlPointOptions {
    events?: any;
}
/**
 * (Highstock) Options for annotation's labels. Each label inherits options from
 * the labelOptions object. An option from the labelOptions can be overwritten
 * by config for a specific label.
 */
export interface AnnotationsPitchforkLabelOptions {
    /**
     * (Highstock) Accessibility options for an annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highstock) The alignment of the annotation's label. If right, the right
     * side of the label should be touching the point.
     */
    align?: AlignValue;
    /**
     * (Highstock) Whether to allow the annotation's labels to overlap. To make
     * the labels less sensitive for overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highstock) Whether to hide the annotation's label that is outside the
     * plot area.
     */
    crop?: boolean;
    /**
     * (Highstock) The label's pixel distance from the point.
     */
    distance?: number;
    /**
     * (Highstock) A format string for the data label.
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the annotation's
     * label. Note that if a `format` or `text` are defined, the format or text
     * take precedence and the formatter is ignored. `This` refers to a point
     * object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highstock) Whether the annotation is visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highstock) How to handle the annotation's label that flow outside the
     * plot area. The justify option aligns the label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highstock) When either the borderWidth or the backgroundColor is set,
     * this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highstock) The shadow of the box. The shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     */
    shape?: string;
    /**
     * (Highstock) Styles for the annotation's label.
     */
    style?: CSSObject;
    /**
     * (Highstock) Alias for the format option.
     */
    text?: string;
    /**
     * (Highstock) Whether to use HTML to render the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of the annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highstock) The x position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highstock) A pitchfork annotation.
 */
export interface AnnotationsPitchforkOptions {
    controlPointOptions?: AnnotationsPitchforkControlPointOptions;
    /**
     * (Highstock) Options for annotation's labels. Each label inherits options
     * from the labelOptions object. An option from the labelOptions can be
     * overwritten by config for a specific label.
     */
    labelOptions?: AnnotationsPitchforkLabelOptions;
    /**
     * (Highstock) Options for annotation's shapes. Each shape inherits options
     * from the shapeOptions object. An option from the shapeOptions can be
     * overwritten by config for a specific shape.
     */
    shapeOptions?: AnnotationsPitchforkShapeOptions;
    /**
     * (Highstock) Additional options for an annotation with the type.
     */
    typeOptions?: AnnotationsPitchforkTypeOptions;
}
/**
 * (Highstock) Options for annotation's shapes. Each shape inherits options from
 * the shapeOptions object. An option from the shapeOptions can be overwritten
 * by config for a specific shape.
 */
export interface AnnotationsPitchforkShapeOptions {
    /**
     * (Highstock) Name of the dash style to use for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The height of the shape.
     */
    height?: number;
    /**
     * (Highstock) The radius of the shape.
     */
    r?: number;
    /**
     * (Highstock) Defines additional snapping area around an annotation making
     * this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highstock) The URL for an image to use as the annotation shape. Note,
     * type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highstock) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highstock) The pixel stroke width of the shape.
     */
    strokeWidth?: number;
    /**
     * (Highstock) The type of the shape, e.g. circle or rectangle.
     */
    type?: string;
    /**
     * (Highstock) The width of the shape.
     */
    width?: number;
}
/**
 * (Highstock) Inner background options.
 */
export interface AnnotationsPitchforkTypeInnerBackgroundOptions {
    /**
     * (Highstock) Name of the dash style to use for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) The color of the shape's fill.
     */
    fill?: string;
    /**
     * (Highstock) Defines additional snapping area around an annotation making
     * this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highstock) The URL for an image to use as the annotation shape. Note,
     * type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highstock) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highstock) The pixel stroke width of the shape.
     */
    strokeWidth?: number;
}
/**
 * (Highstock) Line options.
 */
export interface AnnotationsPitchforkTypeLineOptions {
    fill?: string;
}
/**
 * (Highstock) Additional options for an annotation with the type.
 */
export interface AnnotationsPitchforkTypeOptions {
    /**
     * (Highstock) Inner background options.
     */
    innerBackground?: AnnotationsPitchforkTypeInnerBackgroundOptions;
    /**
     * (Highstock) Line options.
     */
    line?: AnnotationsPitchforkTypeLineOptions;
    /**
     * (Highstock) Outer background options.
     */
    outerBackground?: AnnotationsPitchforkTypeOuterBackgroundOptions;
    points?: Array<AnnotationsPitchforkTypePointsOptions>;
    /**
     * (Highstock) This number defines which xAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    xAxis?: number;
    /**
     * (Highstock) This number defines which yAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    yAxis?: number;
}
/**
 * (Highstock) Outer background options.
 */
export interface AnnotationsPitchforkTypeOuterBackgroundOptions {
    /**
     * (Highstock) Name of the dash style to use for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) The color of the shape's fill.
     */
    fill?: string;
    /**
     * (Highstock) Defines additional snapping area around an annotation making
     * this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highstock) The URL for an image to use as the annotation shape. Note,
     * type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highstock) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highstock) The pixel stroke width of the shape.
     */
    strokeWidth?: number;
}
export interface AnnotationsPitchforkTypePointsOptions {
    controlPoint?: number;
    /**
     * (Highstock) The x position of the point.
     */
    x?: number;
    /**
     * (Highstock) The y position of the point.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's shapes.
 * Each shape inherits options from the shapeOptions object. An option from the
 * shapeOptions can be overwritten by config for a specific shape.
 */
export interface AnnotationsShapeOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Name of the dash style to use
     * for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The height of the shape.
     */
    height?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The radius of the shape.
     */
    r?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Defines additional snapping area
     * around an annotation making this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The URL for an image to use as
     * the annotation shape. Note, type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel stroke width of the
     * shape.
     */
    strokeWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The type of the shape, e.g.
     * circle or rectangle.
     */
    type?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the shape.
     */
    width?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) An array of shapes for the
 * annotation. For options that apply to multiple shapes, then can be added to
 * the shapeOptions.
 */
export interface AnnotationsShapesOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Name of the dash style to use
     * for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The height of the shape.
     */
    height?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Id of the marker which will be
     * drawn at the final vertex of the path. Custom markers can be defined in
     * defs property.
     */
    markerEnd?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Id of the marker which will be
     * drawn at the first vertex of the path. Custom markers can be defined in
     * defs property.
     */
    markerStart?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This option defines the point to
     * which the shape will be connected. It can be either the point which
     * exists in the series - it is referenced by the point's id - or a new
     * point with defined x, y properties and optionally axes.
     */
    point?: (string|AnnotationMockPointOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array of points for the
     * shape. This option is available for shapes which can use multiple points
     * such as path. A point can be either a point object or a point's id.
     */
    points?: Array<(string|AnnotationMockPointOptionsObject)>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The radius of the shape.
     */
    r?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Defines additional snapping area
     * around an annotation making this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The URL for an image to use as
     * the annotation shape. Note, type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel stroke width of the
     * shape.
     */
    strokeWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The type of the shape, e.g.
     * circle or rectangle.
     */
    type?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the shape.
     */
    width?: number;
}
export interface AnnotationsTunnelControlPointOptions {
    events?: any;
}
/**
 * (Highstock) Options for annotation's labels. Each label inherits options from
 * the labelOptions object. An option from the labelOptions can be overwritten
 * by config for a specific label.
 */
export interface AnnotationsTunnelLabelOptions {
    /**
     * (Highstock) Accessibility options for an annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highstock) The alignment of the annotation's label. If right, the right
     * side of the label should be touching the point.
     */
    align?: AlignValue;
    /**
     * (Highstock) Whether to allow the annotation's labels to overlap. To make
     * the labels less sensitive for overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highstock) Whether to hide the annotation's label that is outside the
     * plot area.
     */
    crop?: boolean;
    /**
     * (Highstock) The label's pixel distance from the point.
     */
    distance?: number;
    /**
     * (Highstock) A format string for the data label.
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the annotation's
     * label. Note that if a `format` or `text` are defined, the format or text
     * take precedence and the formatter is ignored. `This` refers to a point
     * object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highstock) Whether the annotation is visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highstock) How to handle the annotation's label that flow outside the
     * plot area. The justify option aligns the label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highstock) When either the borderWidth or the backgroundColor is set,
     * this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highstock) The shadow of the box. The shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     */
    shape?: string;
    /**
     * (Highstock) Styles for the annotation's label.
     */
    style?: CSSObject;
    /**
     * (Highstock) Alias for the format option.
     */
    text?: string;
    /**
     * (Highstock) Whether to use HTML to render the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of the annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highstock) The x position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highstock) A tunnel annotation.
 */
export interface AnnotationsTunnelOptions {
    controlPointOptions?: AnnotationsTunnelControlPointOptions;
    /**
     * (Highstock) Options for annotation's labels. Each label inherits options
     * from the labelOptions object. An option from the labelOptions can be
     * overwritten by config for a specific label.
     */
    labelOptions?: AnnotationsTunnelLabelOptions;
    /**
     * (Highstock) Options for annotation's shapes. Each shape inherits options
     * from the shapeOptions object. An option from the shapeOptions can be
     * overwritten by config for a specific shape.
     */
    shapeOptions?: AnnotationsTunnelShapeOptions;
    /**
     * (Highstock) Additional options for an annotation with the type.
     */
    typeOptions?: AnnotationsTunnelTypeOptions;
}
/**
 * (Highstock) Options for annotation's shapes. Each shape inherits options from
 * the shapeOptions object. An option from the shapeOptions can be overwritten
 * by config for a specific shape.
 */
export interface AnnotationsTunnelShapeOptions {
    /**
     * (Highstock) Name of the dash style to use for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The height of the shape.
     */
    height?: number;
    /**
     * (Highstock) The radius of the shape.
     */
    r?: number;
    /**
     * (Highstock) Defines additional snapping area around an annotation making
     * this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highstock) The URL for an image to use as the annotation shape. Note,
     * type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highstock) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highstock) The pixel stroke width of the shape.
     */
    strokeWidth?: number;
    /**
     * (Highstock) The type of the shape, e.g. circle or rectangle.
     */
    type?: string;
    /**
     * (Highstock) The width of the shape.
     */
    width?: number;
}
/**
 * (Highstock) Background options.
 */
export interface AnnotationsTunnelTypeBackgroundOptions {
    fill?: string;
    strokeWidth?: number;
}
/**
 * (Highstock) Options for the control point which controls the annotation's
 * height.
 */
export interface AnnotationsTunnelTypeHeightControlPointOptions {
    events?: any;
}
/**
 * (Highstock) Line options.
 */
export interface AnnotationsTunnelTypeLineOptions {
    fill?: string;
    strokeWidth?: number;
}
/**
 * (Highstock) Additional options for an annotation with the type.
 */
export interface AnnotationsTunnelTypeOptions {
    /**
     * (Highstock) Background options.
     */
    background?: (object|AnnotationsTunnelTypeBackgroundOptions);
    /**
     * (Highstock) The height of the annotation in terms of yAxis.
     */
    height?: number;
    /**
     * (Highstock) Options for the control point which controls the annotation's
     * height.
     */
    heightControlPoint?: AnnotationsTunnelTypeHeightControlPointOptions;
    /**
     * (Highstock) Line options.
     */
    line?: AnnotationsTunnelTypeLineOptions;
    points?: Array<AnnotationsTunnelTypePointsOptions>;
    /**
     * (Highstock) This number defines which xAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    xAxis?: number;
    /**
     * (Highstock) This number defines which yAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    yAxis?: number;
}
export interface AnnotationsTunnelTypePointsOptions {
    controlPoint?: number;
    /**
     * (Highstock) The x position of the point.
     */
    x?: number;
    /**
     * (Highstock) The y position of the point.
     */
    y?: number;
}
/**
 * (Highstock) Options for annotation's labels. Each label inherits options from
 * the labelOptions object. An option from the labelOptions can be overwritten
 * by config for a specific label.
 */
export interface AnnotationsVerticalLineLabelOptions {
    /**
     * (Highstock) Accessibility options for an annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highstock) The alignment of the annotation's label. If right, the right
     * side of the label should be touching the point.
     */
    align?: AlignValue;
    /**
     * (Highstock) Whether to allow the annotation's labels to overlap. To make
     * the labels less sensitive for overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highstock) Whether to hide the annotation's label that is outside the
     * plot area.
     */
    crop?: boolean;
    /**
     * (Highstock) The label's pixel distance from the point.
     */
    distance?: number;
    /**
     * (Highstock) A format string for the data label.
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the annotation's
     * label. Note that if a `format` or `text` are defined, the format or text
     * take precedence and the formatter is ignored. `This` refers to a point
     * object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highstock) Whether the annotation is visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highstock) How to handle the annotation's label that flow outside the
     * plot area. The justify option aligns the label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highstock) When either the borderWidth or the backgroundColor is set,
     * this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highstock) The shadow of the box. The shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     */
    shape?: string;
    /**
     * (Highstock) Styles for the annotation's label.
     */
    style?: CSSObject;
    /**
     * (Highstock) Alias for the format option.
     */
    text?: string;
    /**
     * (Highstock) Whether to use HTML to render the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of the annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highstock) The x position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highstock) A vertical line annotation.
 */
export interface AnnotationsVerticalLineOptions {
    /**
     * (Highstock) Options for annotation's labels. Each label inherits options
     * from the labelOptions object. An option from the labelOptions can be
     * overwritten by config for a specific label.
     */
    labelOptions?: AnnotationsVerticalLineLabelOptions;
    /**
     * (Highstock) Options for annotation's shapes. Each shape inherits options
     * from the shapeOptions object. An option from the shapeOptions can be
     * overwritten by config for a specific shape.
     */
    shapeOptions?: AnnotationsVerticalLineShapeOptions;
    /**
     * (Highstock) Additional options for an annotation with the type.
     */
    typeOptions?: AnnotationsVerticalLineTypeOptions;
}
/**
 * (Highstock) Options for annotation's shapes. Each shape inherits options from
 * the shapeOptions object. An option from the shapeOptions can be overwritten
 * by config for a specific shape.
 */
export interface AnnotationsVerticalLineShapeOptions {
    /**
     * (Highstock) Name of the dash style to use for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The height of the shape.
     */
    height?: number;
    /**
     * (Highstock) The radius of the shape.
     */
    r?: number;
    /**
     * (Highstock) Defines additional snapping area around an annotation making
     * this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highstock) The URL for an image to use as the annotation shape. Note,
     * type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highstock) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highstock) The pixel stroke width of the shape.
     */
    strokeWidth?: number;
    /**
     * (Highstock) The type of the shape, e.g. circle or rectangle.
     */
    type?: string;
    /**
     * (Highstock) The width of the shape.
     */
    width?: number;
}
/**
 * (Highstock) Connector options.
 */
export interface AnnotationsVerticalLineTypeConnectorOptions {
    /**
     * (Highstock) Name of the dash style to use for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    markerEnd?: string;
    /**
     * (Highstock) Defines additional snapping area around an annotation making
     * this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highstock) The URL for an image to use as the annotation shape. Note,
     * type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highstock) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highstock) The pixel stroke width of the shape.
     */
    strokeWidth?: number;
}
/**
 * (Highstock) Label options.
 */
export interface AnnotationsVerticalLineTypeLabelOptions {
    /**
     * (Highstock) Accessibility options for an annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highstock) The alignment of the annotation's label. If right, the right
     * side of the label should be touching the point.
     */
    align?: AlignValue;
    /**
     * (Highstock) Whether to allow the annotation's labels to overlap. To make
     * the labels less sensitive for overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the annotation's label.
     */
    backgroundColor?: string;
    /**
     * (Highstock) The border color for the annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highstock) Whether to hide the annotation's label that is outside the
     * plot area.
     */
    crop?: boolean;
    /**
     * (Highstock) The label's pixel distance from the point.
     */
    distance?: number;
    /**
     * (Highstock) A format string for the data label.
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the annotation's
     * label. Note that if a `format` or `text` are defined, the format or text
     * take precedence and the formatter is ignored. `This` refers to a point
     * object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highstock) Whether the annotation is visible in the exported data table.
     */
    includeInDataExport?: boolean;
    offset?: number;
    /**
     * (Highstock) How to handle the annotation's label that flow outside the
     * plot area. The justify option aligns the label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highstock) When either the borderWidth or the backgroundColor is set,
     * this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highstock) The shadow of the box. The shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     */
    shape?: string;
    /**
     * (Highstock) Styles for the annotation's label.
     */
    style?: CSSObject;
    /**
     * (Highstock) Alias for the format option.
     */
    text?: string;
    /**
     * (Highstock) Whether to use HTML to render the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of the annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highstock) The x position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point.
     * Note that if a `distance` is defined, the distance takes precedence over
     * `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highstock) Line options.
 */
export interface AnnotationsVerticalLineTypeLineOptions {
    fill?: string;
}
/**
 * (Highstock) Additional options for an annotation with the type.
 */
export interface AnnotationsVerticalLineTypeOptions {
    /**
     * (Highstock) Connector options.
     */
    connector?: AnnotationsVerticalLineTypeConnectorOptions;
    /**
     * (Highstock) Label options.
     */
    label?: AnnotationsVerticalLineTypeLabelOptions;
    /**
     * (Highstock) Line options.
     */
    line?: AnnotationsVerticalLineTypeLineOptions;
    points?: Array<AnnotationsVerticalLineTypePointsOptions>;
    /**
     * (Highstock) This number defines which xAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    xAxis?: number;
    /**
     * (Highstock) This number defines which yAxis the point is connected to. It
     * refers to either the axis id or the index of the axis in the xAxis array.
     */
    yAxis?: number;
}
export interface AnnotationsVerticalLineTypePointsOptions {
    controlPoint?: number;
    /**
     * (Highstock) The x position of the point.
     */
    x?: number;
    /**
     * (Highstock) The y position of the point.
     */
    y?: number;
}
/**
 * (Highcharts) Accessibility options for an axis. Requires the accessibility
 * module.
 */
export interface AxisAccessibilityOptionsObject {
    /**
     * (Highcharts) Description for an axis to expose to screen reader users.
     */
    description?: string;
    /**
     * (Highcharts) Enable axis accessibility features, including axis
     * information in the screen reader information region. If this is disabled
     * on the xAxis, the x values are not exposed to screen readers for the
     * individual data points by default.
     */
    enabled?: boolean;
    /**
     * (Highcharts) Range description for an axis. Overrides the default range
     * description. Set to empty to disable range description for this axis.
     */
    rangeDescription?: string;
}
/**
 * (Highstock) A label on the axis next to the crosshair.
 *
 * In styled mode, the label is styled with the `.highcharts-crosshair-label`
 * class.
 */
export interface AxisCrosshairLabelOptions {
    /**
     * (Highstock) Alignment of the label compared to the axis. Defaults to
     * `"left"` for right-side axes, `"right"` for left-side axes and `"center"`
     * for horizontal axes.
     */
    align?: AlignValue;
    /**
     * (Highstock) The background color for the label. Defaults to the related
     * series color, or `#666666` if that is not available.
     */
    backgroundColor?: ColorType;
    /**
     * (Highstock) The border color for the crosshair label
     */
    borderColor?: ColorType;
    /**
     * (Highstock) The border corner radius of the crosshair label.
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width for the crosshair label.
     */
    borderWidth?: number;
    /**
     * (Highstock) Flag to enable crosshair's label.
     */
    enabled?: boolean;
    /**
     * (Highstock) A format string for the crosshair label. Defaults to
     * `{value}` for numeric axes and `{value:%b %d, %Y}` for datetime axes.
     */
    format?: string;
    /**
     * (Highstock) Formatter function for the label text.
     */
    formatter?: XAxisCrosshairLabelFormatterCallbackFunction;
    /**
     * (Highstock) Padding inside the crosshair label.
     */
    padding?: number;
    /**
     * (Highstock) The shape to use for the label box.
     */
    shape?: string;
    /**
     * (Highstock) Text styles for the crosshair label.
     */
    style?: CSSObject;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Configure a crosshair that follows
 * either the mouse pointer or the hovered point.
 *
 * In styled mode, the crosshairs are styled in the `.highcharts-crosshair`,
 * `.highcharts-crosshair-thin` or `.highcharts-xaxis-category` classes.
 */
export interface AxisCrosshairOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the crosshair,
     * especially as a hook for styling.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the crosshair.
     * Defaults to `#cccccc` for numeric and datetime axes, and
     * `rgba(204,214,235,0.25)` for category axes, where the crosshair by
     * default highlights the whole category.
     */
    color?: ColorType;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The dash style for the
     * crosshair. See plotOptions.series.dashStyle for possible values.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) A label on the axis next to the crosshair.
     *
     * In styled mode, the label is styled with the
     * `.highcharts-crosshair-label` class.
     */
    label?: AxisCrosshairLabelOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether the crosshair should
     * snap to the point or follow the pointer independent of points.
     */
    snap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of the
     * crosshair. Defaults to 1 for numeric or datetime axes, and for one
     * category width for category axes.
     */
    width?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the crosshair.
     * Higher Z indices allow drawing the crosshair on top of the series or
     * behind the grid lines.
     */
    zIndex?: number;
}
/**
 * (Gantt) Show an indicator on the axis for the current date and time. Can be a
 * boolean or a configuration object similar to xAxis.plotLines.
 */
export interface AxisCurrentDateIndicatorOptions {
    /**
     * (Highstock) Flag to decide if plotLine should be rendered across all
     * panes.
     */
    acrossPanes?: boolean;
    /**
     * (Gantt) A custom class name, in addition to the default
     * `highcharts-plot-line`, to apply to each individual line.
     */
    className?: string;
    /**
     * (Gantt) The color of the line.
     */
    color?: ColorString;
    /**
     * (Gantt) The dashing or dot style for the plot line. For possible values
     * see this overview.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Gantt) An object defining mouse events for the plot line. Supported
     * properties are `click`, `mouseover`, `mouseout`, `mousemove`.
     */
    events?: (ColorAxisCurrentDateIndicatorEventsOptions|NavigatorXAxisCurrentDateIndicatorEventsOptions|XAxisCurrentDateIndicatorEventsOptions|ZAxisCurrentDateIndicatorEventsOptions);
    /**
     * (Gantt) An id used for identifying the plot line in Axis.removePlotLine.
     */
    id?: string;
    /**
     * (Gantt) Text labels for the plot bands
     */
    label?: (ColorAxisCurrentDateIndicatorLabelOptions|NavigatorXAxisCurrentDateIndicatorLabelOptions|XAxisCurrentDateIndicatorLabelOptions|ZAxisCurrentDateIndicatorLabelOptions);
    /**
     * (Gantt) The width or thickness of the plot line.
     */
    width?: number;
    /**
     * (Gantt) The z index of the plot line within the chart.
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highstock, Gantt) For a datetime axis, the scale will
 * automatically adjust to the appropriate unit. This member gives the default
 * string representations used for each unit. For intermediate values, different
 * units may be used, for example the `day` unit can be used on midnight and
 * `hour` unit be used for intermediate values on the same axis.
 *
 * For an overview of the replacement codes, see dateFormat.
 *
 * Defaults to: (see online documentation for example)
 */
export interface AxisDateTimeLabelFormatsOptions {
    day?: (string|AxisDateTimeLabelFormatsOptionsObject);
    hour?: (string|AxisDateTimeLabelFormatsOptionsObject);
    millisecond?: (string|AxisDateTimeLabelFormatsOptionsObject);
    minute?: (string|AxisDateTimeLabelFormatsOptionsObject);
    month?: (string|AxisDateTimeLabelFormatsOptionsObject);
    second?: (string|AxisDateTimeLabelFormatsOptionsObject);
    week?: (string|AxisDateTimeLabelFormatsOptionsObject);
    year?: (string|AxisDateTimeLabelFormatsOptionsObject);
}
export interface AxisDateTimeLabelFormatsOptionsObject {
    main?: string;
    range?: boolean;
}
export interface AxisLabelsFormatterContextObject<T> {
    axis: Axis;
    chart: Chart;
    isFirst: boolean;
    isLast: boolean;
    pos: number;
    value: T;
}
/**
 * Options for the path on the Axis to be calculated.
 */
export interface AxisPlotLinePathOptionsObject {
    /**
     * Used in Highstock. When `true`, plot paths (crosshair, plotLines,
     * gridLines) will be rendered on all axes when defined on the first axis.
     */
    acrossPanes?: boolean;
    /**
     * If `false`, the function will return null when it falls outside the axis
     * bounds. If `true`, the function will return a path aligned to the plot
     * area sides if it falls outside. If `pass`, it will return a path outside.
     */
    force?: (boolean|string);
    /**
     * Line width used for calculation crisp line coordinates. Defaults to 1.
     */
    lineWidth?: number;
    /**
     * Use old coordinates (for resizing and rescaling). If not set, defaults to
     * `false`.
     */
    old?: boolean;
    /**
     * Used in Polar axes. Reverse the positions for concatenation of polygonal
     * plot bands
     */
    reverse?: boolean;
    /**
     * If given, return the plot line path of a pixel position on the axis.
     */
    translatedValue?: number;
    /**
     * Axis value.
     */
    value?: number;
}
export interface AxisPointBreakEventObject {
    brk: Dictionary<number>;
    point: Point;
    preventDefault: Function;
    target: SVGElement;
    type: ("pointBreak"|"pointInBreak");
}
export interface AxisSetExtremesEventObject extends ExtremesObject {
    preventDefault: Function;
    target: SVGElement;
    trigger: (string|AxisExtremesTriggerValue);
    type: "setExtremes";
}
/**
 * Time ticks.
 */
export interface AxisTickPositionsArray extends Array<number> {
    info?: TimeTicksInfoObject;
}
/**
 * Bounding box of an element.
 */
export interface BBoxObject extends PositionObject {
    /**
     * Height of the bounding box.
     */
    height: number;
    /**
     * Width of the bounding box.
     */
    width: number;
    /**
     * Horizontal position of the bounding box.
     */
    x: number;
    /**
     * Vertical position of the bounding box.
     */
    y: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Debugging options for boost. Useful
 * for benchmarking, and general timing.
 */
export interface BoostDebugOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Show the number of points
     * skipped through culling.
     *
     * When set to true, the number of points skipped in series processing is
     * outputted. Points are skipped if they are closer than 1 pixel from each
     * other.
     */
    showSkipSummary?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Time the WebGL to SVG buffer
     * copy
     *
     * After rendering, the result is copied to an image which is injected into
     * the SVG.
     *
     * If this property is set to true, the time it takes for the buffer copy to
     * complete is outputted.
     */
    timeBufferCopy?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Time the building of the k-d
     * tree.
     *
     * This outputs the time spent building the k-d tree used for markers etc.
     *
     * Note that the k-d tree is built async, and runs post-rendering.
     * Following, it does not affect the performance of the rendering itself.
     */
    timeKDTree?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Time the series rendering.
     *
     * This outputs the time spent on actual rendering in the console when set
     * to true.
     */
    timeRendering?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Time the series processing.
     *
     * This outputs the time spent on transforming the series data to vertex
     * buffers when set to true.
     */
    timeSeriesProcessing?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Time the the WebGL setup.
     *
     * This outputs the time spent on setting up the WebGL context, creating
     * shaders, and textures.
     */
    timeSetup?: boolean;
}
export interface BoostOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) If set to true, the whole chart
     * will be boosted if one of the series crosses its threshold, and all the
     * series can be boosted.
     */
    allowForce?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Debugging options for boost.
     * Useful for benchmarking, and general timing.
     */
    debug?: BoostDebugOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable boost on a
     * chart.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Set the series threshold for
     * when the boost should kick in globally.
     *
     * Setting to e.g. 20 will cause the whole chart to enter boost mode if
     * there are 20 or more series active. When the chart is in boost mode,
     * every series in it will be rendered to a common canvas. This offers a
     * significant speed improvment in charts with a very high amount of series.
     */
    seriesThreshold?: (number|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable GPU
     * translations. GPU translations are faster than doing the translation in
     * JavaScript.
     *
     * This option may cause rendering issues with certain datasets. Namely, if
     * your dataset has large numbers with small increments (such as
     * timestamps), it won't work correctly. This is due to floating point
     * precission.
     */
    useGPUTranslations?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable pre-allocation
     * of vertex buffers.
     *
     * Enabling this will make it so that the binary data arrays required for
     * storing the series data will be allocated prior to transforming the data
     * to a WebGL-compatible format.
     *
     * This saves a copy operation on the order of O(n) and so is significantly
     * more performant. However, this is currently an experimental option, and
     * may cause visual artifacts with some datasets.
     *
     * As such, care should be taken when using this setting to make sure that
     * it doesn't cause any rendering glitches with the given use-case.
     */
    usePreallocated?: boolean;
}
export interface BubbleLegendFormatterContextObject {
    /**
     * The center y position of the range.
     */
    center: number;
    /**
     * The radius of the bubble range.
     */
    radius: number;
    /**
     * The bubble value.
     */
    value: number;
}
/**
 * The chart caption. The caption has an `update` method that allows modifying
 * the options directly or indirectly via `chart.update`.
 */
export interface CaptionObject extends SVGElement {
    /**
     * Modify options for the caption.
     *
     * @param captionOptions
     *        Options to modify.
     *
     * @param redraw
     *        Whether to redraw the chart after the caption is altered. If doing
     *        more operations on the chart, it is a good idea to set redraw to
     *        false and call Chart#redraw after.
     */
    update(captionOptions: CaptionOptions, redraw?: boolean): void;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The chart's caption, which will
 * render below the chart and will be part of exported charts. The caption can
 * be updated after chart initialization through the `Chart.update` or
 * `Chart.caption.update` methods.
 */
export interface CaptionOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The horizontal alignment of the
     * caption. Can be one of "left", "center" and "right".
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When the caption is floating,
     * the plot area will not move to make space for it.
     */
    floating?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the caption
     * and the plot area.
     */
    margin?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the caption.
     *
     * In styled mode, the caption style is given in the `.highcharts-caption`
     * class.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The caption text of the chart.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the text.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * caption. Can be one of `"top"`, `"middle"` and `"bottom"`. When middle,
     * the caption behaves as floating.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position of the caption
     * relative to the alignment within `chart.spacingLeft` and
     * `chart.spacingRight`.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position of the caption
     * relative to the alignment within `chart.spacingTop` and
     * `chart.spacingBottom`.
     */
    y?: number;
}
/**
 * (Highcharts) The back side of the frame around a 3D chart.
 */
export interface Chart3dFrameBackOptions {
    /**
     * (Highcharts) The color of the panel.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) The bottom of the frame around a 3D chart.
 */
export interface Chart3dFrameBottomOptions {
    /**
     * (Highcharts) The color of the panel.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) The front of the frame around a 3D chart.
 */
export interface Chart3dFrameFrontOptions {
    /**
     * (Highcharts) The color of the panel.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) The left side of the frame around a 3D chart.
 */
export interface Chart3dFrameLeftOptions {
    /**
     * (Highcharts) The color of the panel.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) Provides the option to draw a frame around the charts by
 * defining a bottom, front and back panel.
 */
export interface Chart3dFrameOptions {
    /**
     * (Highcharts) The back side of the frame around a 3D chart.
     */
    back?: Chart3dFrameBackOptions;
    /**
     * (Highcharts) The bottom of the frame around a 3D chart.
     */
    bottom?: Chart3dFrameBottomOptions;
    /**
     * (Highcharts) The front of the frame around a 3D chart.
     */
    front?: Chart3dFrameFrontOptions;
    /**
     * (Highcharts) The left side of the frame around a 3D chart.
     */
    left?: Chart3dFrameLeftOptions;
    /**
     * (Highcharts) The right of the frame around a 3D chart.
     */
    right?: Chart3dFrameRightOptions;
    /**
     * (Highcharts) General pixel thickness for the frame faces.
     */
    size?: number;
    /**
     * (Highcharts) The top of the frame around a 3D chart.
     */
    top?: Chart3dFrameTopOptions;
    /**
     * (Highcharts) Whether the frames are visible.
     */
    visible?: string;
}
/**
 * (Highcharts) The right of the frame around a 3D chart.
 */
export interface Chart3dFrameRightOptions {
    /**
     * (Highcharts) The color of the panel.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) The top of the frame around a 3D chart.
 */
export interface Chart3dFrameTopOptions {
    /**
     * (Highcharts) The color of the panel.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) Options to render charts in 3 dimensions. This feature requires
 * `highcharts-3d.js`, found in the download package or online at
 * code.highcharts.com/highcharts-3d.js.
 */
export interface Chart3dOptions {
    /**
     * (Highcharts) One of the two rotation angles for the chart.
     */
    alpha?: number;
    /**
     * (Highcharts) Set it to `"auto"` to automatically move the labels to the
     * best edge.
     */
    axisLabelPosition?: ("auto"|null);
    /**
     * (Highcharts) One of the two rotation angles for the chart.
     */
    beta?: number;
    /**
     * (Highcharts) The total depth of the chart.
     */
    depth?: number;
    /**
     * (Highcharts) Wether to render the chart using the 3D functionality.
     */
    enabled?: boolean;
    /**
     * (Highcharts) Whether the 3d box should automatically adjust to the chart
     * plot area.
     */
    fitToPlot?: boolean;
    /**
     * (Highcharts) Provides the option to draw a frame around the charts by
     * defining a bottom, front and back panel.
     */
    frame?: Chart3dFrameOptions;
    /**
     * (Highcharts) Defines the distance the viewer is standing in front of the
     * chart, this setting is important to calculate the perspective effect in
     * column and scatter charts. It is not used for 3D pie charts.
     */
    viewDistance?: number;
}
/**
 * Contains common event information. Through the `options` property you can
 * access the series options that were passed to the `addSeries` method.
 */
export interface ChartAddSeriesEventObject {
    /**
     * The series options that were passed to the `addSeries` method.
     */
    options: SeriesOptionsType;
    /**
     * Prevents the default behaviour of the event.
     */
    preventDefault: Function;
    /**
     * The event target.
     */
    target: Chart;
    /**
     * The event type.
     */
    type: "addSeries";
}
/**
 * Contains an axes of the clicked spot.
 */
export interface ChartClickEventAxisObject {
    /**
     * Axis at the clicked spot.
     */
    axis: Axis;
    /**
     * Axis value at the clicked spot.
     */
    value: number;
}
/**
 * Contains information about the clicked spot on the chart. Remember the unit
 * of a datetime axis is milliseconds since 1970-01-01 00:00:00.
 */
export interface ChartClickEventObject extends PointerEventObject {
    /**
     * Information about the x-axis on the clicked spot.
     */
    xAxis: Array<ChartClickEventAxisObject>;
    /**
     * Information about the y-axis on the clicked spot.
     */
    yAxis: Array<ChartClickEventAxisObject>;
    /**
     * Information about the z-axis on the clicked spot.
     */
    zAxis?: Array<ChartClickEventAxisObject>;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Event listeners for the chart.
 */
export interface ChartEventsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires when a series is added to
     * the chart after load time, using the `addSeries` method. One parameter,
     * `event`, is passed to the function, containing common event information.
     * Through `event.options` you can access the series options that were
     * passed to the `addSeries` method. Returning false prevents the series
     * from being added.
     */
    addSeries?: ChartAddSeriesCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires after a chart is printed
     * through the context menu item or the `Chart.print` method.
     */
    afterPrint?: ExportingAfterPrintCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires before a chart is printed
     * through the context menu item or the `Chart.print` method.
     */
    beforePrint?: ExportingBeforePrintCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires when clicking on the plot
     * background. One parameter, `event`, is passed to the function, containing
     * common event information.
     *
     * Information on the clicked spot can be found through `event.xAxis` and
     * `event.yAxis`, which are arrays containing the axes of each dimension and
     * each axis' value at the clicked spot. The primary axes are
     * `event.xAxis[0]` and `event.yAxis[0]`. Remember the unit of a datetime
     * axis is milliseconds since 1970-01-01 00:00:00. (see online documentation
     * for example)
     */
    click?: ChartClickCallbackFunction;
    /**
     * (Highcharts, Highmaps) Fires when a drilldown point is clicked, before
     * the new series is added. This event is also utilized for async drilldown,
     * where the seriesOptions are not added by option, but rather loaded async.
     * Note that when clicking a category label to trigger multiple series
     * drilldown, one `drilldown` event is triggered per point in the category.
     *
     * Event arguments:
     *
     * - `category`: If a category label was clicked, which index.
     *
     * - `originalEvent`: The original browser event (usually click) that
     * triggered the drilldown.
     *
     * - `point`: The originating point.
     *
     * - `points`: If a category label was clicked, this array holds all points
     * corresponding to the category.
     *
     * - `seriesOptions`: Options for the new series.
     */
    drilldown?: DrilldownCallbackFunction;
    /**
     * (Highcharts, Highmaps) Fires when drilling up from a drilldown series.
     */
    drillup?: DrillupCallbackFunction;
    /**
     * (Highcharts, Highmaps) In a chart with multiple drilldown series, this
     * event fires after all the series have been drilled up.
     */
    drillupall?: DrillupAllCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback that fires while
     * exporting data. This allows the modification of data rows before
     * processed into the final format.
     */
    exportData?: ExportDataCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires when the chart is finished
     * loading. Since v4.2.2, it also waits for images to be loaded, for example
     * from point markers. One parameter, `event`, is passed to the function,
     * containing common event information.
     *
     * There is also a second parameter to the chart constructor where a
     * callback function can be passed to be executed on chart.load.
     */
    load?: ChartLoadCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires when the chart is redrawn,
     * either after a call to `chart.redraw()` or after an axis, series or point
     * is modified with the `redraw` option set to `true`. One parameter,
     * `event`, is passed to the function, containing common event information.
     */
    redraw?: ChartRedrawCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires after initial load of the
     * chart (directly after the `load` event), and after each redraw (directly
     * after the `redraw` event).
     */
    render?: ChartRenderCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires when an area of the chart
     * has been selected. Selection is enabled by setting the chart's zoomType.
     * One parameter, `event`, is passed to the function, containing common
     * event information. The default action for the selection event is to zoom
     * the chart to the selected area. It can be prevented by calling
     * `event.preventDefault()` or return false.
     *
     * Information on the selected area can be found through `event.xAxis` and
     * `event.yAxis`, which are arrays containing the axes of each dimension and
     * each axis' min and max values. The primary axes are `event.xAxis[0]` and
     * `event.yAxis[0]`. Remember the unit of a datetime axis is milliseconds
     * since 1970-01-01 00:00:00. (see online documentation for example)
     */
    selection?: ChartSelectionCallbackFunction;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) General options for the chart.
 */
export interface ChartOptions {
    /**
     * (Highcharts, Highstock, Gantt) When using multiple axis, the ticks of two
     * or more opposite axes will automatically be aligned by adding ticks to
     * the axis or axes with the least ticks, as if `tickAmount` were specified.
     *
     * This can be prevented by setting `alignTicks` to false. If the grid lines
     * look messy, it's a good idea to hide them for the secondary axis by
     * setting `gridLineWidth` to 0.
     *
     * If `startOnTick` or `endOnTick` in an Axis options are set to false, then
     * the `alignTicks ` will be disabled for the Axis.
     *
     * Disabled for logarithmic axes.
     */
    alignTicks?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Set the overall animation for
     * all chart updating. Animation can be disabled throughout the chart by
     * setting it to false here. It can be overridden for each individual API
     * method as a function parameter. The only animation not affected by this
     * option is the initial series animation, see plotOptions.series.animation.
     *
     * The animation can either be set as a boolean or a configuration object.
     * If `true`, it will use the 'swing' jQuery easing and a duration of 500
     * ms. If used as a configuration object, the following properties are
     * supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: A string reference to an easing function set on the `Math`
     * object. See the easing demo.
     *
     * When zooming on a series with less than 100 points, the chart redraw will
     * be done with animation, but in case of more data points, it is necessary
     * to set this option to ensure animation on zoom.
     */
    animation?: (boolean|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the outer chart area.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the outer chart
     * border.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The corner radius of the outer
     * chart border.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of the outer
     * chart border.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A CSS class name to apply to the
     * charts container `div`, allowing unique CSS styling for each chart.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In styled mode, this sets how
     * many colors the class names should rotate between. With ten colors,
     * series (or points) are given class names like `highcharts-color-0`,
     * `highcharts-color-0` [...] `highcharts-color-9`. The equivalent in
     * non-styled mode is to set colors using the colors setting.
     */
    colorCount?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to display errors on the
     * chart. When `false`, the errors will be shown only in the console.
     */
    displayErrors?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Event listeners for the chart.
     */
    events?: ChartEventsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An explicit height for the
     * chart. If a _number_, the height is given in pixels. If given a
     * _percentage string_ (for example `'56%'`), the height is given as the
     * percentage of the actual chart width. This allows for preserving the
     * aspect ratio across responsive sizes.
     *
     * By default (when `null`) the height is calculated from the offset height
     * of the containing element, or 400 pixels if the containing element's
     * height is 0.
     */
    height?: (number|string|null);
    /**
     * (Highcharts, Highstock, Gantt) If true, the axes will scale to the
     * remaining visible series once one series is hidden. If false, hiding and
     * showing a series will not affect the axes or the other series. For
     * stacks, once one series within the stack is hidden, the rest of the stack
     * will close in around it even if the axis is not affected.
     */
    ignoreHiddenSeries?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to invert the axes so that the x
     * axis is vertical and y axis is horizontal. When `true`, the x axis is
     * reversed by default.
     */
    inverted?: boolean;
    /**
     * (Highmaps) Default `mapData` for all series. If set to a string, it
     * functions as an index into the `Highcharts.maps` array. Otherwise it is
     * interpreted as map data.
     */
    map?: (string|GeoJSON|Array<any>);
    /**
     * (Highmaps) Set lat/lon transformation definitions for the chart. If not
     * defined, these are extracted from the map data.
     */
    mapTransforms?: any;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the outer
     * edge of the chart and the plot area. The numbers in the array designate
     * top, right, bottom and left respectively. Use the options `marginTop`,
     * `marginRight`, `marginBottom` and `marginLeft` for shorthand setting of
     * one option.
     *
     * By default there is no margin. The actual space is dynamically calculated
     * from the offset of axis labels, axis title, title, subtitle and legend in
     * addition to the `spacingTop`, `spacingRight`, `spacingBottom` and
     * `spacingLeft` options.
     */
    margin?: (number|Array<number>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the bottom
     * outer edge of the chart and the plot area. Use this to set a fixed pixel
     * value for the margin as opposed to the default dynamic margin. See also
     * `spacingBottom`.
     */
    marginBottom?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the left
     * outer edge of the chart and the plot area. Use this to set a fixed pixel
     * value for the margin as opposed to the default dynamic margin. See also
     * `spacingLeft`.
     */
    marginLeft?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the right
     * outer edge of the chart and the plot area. Use this to set a fixed pixel
     * value for the margin as opposed to the default dynamic margin. See also
     * `spacingRight`.
     */
    marginRight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the top outer
     * edge of the chart and the plot area. Use this to set a fixed pixel value
     * for the margin as opposed to the default dynamic margin. See also
     * `spacingTop`.
     */
    marginTop?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback function to override
     * the default function that formats all the numbers in the chart. Returns a
     * string with the formatted number.
     */
    numberFormatter?: NumberFormatterCallbackFunction;
    /**
     * (Highcharts) Options to render charts in 3 dimensions. This feature
     * requires `highcharts-3d.js`, found in the download package or online at
     * code.highcharts.com/highcharts-3d.js.
     */
    options3d?: Chart3dOptions;
    /**
     * (Highcharts, Gantt) Allows setting a key to switch between zooming and
     * panning. Can be one of `alt`, `ctrl`, `meta` (the command key on Mac and
     * Windows key on Windows) or `shift`. The keys are mapped directly to the
     * key properties of the click event argument (`event.altKey`,
     * `event.ctrlKey`, `event.metaKey` and `event.shiftKey`).
     */
    panKey?: OptionsPanKeyValue;
    /**
     * (Highcharts, Highstock, Gantt) Allow panning in a chart. Best used with
     * panKey to combine zooming and panning.
     *
     * On touch devices, when the tooltip.followTouchMove option is `true`
     * (default), panning requires two fingers. To allow panning with one
     * finger, set `followTouchMove` to `false`.
     */
    panning?: ChartPanningOptions;
    /**
     * (Highcharts) Common options for all yAxes rendered in a parallel
     * coordinates plot. This feature requires
     * `modules/parallel-coordinates.js`.
     *
     * The default options are: (see online documentation for example)
     */
    parallelAxes?: (ChartParallelAxesOptions|Array<ChartParallelAxesOptions>);
    /**
     * (Highcharts) Flag to render charts as a parallel coordinates plot. In a
     * parallel coordinates plot (||-coords) by default all required yAxes are
     * generated and the legend is disabled. This feature requires
     * `modules/parallel-coordinates.js`.
     */
    parallelCoordinates?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Equivalent to zoomType, but for multitouch
     * gestures only. By default, the `pinchType` is the same as the `zoomType`
     * setting. However, pinching can be enabled separately in some cases, for
     * example in stock charts where a mouse drag pans the chart, while pinching
     * is enabled. When tooltip.followTouchMove is true, pinchType only applies
     * to two-finger touches.
     */
    pinchType?: OptionsPinchTypeValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the plot area.
     */
    plotBackgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The URL for an image to use as
     * the plot background. To set an image as the background for the entire
     * chart, set a CSS background image to the container element. Note that for
     * the image to be applied to exported charts, its URL needs to be
     * accessible by the export server.
     */
    plotBackgroundImage?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the inner chart or
     * plot area border.
     */
    plotBorderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of the plot area
     * border.
     */
    plotBorderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to apply a drop shadow
     * to the plot area. Requires that plotBackgroundColor be set. The shadow
     * can be an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    plotShadow?: (boolean|CSSObject);
    /**
     * (Highcharts) When true, cartesian charts like line, spline, area and
     * column are transformed into the polar coordinate system. This produces
     * _polar charts_, also known as _radar charts_.
     */
    polar?: boolean;
    /**
     * (Highmaps) Allows to manually load the proj4 library from Highcharts
     * options instead of the `window`. In case of loading the library from a
     * `script` tag, this option is not needed, it will be loaded from there by
     * default.
     */
    proj4?: Function;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to reflow the chart to
     * fit the width of the container div on resizing the window.
     */
    reflow?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The HTML element where the chart
     * will be rendered. If it is a string, the element by that id is used. The
     * HTML element can also be passed by direct reference, or as the first
     * argument of the chart constructor, in which case the option is not
     * needed.
     */
    renderTo?: (string|HTMLDOMElement);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The button that appears after a
     * selection zoom, allowing the user to reset zoom.
     */
    resetZoomButton?: ChartResetZoomButtonOptions;
    /**
     * (Highcharts, Gantt) Options for a scrollable plot area. This feature
     * provides a minimum size for the plot area of the chart. If the size gets
     * smaller than this, typically on mobile devices, a native browser
     * scrollbar is presented. This scrollbar provides smooth scrolling for the
     * contents of the plot area, whereas the title, legend and unaffected axes
     * are fixed.
     *
     * Since v7.1.2, a scrollable plot area can be defined for either horizontal
     * or vertical scrolling, depending on whether the `minWidth` or `minHeight`
     * option is set.
     */
    scrollablePlotArea?: ChartScrollablePlotAreaOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color of the
     * marker square when selecting (zooming in on) an area of the chart.
     */
    selectionMarkerFill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to apply a drop shadow
     * to the outer chart area. Requires that backgroundColor be set. The shadow
     * can be an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|CSSObject);
    /**
     * (Highcharts, Gantt) Whether to show the axes initially. This only applies
     * to empty charts where series are added dynamically, as axes are
     * automatically added to cartesian series.
     */
    showAxes?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The distance between the outer
     * edge of the chart and the content, like title or legend, or axis title
     * and labels if present. The numbers in the array designate top, right,
     * bottom and left respectively. Use the options spacingTop, spacingRight,
     * spacingBottom and spacingLeft options for shorthand setting of one
     * option.
     */
    spacing?: Array<number>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The space between the bottom
     * edge of the chart and the content (plot area, axis title and labels,
     * title, subtitle or legend in top position).
     */
    spacingBottom?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The space between the left edge
     * of the chart and the content (plot area, axis title and labels, title,
     * subtitle or legend in top position).
     */
    spacingLeft?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The space between the right edge
     * of the chart and the content (plot area, axis title and labels, title,
     * subtitle or legend in top position).
     */
    spacingRight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The space between the top edge
     * of the chart and the content (plot area, axis title and labels, title,
     * subtitle or legend in top position).
     */
    spacingTop?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Additional CSS styles to apply
     * inline to the container `div`. Note that since the default font styles
     * are applied in the renderer, it is ignorant of the individual chart
     * options and must be set globally.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to apply styled mode.
     * When in styled mode, no presentational attributes or CSS are applied to
     * the chart SVG. Instead, CSS rules are required to style the chart. The
     * default style sheet is available from
     * `https://code.highcharts.com/css/highcharts.css`.
     */
    styledMode?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The default series type for the
     * chart. Can be any of the chart types listed under plotOptions and series
     * or can be a series provided by an additional module.
     *
     * In TypeScript this option has no effect in sense of typing and instead
     * the `type` option must always be set in the series.
     */
    type?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An explicit width for the chart.
     * By default (when `null`) the width is calculated from the offset width of
     * the containing element.
     */
    width?: (number|string|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Set a key to hold when dragging
     * to zoom the chart. This is useful to avoid zooming while moving points.
     * Should be set different than chart.panKey.
     */
    zoomKey?: OptionsZoomKeyValue;
    /**
     * (Highcharts, Highstock, Gantt) Decides in what dimensions the user can
     * zoom by dragging the mouse. Can be one of `x`, `y` or `xy`.
     */
    zoomType?: OptionsZoomTypeValue;
}
/**
 * (Highcharts, Highstock, Gantt) Allow panning in a chart. Best used with
 * panKey to combine zooming and panning.
 *
 * On touch devices, when the tooltip.followTouchMove option is `true`
 * (default), panning requires two fingers. To allow panning with one finger,
 * set `followTouchMove` to `false`.
 */
export interface ChartPanningOptions {
    /**
     * (Highcharts, Highstock, Gantt) Enable or disable chart panning.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Decides in what dimensions the user can
     * pan the chart. Can be one of `x`, `y`, or `xy`.
     */
    type?: OptionsTypeValue;
}
/**
 * (Highcharts) Event handlers for the axis.
 */
export interface ChartParallelAxesEventsOptions {
    /**
     * (Highcharts, Gantt) An event fired after the breaks have rendered.
     */
    afterBreaks?: AxisEventCallbackFunction;
    /**
     * (Highcharts) As opposed to the `setExtremes` event, this event fires
     * after the final min and max values are computed and corrected for
     * `minRange`.
     *
     * Fires when the minimum and maximum is set for the axis, either by calling
     * the `.setExtremes()` method or by selecting an area in the chart. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in axis
     * values. The actual data extremes are found in `event.dataMin` and
     * `event.dataMax`.
     */
    afterSetExtremes?: AxisSetExtremesEventCallbackFunction;
    /**
     * (Highcharts, Gantt) An event fired when a break from this axis occurs on
     * a point.
     */
    pointBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highcharts, Highstock, Gantt) An event fired when a point falls inside a
     * break from this axis.
     */
    pointInBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highcharts) Fires when the minimum and maximum is set for the axis,
     * either by calling the `.setExtremes()` method or by selecting an area in
     * the chart. One parameter, `event`, is passed to the function, containing
     * common event information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in data
     * values. When an axis is zoomed all the way out from the "Reset zoom"
     * button, `event.min` and `event.max` are null, and the new extremes are
     * set based on `this.dataMin` and `this.dataMax`.
     */
    setExtremes?: AxisSetExtremesEventCallbackFunction;
}
/**
 * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
 */
export interface ChartParallelAxesGridOptions {
    /**
     * (Gantt) Set border color for the label grid lines.
     */
    borderColor?: ColorString;
    /**
     * (Gantt) Set border width of the label grid lines.
     */
    borderWidth?: number;
    /**
     * (Gantt) Set cell height for grid axis labels. By default this is
     * calculated from font size. This option only applies to horizontal axes.
     */
    cellHeight?: number;
    /**
     * (Gantt) Set specific options for each column (or row for horizontal axes)
     * in the grid. Each extra column/row is its own axis, and the axis options
     * can be set here.
     */
    columns?: Array<XAxisOptions>;
    /**
     * (Gantt) Enable grid on the axis labels. Defaults to true for Gantt
     * charts.
     */
    enabled?: boolean;
}
/**
 * (Gantt) Set options on specific levels in a tree grid axis. Takes precedence
 * over labels options.
 */
export interface ChartParallelAxesLabelsLevelsOptions {
    /**
     * (Gantt) Specify the level which the options within this object applies
     * to.
     */
    level?: number;
    style?: CSSObject;
}
/**
 * (Highcharts) The axis labels show the number or category for each tick.
 *
 * Since v8.0.0: Labels are animated in categorized x-axis with updating data if
 * `tickInterval` and `step` is set to 1.
 */
export interface ChartParallelAxesLabelsOptions {
    /**
     * (Highcharts) What part of the string the given position is anchored to.
     * Can be one of `"left"`, `"center"` or `"right"`. The exact position also
     * depends on the `labels.x` setting.
     *
     * Angular gauges and solid gauges defaults to `"center"`. Solid gauges with
     * two labels have additional option `"auto"` for automatic horizontal and
     * vertical alignment.
     */
    align?: string;
    /**
     * (Highcharts, Highstock, Gantt) For horizontal axes, the allowed degrees
     * of label rotation to prevent overlapping labels. If there is enough
     * space, labels are not rotated. As the chart gets narrower, it will start
     * rotating the labels -45 degrees, then remove every second label and try
     * again with rotations 0 and -45 etc. Set it to `false` to disable
     * rotation, which will cause the labels to word-wrap if possible.
     */
    autoRotation?: (false|Array<number>);
    /**
     * (Highcharts, Gantt) When each category width is more than this many
     * pixels, we don't apply auto rotation. Instead, we lay out the axis label
     * with word wrap. A lower limit makes sense when the label contains
     * multiple short words that don't extend the available horizontal space for
     * each label.
     */
    autoRotationLimit?: number;
    /**
     * (Highcharts) Angular gauges and solid gauges only. The label's pixel
     * distance from the perimeter of the plot area.
     *
     * Since v7.1.2: If it's a percentage string, it is interpreted the same as
     * series.radius, so label can be aligned under the gauge's shape.
     */
    distance?: (number|string);
    /**
     * (Highcharts) Enable or disable the axis labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts) A format string for the axis label.
     */
    format?: string;
    /**
     * (Highcharts) Callback JavaScript function to format the label. The value
     * is given by `this.value`. Additional properties for `this` are `axis`,
     * `chart`, `isFirst` and `isLast`. The value of the default label formatter
     * can be retrieved by calling `this.axis.defaultLabelFormatter.call(this)`
     * within the function.
     *
     * Defaults to: (see online documentation for example)
     */
    formatter?: AxisLabelsFormatterCallbackFunction;
    /**
     * (Gantt) The number of pixels to indent the labels per level in a treegrid
     * axis.
     */
    indentation?: number;
    /**
     * (Gantt) Set options on specific levels in a tree grid axis. Takes
     * precedence over labels options.
     */
    levels?: Array<ChartParallelAxesLabelsLevelsOptions>;
    /**
     * (Highcharts) How to handle overflowing labels on horizontal axis. If set
     * to `"allow"`, it will not be aligned at all. By default it `"justify"`
     * labels inside the chart area. If there is room to move it, it will be
     * aligned to the edge, else it will be removed.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highcharts, Gantt) The pixel padding for axis labels, to ensure white
     * space between them.
     */
    padding?: number;
    /**
     * (Highcharts) Defines how the labels are be repositioned according to the
     * 3D chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     */
    position3d?: OptionsPosition3dValue;
    /**
     * (Highcharts, Gantt) Whether to reserve space for the labels. By default,
     * space is reserved for the labels in these cases:
     *
     * * On all horizontal axes.
     *
     * * On vertical axes if `label.align` is `right` on a left-side axis or
     * `left` on a right-side axis.
     *
     * * On vertical axes if `label.align` is `center`.
     *
     * This can be turned off when for example the labels are rendered inside
     * the plot area instead of outside.
     */
    reserveSpace?: boolean;
    /**
     * (Highcharts) Rotation of the labels in degrees.
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis labels will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `labels.position3d`.
     */
    skew3d?: boolean;
    /**
     * (Highcharts) Horizontal axes only. The number of lines to spread the
     * labels over to make room or tighter labels.
     */
    staggerLines?: number;
    /**
     * (Highcharts) To show only every _n_'th label on the axis, set the step to
     * _n_. Setting the step to 2 shows every other label.
     *
     * By default, the step is calculated automatically to avoid overlap. To
     * prevent this, set it to 1\. This usually only happens on a category axis,
     * and is often a sign that you have chosen the wrong axis type.
     *
     * Read more at Axis docs => What axis should I use?
     */
    step?: number;
    /**
     * (Highcharts) CSS styles for the label. Use `whiteSpace: 'nowrap'` to
     * prevent wrapping of category labels. Use `textOverflow: 'none'` to
     * prevent ellipsis (dots).
     *
     * In styled mode, the labels are styled with the `.highcharts-axis-labels`
     * class.
     */
    style?: CSSObject;
    /**
     * (Gantt) The symbol for the collapse and expand icon in a treegrid.
     */
    symbol?: ChartParallelAxesLabelsSymbolOptions;
    /**
     * (Highcharts) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts) The x position offset of all labels relative to the tick
     * positions on the axis. Defaults to -15 for left axis, 15 for right axis.
     */
    x?: number;
    /**
     * (Highcharts) The y position offset of all labels relative to the tick
     * positions on the axis. For polar and radial axis consider the use of the
     * distance option.
     */
    y?: number;
    /**
     * (Highcharts) The Z index for the axis labels.
     */
    zIndex?: number;
}
/**
 * (Gantt) The symbol for the collapse and expand icon in a treegrid.
 */
export interface ChartParallelAxesLabelsSymbolOptions {
    height?: number;
    padding?: number;
    /**
     * (Gantt) The symbol type. Points to a definition function in the
     * `Highcharts.Renderer.symbols` collection.
     */
    type?: SymbolKeyValue;
    width?: number;
    x?: number;
    y?: number;
}
/**
 * (Highcharts) Common options for all yAxes rendered in a parallel coordinates
 * plot. This feature requires `modules/parallel-coordinates.js`.
 *
 * The default options are: (see online documentation for example)
 */
export interface ChartParallelAxesOptions {
    /**
     * (Highcharts) Accessibility options for an axis. Requires the
     * accessibility module.
     */
    accessibility?: AxisAccessibilityOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When using multiple axis, the ticks of two
     * or more opposite axes will automatically be aligned by adding ticks to
     * the axis or axes with the least ticks, as if `tickAmount` were specified.
     *
     * This can be prevented by setting `alignTicks` to false. If the grid lines
     * look messy, it's a good idea to hide them for the secondary axis by
     * setting `gridLineWidth` to 0.
     *
     * If `startOnTick` or `endOnTick` in an Axis options are set to false, then
     * the `alignTicks ` will be disabled for the Axis.
     *
     * Disabled for logarithmic axes.
     */
    alignTicks?: boolean;
    /**
     * (Highcharts) Whether to allow decimals in this axis' ticks. When counting
     * integers, like persons or hits on a web page, decimals should be avoided
     * in the labels.
     */
    allowDecimals?: boolean;
    /**
     * (Highcharts, Gantt) If categories are present for the xAxis, names are
     * used instead of numbers for that axis.
     *
     * Since Highcharts 3.0, categories can also be extracted by giving each
     * point a name and setting axis type to `category`. However, if you have
     * multiple series, best practice remains defining the `categories` array.
     *
     * Example: `categories: ['Apples', 'Bananas', 'Oranges']`
     */
    categories?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) The highest allowed value for
     * automatically computed axis extremes.
     */
    ceiling?: number;
    /**
     * (Highcharts) A class name that opens for styling the axis by CSS,
     * especially in Highcharts styled mode. The class name is applied to group
     * elements for the grid, axis elements and labels.
     */
    className?: string;
    /**
     * (Highcharts) Configure a crosshair that follows either the mouse pointer
     * or the hovered point.
     *
     * In styled mode, the crosshairs are styled in the `.highcharts-crosshair`,
     * `.highcharts-crosshair-thin` or `.highcharts-xaxis-category` classes.
     */
    crosshair?: (boolean|AxisCrosshairOptions);
    /**
     * (Highcharts, Highstock, Gantt) For a datetime axis, the scale will
     * automatically adjust to the appropriate unit. This member gives the
     * default string representations used for each unit. For intermediate
     * values, different units may be used, for example the `day` unit can be
     * used on midnight and `hour` unit be used for intermediate values on the
     * same axis.
     *
     * For an overview of the replacement codes, see dateFormat.
     *
     * Defaults to: (see online documentation for example)
     */
    dateTimeLabelFormats?: AxisDateTimeLabelFormatsOptions;
    /**
     * (Highcharts) Whether to force the axis to end on a tick. Use this option
     * with the `maxPadding` option to control the axis end.
     *
     * This option is always disabled, when panning type is either `y` or `xy`.
     */
    endOnTick?: boolean;
    /**
     * (Highcharts) Event handlers for the axis.
     */
    events?: ChartParallelAxesEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) The lowest allowed value for automatically
     * computed axis extremes.
     */
    floor?: number;
    /**
     * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
     */
    grid?: ChartParallelAxesGridOptions;
    /**
     * (Highcharts, Highstock, Gantt) The Z index of the grid lines.
     */
    gridZIndex?: number;
    /**
     * (Highcharts, Highstock) The height of the Y axis. If it's a number, it is
     * interpreted as pixels.
     *
     * Since Highcharts 2: If it's a percentage string, it is interpreted as
     * percentages of the total plot height.
     */
    height?: (number|string);
    /**
     * (Highcharts) The axis labels show the number or category for each tick.
     *
     * Since v8.0.0: Labels are animated in categorized x-axis with updating
     * data if `tickInterval` and `step` is set to 1.
     */
    labels?: ChartParallelAxesLabelsOptions;
    /**
     * (Highcharts, Highstock) The left position as the horizontal axis. If it's
     * a number, it is interpreted as pixel position relative to the chart.
     *
     * Since Highcharts v5.0.13: If it's a percentage string, it is interpreted
     * as percentages of the plot width, offset from plot area left.
     */
    left?: (number|string);
    /**
     * (Highcharts) The color of the line marking the axis itself.
     *
     * In styled mode, the line stroke is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     */
    lineColor?: ColorType;
    /**
     * (Highcharts) The width of the line marking the axis itself.
     *
     * In styled mode, the stroke width is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) Index of another axis that this axis is
     * linked to. When an axis is linked to a master axis, it will take the same
     * extremes as the master, but as assigned by min or max or by setExtremes.
     * It can be used to show additional info, or to ease reading the chart by
     * duplicating the scales.
     */
    linkedTo?: number;
    /**
     * (Highcharts) If there are multiple axes on the same side of the chart,
     * the pixel margin between the axes. Defaults to 0 on vertical axes, 15 on
     * horizontal axes.
     */
    margin?: number;
    /**
     * (Highcharts) The maximum value of the axis. If `null`, the max value is
     * automatically calculated.
     *
     * If the endOnTick option is true, the `max` value might be rounded up.
     *
     * If a tickAmount is set, the axis may be extended beyond the set max in
     * order to reach the given number of ticks. The same may happen in a chart
     * with multiple axes, determined by chart. alignTicks, where a `tickAmount`
     * is applied internally.
     */
    max?: (number|null);
    /**
     * (Highstock) Maximal size of a resizable axis. Could be set as a percent
     * of plot area or pixel size.
     */
    maxLength?: (number|string);
    /**
     * (Highcharts, Highstock, Gantt) Padding of the max value relative to the
     * length of the axis. A padding of 0.05 will make a 100px axis 5px longer.
     * This is useful when you don't want the highest data value to appear on
     * the edge of the plot area. When the axis' `max` option is set or a max
     * extreme is set using `axis.setExtremes()`, the maxPadding will be
     * ignored.
     *
     * Also the `softThreshold` option takes precedence over `maxPadding`, so if
     * the data is tangent to the threshold, `maxPadding` may not apply unless
     * `softThreshold` is set to false.
     */
    maxPadding?: number;
    /**
     * (Highstock, Gantt) Maximum range which can be set using the navigator's
     * handles. Opposite of xAxis.minRange.
     */
    maxRange?: number;
    /**
     * (Highcharts) The minimum value of the axis. If `null` the min value is
     * automatically calculated.
     *
     * If the startOnTick option is true (default), the `min` value might be
     * rounded down.
     *
     * The automatically calculated minimum value is also affected by floor,
     * softMin, minPadding, minRange as well as series.threshold and
     * series.softThreshold.
     */
    min?: (number|null);
    /**
     * (Highstock) Minimal size of a resizable axis. Could be set as a percent
     * of plot area or pixel size.
     */
    minLength?: (number|string);
    /**
     * (Highcharts) Color for the minor tick marks.
     */
    minorTickColor?: ColorType;
    /**
     * (Highcharts) Specific tick interval in axis units for the minor ticks. On
     * a linear axis, if `"auto"`, the minor tick interval is calculated as a
     * fifth of the tickInterval. If `null` or `undefined`, minor ticks are not
     * shown.
     *
     * On logarithmic axes, the unit is the power of the value. For example,
     * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
     * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
     * and 10, 10 and 100 etc.
     *
     * If user settings dictate minor ticks to become too dense, they don't make
     * sense, and will be ignored to prevent performance problems.
     */
    minorTickInterval?: (number|string|null);
    /**
     * (Highcharts) The pixel length of the minor tick marks.
     */
    minorTickLength?: number;
    /**
     * (Highcharts) The position of the minor tick marks relative to the axis
     * line. Can be one of `inside` and `outside`.
     */
    minorTickPosition?: OptionsMinorTickPositionValue;
    /**
     * (Highcharts) Enable or disable minor ticks. Unless minorTickInterval is
     * set, the tick interval is calculated as a fifth of the `tickInterval`.
     *
     * On a logarithmic axis, minor ticks are laid out based on a best guess,
     * attempting to enter approximately 5 minor ticks between each major tick.
     *
     * Prior to v6.0.0, ticks were unabled in auto layout by setting
     * `minorTickInterval` to `"auto"`.
     */
    minorTicks?: boolean;
    /**
     * (Highcharts) The pixel width of the minor tick mark.
     */
    minorTickWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) Padding of the min value relative to the
     * length of the axis. A padding of 0.05 will make a 100px axis 5px longer.
     * This is useful when you don't want the lowest data value to appear on the
     * edge of the plot area. When the axis' `min` option is set or a max
     * extreme is set using `axis.setExtremes()`, the maxPadding will be
     * ignored.
     *
     * Also the `softThreshold` option takes precedence over `minPadding`, so if
     * the data is tangent to the threshold, `minPadding` may not apply unless
     * `softThreshold` is set to false.
     */
    minPadding?: number;
    /**
     * (Highcharts) The minimum range to display on this axis. The entire axis
     * will not be allowed to span over a smaller interval than this. For
     * example, for a datetime axis the main unit is milliseconds. If minRange
     * is set to 3600000, you can't zoom in more than to one hour.
     *
     * The default minRange for the x axis is five times the smallest interval
     * between any of the data points.
     *
     * On a logarithmic axis, the unit for the minimum range is the power. So a
     * minRange of 1 means that the axis can be zoomed to 10-100, 100-1000,
     * 1000-10000 etc.
     *
     * **Note**: The `minPadding`, `maxPadding`, `startOnTick` and `endOnTick`
     * settings also affect how the extremes of the axis are computed.
     */
    minRange?: number;
    /**
     * (Highcharts) The minimum tick interval allowed in axis values. For
     * example on zooming in on an axis with daily data, this can be used to
     * prevent the axis from showing hours. Defaults to the closest distance
     * between two points on the axis.
     */
    minTickInterval?: number;
    /**
     * (Highcharts) The distance in pixels from the plot area to the axis line.
     * A positive offset moves the axis with it's line, labels and ticks away
     * from the plot area. This is typically used when two or more axes are
     * displayed on the same side of the plot. With multiple axes the offset is
     * dynamically adjusted to avoid collision, this can be overridden by
     * setting offset explicitly.
     */
    offset?: number;
    /**
     * (Highstock, Highcharts, Gantt) Whether to display the axis on the
     * opposite side of the normal. The normal is on the left side for vertical
     * axes and bottom for horizontal, so the opposite sides will be right and
     * top respectively. This is typically used with dual or multiple axes.
     */
    opposite?: boolean;
    /**
     * (Highcharts) Refers to the index in the panes array. Used for circular
     * gauges and polar charts. When the option is not set then first pane will
     * be used.
     */
    pane?: number;
    /**
     * (Highstock) The zoomed range to display when only defining one or none of
     * `min` or `max`. For example, to show the latest month, a range of one
     * month can be set.
     */
    range?: number;
    /**
     * (Highstock) Options for axis resizing. It adds a thick line between panes
     * which the user can drag in order to resize the panes.
     */
    resize?: ChartParallelAxesResizeOptions;
    /**
     * (Highcharts) Whether to reverse the axis so that the highest number is
     * closest to the origin.
     */
    reversed?: boolean;
    /**
     * (Highcharts, Highstock) If `true`, the first series in a stack will be
     * drawn on top in a positive, non-reversed Y axis. If `false`, the first
     * series is in the base of the stack.
     */
    reversedStacks?: boolean;
    /**
     * (Highcharts) Whether to show the axis line and title when the axis has no
     * data.
     */
    showEmpty?: boolean;
    /**
     * (Highcharts) Whether to show the first tick label.
     */
    showFirstLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to show the last tick label.
     * Defaults to `true` on cartesian charts, and `false` on polar charts.
     */
    showLastLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A soft maximum for the axis. If the series
     * data maximum is less than this, the axis will stay at this maximum, but
     * if the series data maximum is higher, the axis will flex to show all
     * data.
     *
     * **Note**: The series.softThreshold option takes precedence over this
     * option.
     */
    softMax?: number;
    /**
     * (Highcharts, Highstock, Gantt) A soft minimum for the axis. If the series
     * data minimum is greater than this, the axis will stay at this minimum,
     * but if the series data minimum is lower, the axis will flex to show all
     * data.
     *
     * **Note**: The series.softThreshold option takes precedence over this
     * option.
     */
    softMin?: number;
    /**
     * (Highcharts, Highstock, Gantt) For datetime axes, this decides where to
     * put the tick between weeks. 0 = Sunday, 1 = Monday.
     */
    startOfWeek?: number;
    /**
     * (Highcharts, Highstock, Gantt) Whether to force the axis to start on a
     * tick. Use this option with the `maxPadding` option to control the axis
     * start.
     *
     * This option is always disabled, when panning type is either `y` or `xy`.
     */
    startOnTick?: boolean;
    /**
     * (Gantt) For vertical axes only. Setting the static scale ensures that
     * each tick unit is translated into a fixed pixel height. For example,
     * setting the static scale to 24 results in each Y axis category taking up
     * 24 pixels, and the height of the chart adjusts. Adding or removing items
     * will make the chart resize.
     */
    staticScale?: number;
    /**
     * (Highcharts, Highstock, Gantt) The amount of ticks to draw on the axis.
     * This opens up for aligning the ticks of multiple charts or panes within a
     * chart. This option overrides the `tickPixelInterval` option.
     *
     * This option only has an effect on linear axes. Datetime, logarithmic or
     * category axes are not affected.
     */
    tickAmount?: number;
    /**
     * (Highcharts) Color for the main tick marks.
     *
     * In styled mode, the stroke is given in the `.highcharts-tick` class.
     */
    tickColor?: ColorType;
    /**
     * (Highcharts) The interval of the tick marks in axis units. When
     * `undefined`, the tick interval is computed to approximately follow the
     * tickPixelInterval on linear and datetime axes. On categorized axes, a
     * `undefined` tickInterval will default to 1, one category. Note that
     * datetime axes are based on milliseconds, so for example an interval of
     * one day is expressed as `24 * 3600 * 1000`.
     *
     * On logarithmic axes, the tickInterval is based on powers, so a
     * tickInterval of 1 means one tick on each of 0.1, 1, 10, 100 etc. A
     * tickInterval of 2 means a tick of 0.1, 10, 1000 etc. A tickInterval of
     * 0.2 puts a tick on 0.1, 0.2, 0.4, 0.6, 0.8, 1, 2, 4, 6, 8, 10, 20, 40
     * etc.
     *
     * If the tickInterval is too dense for labels to be drawn, Highcharts may
     * remove ticks.
     *
     * If the chart has multiple axes, the alignTicks option may interfere with
     * the `tickInterval` setting.
     */
    tickInterval?: number;
    /**
     * (Highcharts) The pixel length of the main tick marks.
     */
    tickLength?: number;
    /**
     * (Highcharts, Gantt) For categorized axes only. If `on` the tick mark is
     * placed in the center of the category, if `between` the tick mark is
     * placed between categories. The default is `between` if the `tickInterval`
     * is 1, else `on`.
     */
    tickmarkPlacement?: OptionsTickmarkPlacementValue;
    /**
     * (Highcharts) If tickInterval is `null` this option sets the approximate
     * pixel interval of the tick marks. Not applicable to categorized axis.
     *
     * The tick interval is also influenced by the minTickInterval option, that,
     * by default prevents ticks from being denser than the data points.
     */
    tickPixelInterval?: number;
    /**
     * (Highcharts) The position of the major tick marks relative to the axis
     * line. Can be one of `inside` and `outside`.
     */
    tickPosition?: OptionsTickPositionValue;
    /**
     * (Highcharts) A callback function returning array defining where the ticks
     * are laid out on the axis. This overrides the default behaviour of
     * tickPixelInterval and tickInterval. The automatic tick positions are
     * accessible through `this.tickPositions` and can be modified by the
     * callback.
     */
    tickPositioner?: AxisTickPositionerCallbackFunction;
    /**
     * (Highcharts) An array defining where the ticks are laid out on the axis.
     * This overrides the default behaviour of tickPixelInterval and
     * tickInterval.
     */
    tickPositions?: Array<number>;
    /**
     * (Highcharts, Highstock, Gantt) The pixel width of the major tick marks.
     */
    tickWidth?: number;
    /**
     * (Highcharts) Titles for yAxes are taken from xAxis.categories. All
     * options for `xAxis.labels` applies to parallel coordinates titles. For
     * example, to style categories, use xAxis.labels.style.
     */
    title?: ChartParallelAxesTitleOptions;
    /**
     * (Highcharts) Parallel coordinates only. Format that will be used for
     * point.y and available in tooltip.pointFormat as `{point.formattedValue}`.
     * If not set, `{point.formattedValue}` will use other options, in this
     * order:
     *
     * 1. yAxis.labels.format will be used if set
     *
     * 2. If yAxis is a category, then category name will be displayed
     *
     * 3. If yAxis is a datetime, then value will use the same format as yAxis
     * labels
     *
     * 4. If yAxis is linear/logarithmic type, then simple value will be used
     */
    tooltipValueFormat?: string;
    /**
     * (Highcharts, Highstock) The top position of the Y axis. If it's a number,
     * it is interpreted as pixel position relative to the chart.
     *
     * Since Highcharts 2: If it's a percentage string, it is interpreted as
     * percentages of the plot height, offset from plot area top.
     */
    top?: (number|string);
    /**
     * (Highcharts, Gantt) The type of axis. Can be one of `linear`,
     * `logarithmic`, `datetime`, `category` or `treegrid`. Defaults to
     * `treegrid` for Gantt charts, `linear` for other chart types.
     *
     * In a datetime axis, the numbers are given in milliseconds, and tick marks
     * are placed on appropriate values, like full hours or days. In a category
     * or treegrid axis, the point names of the chart's series are used for
     * categories, if a categories array is not defined.
     */
    type?: AxisTypeValue;
    /**
     * (Highcharts, Gantt) Applies only when the axis `type` is `category`. When
     * `uniqueNames` is true, points are placed on the X axis according to their
     * names. If the same point name is repeated in the same or another series,
     * the point is placed on the same X position as other points of the same
     * name. When `uniqueNames` is false, the points are laid out in increasing
     * X positions regardless of their names, and the X axis category will take
     * the name of the last point in each position.
     */
    uniqueNames?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Datetime axis only. An array determining
     * what time intervals the ticks are allowed to fall on. Each array item is
     * an array where the first value is the time unit and the second value
     * another array of allowed multiples.
     *
     * Defaults to: (see online documentation for example)
     */
    units?: Array<[string, (Array<number>|null)]>;
    /**
     * (Highcharts, Highstock, Gantt) Whether axis, including axis title, line,
     * ticks and labels, should be visible.
     */
    visible?: boolean;
    /**
     * (Highcharts, Highstock) The width as the horizontal axis. If it's a
     * number, it is interpreted as pixels.
     *
     * Since Highcharts v5.0.13: If it's a percentage string, it is interpreted
     * as percentages of the total plot width.
     */
    width?: (number|string);
    /**
     * (Highcharts) Whether to zoom axis. If `chart.zoomType` is set, the option
     * allows to disable zooming on an individual axis.
     */
    zoomEnabled?: boolean;
}
/**
 * (Highstock) Contains two arrays of axes that are controlled by control line
 * of the axis.
 */
export interface ChartParallelAxesResizeControlledAxisOptions {
    /**
     * (Highstock) Array of axes that should move out of the way of resizing
     * being done for the current axis. If not set, the next axis will be used.
     */
    next?: Array<(number|string)>;
    /**
     * (Highstock) Array of axes that should move with the current axis while
     * resizing.
     */
    prev?: Array<(number|string)>;
}
/**
 * (Highstock) Options for axis resizing. It adds a thick line between panes
 * which the user can drag in order to resize the panes.
 */
export interface ChartParallelAxesResizeOptions {
    /**
     * (Highstock) Contains two arrays of axes that are controlled by control
     * line of the axis.
     */
    controlledAxis?: ChartParallelAxesResizeControlledAxisOptions;
    /**
     * (Highstock) Cursor style for the control line.
     *
     * In styled mode use class `highcharts-axis-resizer` instead.
     */
    cursor?: string;
    /**
     * (Highstock) Enable or disable resize by drag for the axis.
     */
    enabled?: boolean;
    /**
     * (Highstock) Color of the control line.
     *
     * In styled mode use class `highcharts-axis-resizer` instead.
     */
    lineColor?: ColorString;
    /**
     * (Highstock) Dash style of the control line.
     *
     * In styled mode use class `highcharts-axis-resizer` instead.
     */
    lineDashStyle?: string;
    /**
     * (Highstock) Width of the control line.
     *
     * In styled mode use class `highcharts-axis-resizer` instead.
     */
    lineWidth?: number;
    /**
     * (Highstock) Horizontal offset of the control line.
     */
    x?: number;
    /**
     * (Highstock) Vertical offset of the control line.
     */
    y?: number;
}
/**
 * (Highcharts) Titles for yAxes are taken from xAxis.categories. All options
 * for `xAxis.labels` applies to parallel coordinates titles. For example, to
 * style categories, use xAxis.labels.style.
 */
export interface ChartParallelAxesTitleOptions {
    reserveSpace?: boolean;
    text?: string;
    /**
     * (Highcharts) Alignment of the text, can be `"left"`, `"right"` or
     * `"center"`. Default alignment depends on the title.align:
     *
     * Horizontal axes:
     *
     * - for `align` = `"low"`, `textAlign` is set to `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"`, `textAlign` is set to `right`
     *
     * Vertical axes:
     *
     * - for `align` = `"low"` and `opposite` = `true`, `textAlign` is set to
     * `right`
     *
     * - for `align` = `"low"` and `opposite` = `false`, `textAlign` is set to
     * `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"` and `opposite` = `true` `textAlign` is set to
     * `left`
     *
     * - for `align` = `"high"` and `opposite` = `false` `textAlign` is set to
     * `right`
     */
    textAlign?: AlignValue;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The button that appears after a
 * selection zoom, allowing the user to reset zoom.
 */
export interface ChartResetZoomButtonOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The position of the button.
     */
    position?: (AlignObject|ChartResetZoomButtonPositionOptions);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) What frame the button placement
     * should be related to. Can be either `plotBox` or `spacingBox`.
     */
    relativeTo?: ButtonRelativeToValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A collection of attributes for
     * the button. The object takes SVG attributes like `fill`, `stroke`,
     * `stroke-width` or `r`, the border radius. The theme also supports
     * `style`, a collection of CSS properties for the text. Equivalent
     * attributes for the hover state are given in `theme.states.hover`.
     */
    theme?: SVGAttributes;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The position of the button.
 */
export interface ChartResetZoomButtonPositionOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The horizontal alignment of the
     * button.
     */
    align?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * button.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The horizontal offset of the
     * button.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical offset of the
     * button.
     */
    y?: number;
}
/**
 * (Highcharts, Gantt) Options for a scrollable plot area. This feature provides
 * a minimum size for the plot area of the chart. If the size gets smaller than
 * this, typically on mobile devices, a native browser scrollbar is presented.
 * This scrollbar provides smooth scrolling for the contents of the plot area,
 * whereas the title, legend and unaffected axes are fixed.
 *
 * Since v7.1.2, a scrollable plot area can be defined for either horizontal or
 * vertical scrolling, depending on whether the `minWidth` or `minHeight` option
 * is set.
 */
export interface ChartScrollablePlotAreaOptions {
    /**
     * (Highcharts, Gantt) The minimum height for the plot area. If it gets
     * smaller than this, the plot area will become scrollable.
     */
    minHeight?: number;
    /**
     * (Highcharts, Gantt) The minimum width for the plot area. If it gets
     * smaller than this, the plot area will become scrollable.
     */
    minWidth?: number;
    /**
     * (Highcharts, Gantt) The opacity of mask applied on one of the sides of
     * the plot area.
     */
    opacity?: number;
    /**
     * (Highcharts, Gantt) The initial scrolling position of the scrollable plot
     * area. Ranges from 0 to 1, where 0 aligns the plot area to the left and 1
     * aligns it to the right. Typically we would use 1 if the chart has right
     * aligned Y axes.
     */
    scrollPositionX?: number;
    /**
     * (Highcharts, Gantt) The initial scrolling position of the scrollable plot
     * area. Ranges from 0 to 1, where 0 aligns the plot area to the top and 1
     * aligns it to the bottom.
     */
    scrollPositionY?: number;
}
/**
 * Axis context of the selection.
 */
export interface ChartSelectionAxisContextObject {
    /**
     * The selected Axis.
     */
    axis: Axis;
    /**
     * The maximum axis value, either automatic or set manually.
     */
    max: number;
    /**
     * The minimum axis value, either automatic or set manually.
     */
    min: number;
}
/**
 * The primary axes are `xAxis[0]` and `yAxis[0]`. Remember the unit of a
 * datetime axis is milliseconds since 1970-01-01 00:00:00.
 */
export interface ChartSelectionContextObject extends Event {
    /**
     * Arrays containing the axes of each dimension and each axis' min and max
     * values.
     */
    xAxis: Array<ChartSelectionAxisContextObject>;
    /**
     * Arrays containing the axes of each dimension and each axis' min and max
     * values.
     */
    yAxis: Array<ChartSelectionAxisContextObject>;
}
/**
 * Interface description for a class.
 */
export interface Class<T> extends Function {
    /**
     * Class costructor.
     *
     * @param args
     *        Constructor arguments.
     *
     * @return Class instance.
     */
    new(...args: Array<any>): T;
}
/**
 * (Gantt) An object defining mouse events for the plot line. Supported
 * properties are `click`, `mouseover`, `mouseout`, `mousemove`.
 */
export interface ColorAxisCurrentDateIndicatorEventsOptions {
    /**
     * (Gantt) Click event on a plot band.
     */
    click?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Gantt) Mouse move event on a plot band.
     */
    mousemove?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Gantt) Mouse out event on the corner of a plot band.
     */
    mouseout?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Gantt) Mouse over event on a plot band.
     */
    mouseover?: EventCallbackFunction<PlotLineOrBand>;
}
/**
 * (Gantt) Text labels for the plot bands
 */
export interface ColorAxisCurrentDateIndicatorLabelOptions {
    /**
     * (Gantt) Horizontal alignment of the label. Can be one of "left", "center"
     * or "right".
     */
    align?: AlignValue;
    /**
     * (Gantt) Format of the label. This options is passed as the fist argument
     * to dateFormat function.
     */
    format?: string;
    /**
     * (Gantt) Callback JavaScript function to format the label. Useful
     * properties like the value of plot line or the range of plot band (`from`
     * & `to` properties) can be found in `this.options` object.
     */
    formatter?: FormatterCallbackFunction<PlotLineOrBand>;
    /**
     * (Gantt) Rotation of the text label in degrees. Defaults to 0 for
     * horizontal plot lines and 90 for vertical lines.
     */
    rotation?: number;
    /**
     * (Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-line-label` class.
     */
    style?: CSSObject;
    /**
     * (Gantt) The text itself. A subset of HTML is supported.
     */
    text?: string;
    /**
     * (Gantt) The text alignment for the label. While `align` determines where
     * the texts anchor point is placed within the plot band, `textAlign`
     * determines how the text is aligned against its anchor point. Possible
     * values are "left", "center" and "right". Defaults to the same as the
     * `align` option.
     */
    textAlign?: AlignValue;
    /**
     * (Gantt) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Gantt) Vertical alignment of the label relative to the plot line. Can be
     * one of "top", "middle" or "bottom".
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Gantt) Horizontal position relative the alignment. Default varies by
     * orientation.
     */
    x?: number;
    /**
     * (Gantt) Vertical position of the text baseline relative to the alignment.
     * Default varies by orientation.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps) An array of data classes or ranges for the
 * choropleth map. If none given, the color axis is scalar and values are
 * distributed as a gradient between the minimum and maximum colors.
 */
export interface ColorAxisDataClassesOptions {
    /**
     * (Highcharts, Highstock, Highmaps) The color of each data class. If not
     * set, the color is pulled from the global or chart-specific colors array.
     * In styled mode, this option is ignored. Instead, use colors defined in
     * CSS.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) The start of the value range that the
     * data class represents, relating to the point value.
     *
     * The range of each `dataClass` is closed in both ends, but can be
     * overridden by the next `dataClass`.
     */
    from?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The name of the data class as it
     * appears in the legend. If no name is given, it is automatically created
     * based on the `from` and `to` values. For full programmatic control,
     * legend.labelFormatter can be used. In the formatter, `this.from` and
     * `this.to` can be accessed.
     */
    name?: string;
    /**
     * (Highcharts, Highstock, Highmaps) The end of the value range that the
     * data class represents, relating to the point value.
     *
     * The range of each `dataClass` is closed in both ends, but can be
     * overridden by the next `dataClass`.
     */
    to?: number;
}
/**
 * (Highcharts, Highstock, Highmaps) Event handlers for the axis.
 */
export interface ColorAxisEventsOptions {
    /**
     * (Highcharts, Highstock, Highmaps) As opposed to the `setExtremes` event,
     * this event fires after the final min and max values are computed and
     * corrected for `minRange`.
     *
     * Fires when the minimum and maximum is set for the axis, either by calling
     * the `.setExtremes()` method or by selecting an area in the chart. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in axis
     * values. The actual data extremes are found in `event.dataMin` and
     * `event.dataMax`.
     */
    afterSetExtremes?: AxisSetExtremesEventCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps) Fires when the legend item belonging to
     * the colorAxis is clicked. One parameter, `event`, is passed to the
     * function.
     */
    legendItemClick?: Function;
    /**
     * (Highcharts, Highstock, Highmaps) Fires when the minimum and maximum is
     * set for the axis, either by calling the `.setExtremes()` method or by
     * selecting an area in the chart. One parameter, `event`, is passed to the
     * function, containing common event information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in data
     * values. When an axis is zoomed all the way out from the "Reset zoom"
     * button, `event.min` and `event.max` are null, and the new extremes are
     * set based on `this.dataMin` and `this.dataMax`.
     */
    setExtremes?: AxisSetExtremesEventCallbackFunction;
}
/**
 * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
 */
export interface ColorAxisGridOptions {
    /**
     * (Gantt) Set border color for the label grid lines.
     */
    borderColor?: ColorString;
    /**
     * (Gantt) Set border width of the label grid lines.
     */
    borderWidth?: number;
    /**
     * (Gantt) Set cell height for grid axis labels. By default this is
     * calculated from font size. This option only applies to horizontal axes.
     */
    cellHeight?: number;
    /**
     * (Gantt) Set specific options for each column (or row for horizontal axes)
     * in the grid. Each extra column/row is its own axis, and the axis options
     * can be set here.
     */
    columns?: Array<XAxisOptions>;
    /**
     * (Gantt) Enable grid on the axis labels. Defaults to true for Gantt
     * charts.
     */
    enabled?: boolean;
}
/**
 * (Highcharts, Highstock, Highmaps) The axis labels show the number for each
 * tick.
 *
 * For more live examples on label options, see xAxis.labels in the Highcharts
 * API.
 */
export interface ColorAxisLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps) What part of the string the given
     * position is anchored to. If `left`, the left side of the string is at the
     * axis position. Can be one of `"left"`, `"center"` or `"right"`. Defaults
     * to an intelligent guess based on which side of the chart the axis is on
     * and the rotation of the label.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) For horizontal axes, the allowed degrees
     * of label rotation to prevent overlapping labels. If there is enough
     * space, labels are not rotated. As the chart gets narrower, it will start
     * rotating the labels -45 degrees, then remove every second label and try
     * again with rotations 0 and -45 etc. Set it to `false` to disable
     * rotation, which will cause the labels to word-wrap if possible.
     */
    autoRotation?: (false|Array<number>);
    /**
     * (Highcharts, Gantt) When each category width is more than this many
     * pixels, we don't apply auto rotation. Instead, we lay out the axis label
     * with word wrap. A lower limit makes sense when the label contains
     * multiple short words that don't extend the available horizontal space for
     * each label.
     */
    autoRotationLimit?: number;
    /**
     * (Highcharts, Gantt) Polar charts only. The label's pixel distance from
     * the perimeter of the plot area.
     */
    distance?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Enable or disable the axis labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) A format string for the axis label.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps) Callback JavaScript function to format
     * the label. The value is given by `this.value`. Additional properties for
     * `this` are `axis`, `chart`, `isFirst` and `isLast`. The value of the
     * default label formatter can be retrieved by calling
     * `this.axis.defaultLabelFormatter.call(this)` within the function.
     *
     * Defaults to: (see online documentation for example)
     */
    formatter?: AxisLabelsFormatterCallbackFunction;
    /**
     * (Gantt) The number of pixels to indent the labels per level in a treegrid
     * axis.
     */
    indentation?: number;
    /**
     * (Highcharts, Highstock, Highmaps) How to handle overflowing labels on
     * horizontal color axis. If set to `"allow"`, it will not be aligned at
     * all. By default it `"justify"` labels inside the chart area. If there is
     * room to move it, it will be aligned to the edge, else it will be removed.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highcharts, Gantt) The pixel padding for axis labels, to ensure white
     * space between them.
     */
    padding?: number;
    /**
     * (Highcharts) Defines how the labels are be repositioned according to the
     * 3D chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     */
    position3d?: OptionsPosition3dValue;
    /**
     * (Highcharts, Gantt) Whether to reserve space for the labels. By default,
     * space is reserved for the labels in these cases:
     *
     * * On all horizontal axes.
     *
     * * On vertical axes if `label.align` is `right` on a left-side axis or
     * `left` on a right-side axis.
     *
     * * On vertical axes if `label.align` is `center`.
     *
     * This can be turned off when for example the labels are rendered inside
     * the plot area instead of outside.
     */
    reserveSpace?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) Rotation of the labels in degrees.
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis labels will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `labels.position3d`.
     */
    skew3d?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) Horizontal axes only. The number of
     * lines to spread the labels over to make room or tighter labels.
     */
    staggerLines?: number;
    /**
     * (Highcharts, Highstock, Highmaps) To show only every _n_'th label on the
     * axis, set the step to _n_. Setting the step to 2 shows every other label.
     *
     * By default, the step is calculated automatically to avoid overlap. To
     * prevent this, set it to 1\. This usually only happens on a category axis,
     * and is often a sign that you have chosen the wrong axis type.
     *
     * Read more at Axis docs => What axis should I use?
     */
    step?: number;
    /**
     * (Highcharts, Highstock, Highmaps) CSS styles for the label. Use
     * `whiteSpace: 'nowrap'` to prevent wrapping of category labels. Use
     * `textOverflow: 'none'` to prevent ellipsis (dots).
     *
     * In styled mode, the labels are styled with the `.highcharts-axis-labels`
     * class.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps) Whether to use HTML to render the
     * labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) The x position offset of all labels
     * relative to the tick positions on the axis.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The y position offset of all labels
     * relative to the tick positions on the axis. The default makes it adapt to
     * the font size of the bottom axis.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The Z index for the axis labels.
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highstock, Highmaps) A color axis for series. Visually, the
 * color axis will appear as a gradient or as separate items inside the legend,
 * depending on whether the axis is scalar or based on data classes.
 *
 * For supported color formats, see the docs article about colors.
 *
 * A scalar color axis is represented by a gradient. The colors either range
 * between the minColor and the maxColor, or for more fine grained control the
 * colors can be defined in stops. Often times, the color axis needs to be
 * adjusted to get the right color spread for the data. In addition to stops,
 * consider using a logarithmic axis type, or setting min and max to avoid the
 * colors being determined by outliers.
 *
 * When dataClasses are used, the ranges are subdivided into separate classes
 * like categories based on their values. This can be used for ranges between
 * two values, but also for a true category. However, when your data is
 * categorized, it may be as convenient to add each category to a separate
 * series.
 *
 * Color axis does not work with: `sankey`, `sunburst`, `dependencywheel`,
 * `networkgraph`, `wordcloud`, `venn`, `gauge` and `solidgauge` series types.
 *
 * Since v7.2.0 `colorAxis` can also be an array of options objects.
 *
 * See the Axis object for programmatic access to the axis.
 */
export interface ColorAxisOptions {
    /**
     * (Highcharts, Highstock, Highmaps) Accessibility options for an axis.
     * Requires the accessibility module.
     */
    accessibility?: AxisAccessibilityOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps) Whether to allow decimals on the color
     * axis.
     */
    allowDecimals?: boolean;
    /**
     * (Highcharts) In a polar chart, this is the angle of the Y axis in
     * degrees, where 0 is up and 90 is right. The angle determines the position
     * of the axis line and the labels, though the coordinate system is
     * unaffected. Since v8.0.0 this option is also applicable for X axis
     * (inverted polar).
     */
    angle?: number;
    /**
     * (Highcharts, Highstock, Gantt) The highest allowed value for
     * automatically computed axis extremes.
     */
    ceiling?: number;
    /**
     * (Highcharts, Highstock, Highmaps) A class name that opens for styling the
     * axis by CSS, especially in Highcharts styled mode. The class name is
     * applied to group elements for the grid, axis elements and labels.
     */
    className?: string;
    /**
     * (Gantt) Show an indicator on the axis for the current date and time. Can
     * be a boolean or a configuration object similar to xAxis.plotLines.
     */
    currentDateIndicator?: (boolean|AxisCurrentDateIndicatorOptions);
    /**
     * (Highcharts, Highstock, Highmaps) Determines how to set each data class'
     * color if no individual color is set. The default value, `tween`, computes
     * intermediate colors between `minColor` and `maxColor`. The other possible
     * value, `category`, pulls colors from the global or chart specific colors
     * array.
     */
    dataClassColor?: OptionsDataClassColorValue;
    /**
     * (Highcharts, Highstock, Highmaps) An array of data classes or ranges for
     * the choropleth map. If none given, the color axis is scalar and values
     * are distributed as a gradient between the minimum and maximum colors.
     */
    dataClasses?: Array<ColorAxisDataClassesOptions>;
    /**
     * (Highcharts, Highstock, Highmaps) Whether to force the axis to end on a
     * tick. Use this option with the maxPadding option to control the axis end.
     */
    endOnTick?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) Event handlers for the axis.
     */
    events?: ColorAxisEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) The lowest allowed value for automatically
     * computed axis extremes.
     */
    floor?: number;
    /**
     * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
     */
    grid?: ColorAxisGridOptions;
    /**
     * (Highcharts, Highstock, Highmaps) Color of the grid lines extending from
     * the axis across the gradient.
     */
    gridLineColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) The dash or dot style of the grid
     * lines. For possible values, see this demonstration.
     */
    gridLineDashStyle?: DashStyleValue;
    /**
     * (Highcharts) Polar charts only. Whether the grid lines should draw as a
     * polygon with straight lines between categories, or as circles. Can be
     * either `circle` or `polygon`. Since v8.0.0 this option is also applicable
     * for X axis (inverted polar).
     */
    gridLineInterpolation?: OptionsGridLineInterpolationValue;
    /**
     * (Highcharts, Highstock, Highmaps) The width of the grid lines extending
     * from the axis across the gradient of a scalar color axis.
     */
    gridLineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The Z index of the grid lines.
     */
    gridZIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) An id for the axis. This can be used
     * after render time to get a pointer to the axis object through
     * `chart.get()`.
     */
    id?: string;
    /**
     * (Highcharts, Highstock, Highmaps) The axis labels show the number for
     * each tick.
     *
     * For more live examples on label options, see xAxis.labels in the
     * Highcharts API.
     */
    labels?: ColorAxisLabelsOptions;
    /**
     * (Highcharts, Highstock, Highmaps) The layout of the color axis. Can be
     * `'horizontal'` or `'vertical'`. If none given, the color axis has the
     * same layout as the legend.
     */
    layout?: (string|undefined);
    /**
     * (Highcharts, Highstock, Highmaps) The color of the line marking the axis
     * itself.
     *
     * In styled mode, the line stroke is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     */
    lineColor?: ColorType;
    /**
     * (Highcharts, Highstock, Highmaps) If there are multiple axes on the same
     * side of the chart, the pixel margin between the axes. Defaults to 0 on
     * vertical axes, 15 on horizontal axes.
     */
    margin?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The triangular marker on a scalar color
     * axis that points to the value of the hovered area. To disable the marker,
     * set `marker: null`.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps) The maximum value of the axis in terms
     * of map point values. If `null`, the max value is automatically
     * calculated. If the `endOnTick` option is true, the max value might be
     * rounded up.
     */
    max?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The color to represent the maximum of
     * the color axis. Unless dataClasses or stops are set, the gradient ends at
     * this value.
     *
     * If dataClasses are set, the color is based on minColor and maxColor
     * unless a color is set for each data class, or the dataClassColor is set.
     */
    maxColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) Padding of the max value relative to
     * the length of the axis. A padding of 0.05 will make a 100px axis 5px
     * longer.
     */
    maxPadding?: number;
    /**
     * (Highstock, Gantt) Maximum range which can be set using the navigator's
     * handles. Opposite of xAxis.minRange.
     */
    maxRange?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The minimum value of the axis in terms
     * of map point values. If `null`, the min value is automatically
     * calculated. If the `startOnTick` option is true, the min value might be
     * rounded down.
     */
    min?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The color to represent the minimum of
     * the color axis. Unless dataClasses or stops are set, the gradient starts
     * at this value.
     *
     * If dataClasses are set, the color is based on minColor and maxColor
     * unless a color is set for each data class, or the dataClassColor is set.
     */
    minColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) Color of the minor, secondary grid
     * lines.
     *
     * In styled mode, the stroke width is given in the
     * `.highcharts-minor-grid-line` class.
     */
    minorGridLineColor?: ColorType;
    /**
     * (Highcharts, Highstock, Highmaps) The dash or dot style of the minor grid
     * lines. For possible values, see this demonstration.
     */
    minorGridLineDashStyle?: DashStyleValue;
    /**
     * (Highcharts, Highstock, Highmaps) Width of the minor, secondary grid
     * lines.
     *
     * In styled mode, the stroke width is given in the `.highcharts-grid-line`
     * class.
     */
    minorGridLineWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Color for the minor tick marks.
     */
    minorTickColor?: ColorType;
    /**
     * (Highcharts, Highstock, Highmaps) Specific tick interval in axis units
     * for the minor ticks. On a linear axis, if `"auto"`, the minor tick
     * interval is calculated as a fifth of the tickInterval. If `null` or
     * `undefined`, minor ticks are not shown.
     *
     * On logarithmic axes, the unit is the power of the value. For example,
     * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
     * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
     * and 10, 10 and 100 etc.
     *
     * If user settings dictate minor ticks to become too dense, they don't make
     * sense, and will be ignored to prevent performance problems.
     */
    minorTickInterval?: (number|string|null);
    /**
     * (Highcharts, Highstock, Highmaps) The pixel length of the minor tick
     * marks.
     */
    minorTickLength?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The position of the minor tick marks
     * relative to the axis line. Can be one of `inside` and `outside`.
     */
    minorTickPosition?: OptionsMinorTickPositionValue;
    /**
     * (Highcharts, Highstock, Highmaps) Enable or disable minor ticks. Unless
     * minorTickInterval is set, the tick interval is calculated as a fifth of
     * the `tickInterval`.
     *
     * On a logarithmic axis, minor ticks are laid out based on a best guess,
     * attempting to enter approximately 5 minor ticks between each major tick.
     *
     * Prior to v6.0.0, ticks were unabled in auto layout by setting
     * `minorTickInterval` to `"auto"`.
     */
    minorTicks?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) The pixel width of the minor tick mark.
     */
    minorTickWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Padding of the min value relative to
     * the length of the axis. A padding of 0.05 will make a 100px axis 5px
     * longer.
     */
    minPadding?: number;
    /**
     * (Highstock) In an ordinal axis, the points are equally spaced in the
     * chart regardless of the actual time or x distance between them. This
     * means that missing data periods (e.g. nights or weekends for a stock
     * chart) will not take up space in the chart. Having `ordinal: false` will
     * show any gaps created by the `gapSize` setting proportionate to their
     * duration.
     *
     * In stock charts the X axis is ordinal by default, unless the boost module
     * is used and at least one of the series' data length exceeds the
     * boostThreshold.
     */
    ordinal?: boolean;
    /**
     * (Highstock) Additional range on the right side of the xAxis. Works
     * similar to `xAxis.maxPadding`, but value is set in milliseconds. Can be
     * set for both main `xAxis` and the navigator's `xAxis`.
     */
    overscroll?: number;
    /**
     * (Highstock) The zoomed range to display when only defining one or none of
     * `min` or `max`. For example, to show the latest month, a range of one
     * month can be set.
     */
    range?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Whether to reverse the axis so that the
     * highest number is closest to the origin. Defaults to `false` in a
     * horizontal legend and `true` in a vertical legend, where the smallest
     * value starts on top.
     */
    reversed?: boolean;
    /**
     * (Highstock) An optional scrollbar to display on the X axis in response to
     * limiting the minimum and maximum of the axis values.
     *
     * In styled mode, all the presentational options for the scrollbar are
     * replaced by the classes `.highcharts-scrollbar-thumb`,
     * `.highcharts-scrollbar-arrow`, `.highcharts-scrollbar-button`,
     * `.highcharts-scrollbar-rifles` and `.highcharts-scrollbar-track`.
     */
    scrollbar?: ColorAxisScrollbarOptions;
    /**
     * (Highcharts, Highstock, Highmaps) Whether to show the first tick label.
     */
    showFirstLabel?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) Whether to display the colorAxis in the
     * legend.
     */
    showInLegend?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to show the last tick label.
     * Defaults to `true` on cartesian charts, and `false` on polar charts.
     */
    showLastLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A soft maximum for the axis. If the series
     * data maximum is less than this, the axis will stay at this maximum, but
     * if the series data maximum is higher, the axis will flex to show all
     * data.
     */
    softMax?: number;
    /**
     * (Highcharts, Highstock, Gantt) A soft minimum for the axis. If the series
     * data minimum is greater than this, the axis will stay at this minimum,
     * but if the series data minimum is lower, the axis will flex to show all
     * data.
     */
    softMin?: number;
    /**
     * (Highcharts, Highstock, Gantt) For datetime axes, this decides where to
     * put the tick between weeks. 0 = Sunday, 1 = Monday.
     */
    startOfWeek?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Whether to force the axis to start on a
     * tick. Use this option with the `maxPadding` option to control the axis
     * start.
     */
    startOnTick?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) Color stops for the gradient of a
     * scalar color axis. Use this in cases where a linear gradient between a
     * `minColor` and `maxColor` is not sufficient. The stops is an array of
     * tuples, where the first item is a float between 0 and 1 assigning the
     * relative position in the gradient, and the second item is the color.
     */
    stops?: Array<[number, ColorString]>;
    /**
     * (Highcharts, Highstock, Gantt) The amount of ticks to draw on the axis.
     * This opens up for aligning the ticks of multiple charts or panes within a
     * chart. This option overrides the `tickPixelInterval` option.
     *
     * This option only has an effect on linear axes. Datetime, logarithmic or
     * category axes are not affected.
     */
    tickAmount?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Color for the main tick marks.
     *
     * In styled mode, the stroke is given in the `.highcharts-tick` class.
     */
    tickColor?: ColorType;
    /**
     * (Highcharts, Highstock, Highmaps) The interval of the tick marks in axis
     * units. When `null`, the tick interval is computed to approximately follow
     * the `tickPixelInterval`.
     */
    tickInterval?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The pixel length of the main tick marks
     * on the color axis.
     */
    tickLength?: number;
    /**
     * (Highcharts, Gantt) For categorized axes only. If `on` the tick mark is
     * placed in the center of the category, if `between` the tick mark is
     * placed between categories. The default is `between` if the `tickInterval`
     * is 1, else `on`.
     */
    tickmarkPlacement?: OptionsTickmarkPlacementValue;
    /**
     * (Highcharts, Highstock, Highmaps) If tickInterval is `null` this option
     * sets the approximate pixel interval of the tick marks.
     */
    tickPixelInterval?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The position of the major tick marks
     * relative to the axis line. Can be one of `inside` and `outside`.
     */
    tickPosition?: OptionsTickPositionValue;
    /**
     * (Highcharts, Highstock, Highmaps) A callback function returning array
     * defining where the ticks are laid out on the axis. This overrides the
     * default behaviour of tickPixelInterval and tickInterval. The automatic
     * tick positions are accessible through `this.tickPositions` and can be
     * modified by the callback.
     */
    tickPositioner?: AxisTickPositionerCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps) An array defining where the ticks are
     * laid out on the axis. This overrides the default behaviour of
     * tickPixelInterval and tickInterval.
     */
    tickPositions?: Array<number>;
    /**
     * (Highcharts, Highstock, Highmaps) The pixel width of the major tick
     * marks. Defaults to 0 on category axes, otherwise 1.
     *
     * In styled mode, the stroke width is given in the `.highcharts-tick`
     * class, but in order for the element to be generated on category axes, the
     * option must be explicitly set to 1.
     */
    tickWidth?: (number|undefined);
    /**
     * (Highcharts, Highstock, Highmaps) The type of interpolation to use for
     * the color axis. Can be `linear` or `logarithmic`.
     */
    type?: ColorAxisTypeValue;
    /**
     * (Highcharts, Gantt) Applies only when the axis `type` is `category`. When
     * `uniqueNames` is true, points are placed on the X axis according to their
     * names. If the same point name is repeated in the same or another series,
     * the point is placed on the same X position as other points of the same
     * name. When `uniqueNames` is false, the points are laid out in increasing
     * X positions regardless of their names, and the X axis category will take
     * the name of the last point in each position.
     */
    uniqueNames?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Datetime axis only. An array determining
     * what time intervals the ticks are allowed to fall on. Each array item is
     * an array where the first value is the time unit and the second value
     * another array of allowed multiples.
     *
     * Defaults to: (see online documentation for example)
     */
    units?: Array<[string, (Array<number>|null)]>;
    /**
     * (Highcharts, Highstock, Gantt) Whether axis, including axis title, line,
     * ticks and labels, should be visible.
     */
    visible?: boolean;
}
/**
 * (Highstock) An optional scrollbar to display on the X axis in response to
 * limiting the minimum and maximum of the axis values.
 *
 * In styled mode, all the presentational options for the scrollbar are replaced
 * by the classes `.highcharts-scrollbar-thumb`, `.highcharts-scrollbar-arrow`,
 * `.highcharts-scrollbar-button`, `.highcharts-scrollbar-rifles` and
 * `.highcharts-scrollbar-track`.
 */
export interface ColorAxisScrollbarOptions {
    /**
     * (Highstock) The background color of the scrollbar itself.
     */
    barBackgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of the scrollbar's border.
     */
    barBorderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border rounding radius of the bar.
     */
    barBorderRadius?: number;
    /**
     * (Highstock) The width of the bar's border.
     */
    barBorderWidth?: number;
    /**
     * (Highstock) The color of the small arrow inside the scrollbar buttons.
     */
    buttonArrowColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of scrollbar buttons.
     */
    buttonBackgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of the border of the scrollbar buttons.
     */
    buttonBorderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The corner radius of the scrollbar buttons.
     */
    buttonBorderRadius?: number;
    /**
     * (Highstock) The border width of the scrollbar buttons.
     */
    buttonBorderWidth?: number;
    /**
     * (Highstock) Enable or disable the scrollbar.
     */
    enabled?: boolean;
    /**
     * (Highstock) The height of the scrollbar. The height also applies to the
     * width of the scroll arrows so that they are always squares. Defaults to
     * 20 for touch devices and 14 for mouse devices.
     */
    height?: number;
    /**
     * (Highstock) Whether to redraw the main chart as the scrollbar or the
     * navigator zoomed window is moved. Defaults to `true` for modern browsers
     * and `false` for legacy IE browsers as well as mobile devices.
     */
    liveRedraw?: boolean;
    /**
     * (Highstock) The margin between the scrollbar and its axis when the
     * scrollbar is applied directly to an axis.
     */
    margin?: number;
    /**
     * (Highstock) The minimum width of the scrollbar.
     */
    minWidth?: number;
    /**
     * (Highstock) The color of the small rifles in the middle of the scrollbar.
     */
    rifleColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Whether to show or hide the scrollbar when the scrolled
     * content is zoomed out to it full extent.
     */
    showFull?: boolean;
    /**
     * (Highstock) The color of the track background.
     */
    trackBackgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of the border of the scrollbar track.
     */
    trackBorderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The corner radius of the border of the scrollbar track.
     */
    trackBorderRadius?: number;
    /**
     * (Highstock) The width of the border of the scrollbar track.
     */
    trackBorderWidth?: number;
    /**
     * (Highstock) The z index of the scrollbar group.
     */
    zIndex?: number;
}
export interface ConnectorsAnimationOptionsObject {
    reversed?: boolean;
}
/**
 * (Gantt) Marker options specific to the end markers for this chart's
 * Pathfinder connectors. Overrides the generic marker options.
 */
export interface ConnectorsEndMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     */
    align?: (string|AlignValue);
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     */
    radius?: number;
    /**
     * (Gantt) Set the symbol of the connector end markers.
     */
    symbol?: string;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     */
    width?: number;
}
/**
 * (Gantt) Marker options for this chart's Pathfinder connectors. Note that this
 * option is overridden by the `startMarker` and `endMarker` options.
 */
export interface ConnectorsMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     */
    align?: AlignValue;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     */
    radius?: number;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     */
    width?: number;
}
/**
 * (Gantt) The Pathfinder module allows you to define connections between any
 * two points, represented as lines - optionally with markers for the start
 * and/or end points. Multiple algorithms are available for calculating how the
 * connecting lines are drawn.
 *
 * Connector functionality requires Highcharts Gantt to be loaded. In Gantt
 * charts, the connectors are used to draw dependencies between tasks.
 */
export interface ConnectorsOptions {
    /**
     * (Gantt) Set the default pathfinder margin to use, in pixels. Some
     * Pathfinder algorithms attempt to avoid obstacles, such as other points in
     * the chart. These algorithms use this margin to determine how close lines
     * can be to an obstacle. The default is to compute this automatically from
     * the size of the obstacles in the chart.
     *
     * To draw connecting lines close to existing points, set this to a low
     * number. For more space around existing points, set this number higher.
     */
    algorithmMargin?: number;
    /**
     * (Gantt) Set the default dash style for this chart's connecting lines.
     */
    dashStyle?: string;
    /**
     * (Gantt) Enable connectors for this chart. Requires Highcharts Gantt.
     */
    enabled?: boolean;
    /**
     * (Gantt) Marker options specific to the end markers for this chart's
     * Pathfinder connectors. Overrides the generic marker options.
     */
    endMarker?: ConnectorsEndMarkerOptions;
    /**
     * (Gantt) Set the default color for this chart's Pathfinder connecting
     * lines. Defaults to the color of the point being connected.
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the default pixel width for this chart's Pathfinder
     * connecting lines.
     */
    lineWidth?: number;
    /**
     * (Gantt) Marker options for this chart's Pathfinder connectors. Note that
     * this option is overridden by the `startMarker` and `endMarker` options.
     */
    marker?: ConnectorsMarkerOptions;
    /**
     * (Gantt) Marker options specific to the start markers for this chart's
     * Pathfinder connectors. Overrides the generic marker options.
     */
    startMarker?: ConnectorsStartMarkerOptions;
    /**
     * (Gantt) Set the default pathfinder algorithm to use for this chart. It is
     * possible to define your own algorithms by adding them to the
     * Highcharts.Pathfinder.prototype.algorithms object before the chart has
     * been created.
     *
     * The default algorithms are as follows:
     *
     * `straight`: Draws a straight line between the connecting points. Does not
     * avoid other points when drawing.
     *
     * `simpleConnect`: Finds a path between the points using right angles only.
     * Takes only starting/ending points into account, and will not avoid other
     * points.
     *
     * `fastAvoid`: Finds a path between the points using right angles only.
     * Will attempt to avoid other points, but its focus is performance over
     * accuracy. Works well with less dense datasets.
     *
     * Default value: `straight` is used as default for most series types, while
     * `simpleConnect` is used as default for Gantt series, to show dependencies
     * between points.
     */
    type?: PathfinderTypeValue;
}
/**
 * (Gantt) Marker options specific to the start markers for this chart's
 * Pathfinder connectors. Overrides the generic marker options.
 */
export interface ConnectorsStartMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     */
    align?: (string|AlignValue);
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     */
    enabled?: boolean;
    fill?: string;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     */
    radius?: number;
    /**
     * (Gantt) Set the symbol of the connector start markers.
     */
    symbol?: string;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     */
    width?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Highchart by default puts a credits
 * label in the lower right corner of the chart. This can be changed using these
 * options.
 */
export interface CreditsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to show the credits
     * text.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The URL for the credits label.
     */
    href?: string;
    /**
     * (Highmaps) Credits for map source to be concatenated with conventional
     * credit text. By default this is a format string that collects copyright
     * information from the map if available.
     */
    mapText?: string;
    /**
     * (Highmaps) Detailed credits for map source to be displayed on hover of
     * credits text. By default this is a format string that collects copyright
     * information from the map if available.
     */
    mapTextFull?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Position configuration for the
     * credits label.
     */
    position?: AlignObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the credits
     * label.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text for the credits label.
     */
    text?: string;
}
/**
 * A style object with camel case property names to define visual appearance of
 * a SVG element or HTML element. The properties can be whatever styles are
 * supported on the given SVG or HTML element.
 */
export interface CSSObject {
    [key: string]: (boolean|number|string|undefined);
    /**
     * Background style for the element.
     */
    background?: string;
    /**
     * Background color of the element.
     */
    backgroundColor?: ColorString;
    /**
     * Border style for the element.
     */
    border?: string;
    /**
     * Radius of the element border.
     */
    borderRadius?: number;
    /**
     * Color used in the element. The 'contrast' option is a Highcharts custom
     * property that results in black or white, depending on the background of
     * the element.
     */
    color?: ('contrast'|ColorString);
    /**
     * Style of the mouse cursor when resting over the element.
     */
    cursor?: CursorValue;
    /**
     * Font family of the element text. Multiple values have to be in decreasing
     * preference order and separated by comma.
     */
    fontFamily?: string;
    /**
     * Font size of the element text.
     */
    fontSize?: string;
    /**
     * Font weight of the element text.
     */
    fontWeight?: string;
    /**
     * Height of the element.
     */
    height?: number;
    /**
     * Width of the element border.
     */
    lineWidth?: number;
    /**
     * Opacity of the element.
     */
    opacity?: number;
    /**
     * Space around the element content.
     */
    padding?: string;
    /**
     * Behaviour of the element when the mouse cursor rests over it.
     */
    pointerEvents?: string;
    /**
     * Positioning of the element.
     */
    position?: string;
    /**
     * Alignment of the element text.
     */
    textAlign?: string;
    /**
     * Additional decoration of the element text.
     */
    textDecoration?: string;
    /**
     * Outline style of the element text.
     */
    textOutline?: string;
    /**
     * Line break style of the element text. Highcharts SVG elements support
     * `ellipsis` when a `width` is set.
     */
    textOverflow?: string;
    /**
     * Top spacing of the element relative to the parent element.
     */
    top?: string;
    /**
     * Animated transition of selected element properties.
     */
    transition?: string;
    /**
     * Line break style of the element text.
     */
    whiteSpace?: string;
    /**
     * Width of the element.
     */
    width?: number;
}
export interface DataGroupingInfoObject {
    length: number;
    options?: SeriesOptionsType;
    start: number;
}
/**
 * (Highstock) A custom data grouping object for each button.
 */
export interface DataGroupingOptionsObject {
    /**
     * (Highstock) The method of approximation inside a group. When for example
     * 30 days are grouped into one month, this determines what value should
     * represent the group. Possible values are "average", "averages", "open",
     * "high", "low", "close" and "sum". For OHLC and candlestick series the
     * approximation is "ohlc" by default, which finds the open, high, low and
     * close values within all the grouped data. For ranges, the approximation
     * is "range", which finds the low and high values. For multi-dimensional
     * data, like ranges and OHLC, "averages" will compute the average for each
     * dimension.
     *
     * Custom aggregate methods can be added by assigning a callback function as
     * the approximation. This function takes a numeric array as the argument
     * and should return a single numeric value or `null`. Note that the numeric
     * array will never contain null values, only true numbers. Instead, if null
     * values are present in the raw data, the numeric array will have an
     * `.hasNulls` property set to `true`. For single-value data sets the data
     * is available in the first argument of the callback function. For OHLC
     * data sets, all the open values are in the first argument, all high values
     * in the second etc.
     *
     * Since v4.2.7, grouping meta data is available in the approximation
     * callback from `this.dataGroupInfo`. It can be used to extract information
     * from the raw data.
     *
     * Defaults to `average` for line-type series, `sum` for columns, `range`
     * for range series and `ohlc` for OHLC and candlestick.
     */
    approximation?: (string|DataGroupingApproximationValue|Function);
    /**
     * (Highstock) Datetime formats for the header of the tooltip in a stock
     * chart. The format can vary within a chart depending on the currently
     * selected time range and the current data grouping.
     *
     * The default formats are: (see online documentation for example)
     *
     * For each of these array definitions, the first item is the format used
     * when the active time span is one unit. For instance, if the current data
     * applies to one week, the first item of the week array is used. The second
     * and third items are used when the active time span is more than two
     * units. For instance, if the current data applies to two weeks, the second
     * and third item of the week array are used, and applied to the start and
     * end date of the time span.
     */
    dateTimeLabelFormats?: object;
    /**
     * (Highstock) Enable or disable data grouping.
     */
    enabled?: boolean;
    /**
     * (Highstock) When data grouping is forced, it runs no matter how small the
     * intervals are. This can be handy for example when the sum should be
     * calculated for values appearing at random times within each hour.
     */
    forced?: boolean;
    /**
     * (Highstock) By default only points within the visible range are grouped.
     * Enabling this option will force data grouping to calculate all grouped
     * points for a given dataset. That option prevents for example a column
     * series from calculating a grouped point partially. The effect is similar
     * to Series.getExtremesFromAll but does not affect yAxis extremes.
     */
    groupAll?: boolean;
    /**
     * (Highstock) The approximate pixel width of each group. If for example a
     * series with 30 points is displayed over a 600 pixel wide plot area, no
     * grouping is performed. If however the series contains so many points that
     * the spacing is less than the groupPixelWidth, Highcharts will try to
     * group it into appropriate groups so that each is more or less two pixels
     * wide. If multiple series with different group pixel widths are drawn on
     * the same x axis, all series will take the greatest width. For example,
     * line series have 2px default group width, while column series have 10px.
     * If combined, both the line and the column will have 10px by default.
     */
    groupPixelWidth?: number;
    /**
     * (Highstock) Normally, a group is indexed by the start of that group, so
     * for example when 30 daily values are grouped into one month, that month's
     * x value will be the 1st of the month. This apparently shifts the data to
     * the left. When the smoothed option is true, this is compensated for. The
     * data is shifted to the middle of the group, and min and max values are
     * preserved. Internally, this is used in the Navigator series.
     */
    smoothed?: boolean;
    /**
     * (Highstock) An array determining what time intervals the data is allowed
     * to be grouped to. Each array item is an array where the first value is
     * the time unit and the second value another array of allowed multiples.
     *
     * Defaults to: (see online documentation for example)
     */
    units?: Array<[string, (Array<number>|null)]>;
}
/**
 * (Highcharts, Highstock, Gantt) A declarative filter to control of which data
 * labels to display. The declarative filter is designed for use when callback
 * functions are not available, like when the chart options require a pure JSON
 * structure or for use with graphical editors. For programmatic control, use
 * the `formatter` instead, and return `undefined` to disable a single data
 * label.
 */
export interface DataLabelsFilterOptionsObject {
    /**
     * (Highcharts, Highstock, Gantt) The operator to compare by. Can be one of
     * `>`, `<`, `>=`, `<=`, `==`, and `===`.
     */
    operator?: OptionsOperatorValue;
    /**
     * (Highcharts, Highstock, Gantt) The point property to filter by. Point
     * options are passed directly to properties, additionally there are `y`
     * value, `percentage` and others listed under Highcharts.Point members.
     */
    property?: string;
    /**
     * (Highcharts, Highstock, Gantt) The value to compare against.
     */
    value?: number;
}
/**
 * (Highcharts, Highstock, Gantt) Individual data label for each point. The
 * options are the same as the ones for plotOptions.series.dataLabels.
 */
export interface DataLabelsOptions {
    /**
     * (Highcharts, Highstock, Gantt) The alignment of the data label compared
     * to the point. If `right`, the right side of the label should be touching
     * the point. For points with an extent, like columns, the alignments also
     * dictates how to align it inside the box, as given with the inside option.
     * Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Gantt) Whether to allow data labels to overlap.
     * To make the labels less sensitive for overlapping, the dataLabels.padding
     * can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Enable or disable the initial animation
     * when a series is displayed for the `dataLabels`. The animation can also
     * be set as a configuration object. Please note that this option only
     * applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|SeriesAreaDataDataLabelsAnimationOptions|SeriesArearangeDataDataLabelsAnimationOptions|SeriesAreasplineDataDataLabelsAnimationOptions|
SeriesAreasplinerangeDataDataLabelsAnimationOptions|SeriesBarDataDataLabelsAnimationOptions|SeriesBoxplotDataDataLabelsAnimationOptions|SeriesBubbleDataDataLabelsAnimationOptions|
SeriesBulletDataDataLabelsAnimationOptions|SeriesCandlestickDataDataLabelsAnimationOptions|SeriesColumnDataDataLabelsAnimationOptions|SeriesColumnpyramidDataDataLabelsAnimationOptions|
SeriesColumnrangeDataDataLabelsAnimationOptions|SeriesCylinderDataDataLabelsAnimationOptions|SeriesDumbbellDataDataLabelsAnimationOptions|SeriesFunnelDataDataLabelsAnimationOptions|
SeriesGaugeDataDataLabelsAnimationOptions|SeriesHeatmapDataDataLabelsAnimationOptions|SeriesItemDataDataLabelsAnimationOptions|SeriesLineDataDataLabelsAnimationOptions|
SeriesLollipopDataDataLabelsAnimationOptions|SeriesNetworkgraphDataDataLabelsAnimationOptions|SeriesOhlcDataDataLabelsAnimationOptions|SeriesOrganizationDataDataLabelsAnimationOptions|
SeriesPackedbubbleDataDataLabelsAnimationOptions|SeriesParetoDataDataLabelsAnimationOptions|SeriesPieDataDataLabelsAnimationOptions|SeriesPolygonDataDataLabelsAnimationOptions|
SeriesPyramidDataDataLabelsAnimationOptions|SeriesSankeyDataDataLabelsAnimationOptions|SeriesScatter3dDataDataLabelsAnimationOptions|SeriesScatterDataDataLabelsAnimationOptions|
SeriesSolidgaugeDataDataLabelsAnimationOptions|SeriesSplineDataDataLabelsAnimationOptions|SeriesStreamgraphDataDataLabelsAnimationOptions|SeriesSunburstDataDataLabelsAnimationOptions|
SeriesTilemapDataDataLabelsAnimationOptions|SeriesTimelineDataDataLabelsAnimationOptions|SeriesTreemapDataDataLabelsAnimationOptions|SeriesVariablepieDataDataLabelsAnimationOptions|
SeriesVariwideDataDataLabelsAnimationOptions|SeriesVectorDataDataLabelsAnimationOptions|SeriesVennDataDataLabelsAnimationOptions|SeriesWaterfallDataDataLabelsAnimationOptions|
SeriesWindbarbDataDataLabelsAnimationOptions|SeriesWordcloudDataDataLabelsAnimationOptions|SeriesXrangeDataDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Gantt) The background color or gradient for the
     * data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Gantt) The border color for the data label.
     * Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Gantt) The border radius in pixels for the data
     * label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Gantt) The border width in pixels for the data
     * label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) The text color for the data labels.
     * Defaults to `undefined`. For certain series types, like column or map,
     * the data labels can be drawn inside the points. In this case the data
     * label will be drawn with maximum contrast by default. Additionally, it
     * will be given a `text-outline` style with the opposite color, to further
     * increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Gantt) Whether to hide data labels that are
     * outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Enable or disable the data labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A declarative filter to control of which
     * data labels to display. The declarative filter is designed for use when
     * callback functions are not available, like when the chart options require
     * a pure JSON structure or for use with graphical editors. For programmatic
     * control, use the `formatter` instead, and return `undefined` to disable a
     * single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) A format string for the data label.
     * Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Gantt) Callback JavaScript function to format the
     * data label. Note that if a `format` is defined, the format takes
     * precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Gantt) For points with an extent, like columns or
     * map areas, whether to align the data label inside the box or to the
     * actual value point. Defaults to `false` in most cases, `true` in stacked
     * columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Format for points with the value of null.
     * Works analogously to format. `nullFormat` can be applied only to series
     * which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Gantt) Callback JavaScript function that defines
     * formatting for points with the value of null. Works analogously to
     * formatter. `nullPointFormatter` can be applied only to series which
     * support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Gantt) How to handle data labels that flow
     * outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Gantt) When either the `borderWidth` or the
     * `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Gantt) Aligns data labels relative to points. If
     * `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Text rotation in degrees. Note that due to
     * a more complex structure, backgrounds, borders and padding will be lost
     * on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Gantt) The shadow of the box. Works best with
     * `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be an object
     * configuration containing `color`, `offsetX`, `offsetY`, `opacity` and
     * `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Gantt) The name of a symbol to use for the border
     * around the label. Symbols are predefined functions on the Renderer
     * object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Gantt) Styles for the label. The default `color`
     * setting is `"contrast"`, which is a pseudo color that Highcharts picks up
     * and applies the maximum contrast to the underlying point item, for
     * example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) Options for a label text which should
     * follow marker's shape. Border and background are disabled for a label
     * that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) The vertical alignment of a data label.
     * Can be one of `top`, `middle` or `bottom`. The default value depends on
     * the data, for instance in a column chart, the label is above positive
     * values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Gantt) The x position offset of the label
     * relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock) X offset of the higher data labels relative to
     * the point value.
     */
    xHigh?: number;
    /**
     * (Highcharts, Highstock) X offset of the lower data labels relative to the
     * point value.
     */
    xLow?: number;
    /**
     * (Highcharts, Highstock, Gantt) The y position offset of the label
     * relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock) Y offset of the higher data labels relative to
     * the point value.
     */
    yHigh?: number;
    /**
     * (Highcharts, Highstock) Y offset of the lower data labels relative to the
     * point value.
     */
    yLow?: number;
    /**
     * (Highcharts, Highstock, Gantt) The Z index of the data labels. The
     * default Z index puts it above the series. Use a Z index of 2 to display
     * it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock, Gantt) Options for a label text which should follow
 * marker's shape. Border and background are disabled for a label that follows a
 * path.
 *
 * **Note:** Only SVG-based renderer supports this option. Setting `useHTML` to
 * true will disable this option.
 */
export interface DataLabelsTextPathOptionsObject {
    /**
     * (Highcharts, Highstock, Gantt) Presentation attributes for the text path.
     */
    attributes?: SVGAttributes;
    /**
     * (Highcharts, Highstock, Gantt) Enable or disable `textPath` option for
     * link's or marker's data labels.
     */
    enabled?: boolean;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The Data module provides a
 * simplified interface for adding data to a chart from sources like CVS, HTML
 * tables or grid views. See also the tutorial article on the Data module.
 *
 * It requires the `modules/data.js` file to be loaded.
 *
 * Please note that the default way of adding data in Highcharts, without the
 * need of a module, is through the series._type_.data option.
 */
export interface DataOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A callback function to modify
     * the CSV before parsing it. Return the modified string.
     */
    beforeParse?: DataBeforeParseCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A two-dimensional array
     * representing the input data on tabular form. This input can be used when
     * the data is already parsed, for example from a grid view component. Each
     * cell can be a string or number. If not switchRowsAndColumns is set, the
     * columns are interpreted as series.
     */
    columns?: Array<Array<DataValueType>>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A URL to a remote JSON dataset,
     * structured as a column array. Will be fetched when the chart is created
     * using Ajax.
     */
    columnsURL?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The callback that is evaluated
     * when the data is finished loading, optionally from an external source,
     * and parsed. The first argument passed is a finished chart options object,
     * containing the series. These options can be extended with additional
     * options and passed directly to the chart constructor.
     */
    complete?: DataCompleteCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A comma delimited string to be
     * parsed. Related options are startRow, endRow, startColumn and endColumn
     * to delimit what part of the table is used. The lineDelimiter and
     * itemDelimiter options define the CSV delimiter formats.
     *
     * The built-in CSV parser doesn't support all flavours of CSV, so in some
     * cases it may be necessary to use an external CSV parser. See this example
     * of parsing CSV through the MIT licensed Papa Parse library.
     */
    csv?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An URL to a remote CSV dataset.
     * Will be fetched when the chart is created using Ajax.
     */
    csvURL?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Sets the refresh rate for data
     * polling when importing remote dataset by setting data.csvURL,
     * data.rowsURL, data.columnsURL, or data.googleSpreadsheetKey.
     *
     * Note that polling must be enabled by setting data.enablePolling to true.
     *
     * The value is the number of seconds between pollings. It cannot be set to
     * less than 1 second.
     */
    dataRefreshRate?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Which of the predefined date
     * formats in Date.prototype.dateFormats to use to parse date values.
     * Defaults to a best guess based on what format gives valid and ordered
     * dates. Valid options include: `YYYY/mm/dd`, `dd/mm/YYYY`, `mm/dd/YYYY`,
     * `dd/mm/YY`, `mm/dd/YY`.
     */
    dateFormat?: OptionsDateFormatValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The decimal point used for
     * parsing numbers in the CSV.
     *
     * If both this and data.delimiter is set to `undefined`, the parser will
     * attempt to deduce the decimal point automatically.
     */
    decimalPoint?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enables automatic refetching of
     * remote datasets every _n_ seconds (defined by setting
     * data.dataRefreshRate).
     *
     * Only works when either data.csvURL, data.rowsURL, data.columnsURL, or
     * data.googleSpreadsheetKey.
     */
    enablePolling?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In tabular input data, the last
     * column (indexed by 0) to use. Defaults to the last column containing
     * data.
     */
    endColumn?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In tabular input data, the last
     * row (indexed by 0) to use. Defaults to the last row containing data.
     */
    endRow?: number;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the first row in the data
     * set as series names.
     */
    firstRowAsNames?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The key for a Google Spreadsheet
     * to load. See general information on GS.
     */
    googleSpreadsheetKey?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Google Spreadsheet worksheet
     * to use in combination with googleSpreadsheetKey. The available id's from
     * your sheet can be read from
     * `https://spreadsheets.google.com/feeds/worksheets/{key}/public/basic`.
     */
    googleSpreadsheetWorksheet?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Item or cell delimiter for
     * parsing CSV. Defaults to the tab character `\t` if a tab character is
     * found in the CSV string, if not it defaults to `,`.
     *
     * If this is set to false or undefined, the parser will attempt to deduce
     * the delimiter automatically.
     */
    itemDelimiter?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Line delimiter for parsing CSV.
     */
    lineDelimiter?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A callback function to access
     * the parsed columns, the two-dimentional input data array directly, before
     * they are interpreted into series data and categories. Return `false` to
     * stop completion, or call `this.complete()` to continue async.
     */
    parsed?: DataParsedCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A callback function to parse
     * string representations of dates into JavaScript timestamps. Should return
     * an integer timestamp on success.
     */
    parseDate?: DataParseDateCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The same as the columns input
     * option, but defining rows intead of columns.
     */
    rows?: Array<Array<DataValueType>>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A URL to a remote JSON dataset,
     * structured as a row array. Will be fetched when the chart is created
     * using Ajax.
     */
    rowsURL?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array containing dictionaries
     * for each series. A dictionary exists of Point property names as the key
     * and the CSV column index as the value.
     */
    seriesMapping?: Array<Dictionary<number>>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In tabular input data, the first
     * column (indexed by 0) to use.
     */
    startColumn?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In tabular input data, the first
     * row (indexed by 0) to use.
     */
    startRow?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Switch rows and columns of the
     * input data, so that `this.columns` effectively becomes the rows of the
     * data set, and the rows are interpreted as series.
     */
    switchRowsAndColumns?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An HTML table or the id of such
     * to be parsed as input data. Related options are `startRow`, `endRow`,
     * `startColumn` and `endColumn` to delimit what part of the table is used.
     */
    table?: (string|HTMLElement);
}
/**
 * Options for `dataSorting`.
 */
export interface DataSortingOptionsObject {
    /**
     * Enable or disable data sorting for the series.
     */
    enabled?: boolean;
    /**
     * Whether to allow matching points by name in an update.
     */
    matchByName?: boolean;
    /**
     * Determines what data value should be used to sort by.
     */
    sortKey?: string;
}
export interface DefsArrowOptions {
    children?: Array<DefsOptions>;
    id?: string;
    markerHeight?: number;
    markerWidth?: number;
    refX?: number;
    refY?: number;
    render?: boolean;
    tagName?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for configuring markers for
 * annotations.
 *
 * An example of the arrow marker: (see online documentation for example)
 */
export interface DefsOptions {
    arrow?: (DefsArrowOptions|SVGDefinitionObject);
    "reverse-arrow"?: (DefsReverseArrowOptions|SVGDefinitionObject);
}
export interface DefsReverseArrowOptions {
    id?: string;
    markerHeight?: number;
    markerWidth?: number;
    refX?: number;
    refY?: number;
    render?: boolean;
    tagName?: string;
}
/**
 * Generic dictionary in TypeScript notation. Use the native `Record<string,
 * any>` instead.
 */
export interface Dictionary<T> {
    [key: string]: T;
}
/**
 * (Highcharts, Highstock, Gantt) Style options for the guide box default state.
 */
export interface DragDropGuideBoxOptionsObject {
    /**
     * (Highcharts, Highstock, Gantt) CSS class name of the guide box in this
     * state. Defaults to `highcharts-drag-box-default`.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) Guide box fill color.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Gantt) Guide box cursor.
     */
    cursor?: string;
    /**
     * (Highcharts, Highstock, Gantt) Color of the border around the guide box.
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) Width of the line around the guide box.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) Guide box zIndex.
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highstock, Gantt) Options for the drag handles.
 */
export interface DragDropHandleOptionsObject {
    /**
     * (Highcharts, Highstock, Gantt) The class name of the drag handles.
     * Defaults to `highcharts-drag-handle`.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) The fill color of the drag handles.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Gantt) The mouse cursor to use for the drag
     * handles. By default this is intelligently switching between `ew-resize`
     * and `ns-resize` depending on the direction the point is being dragged.
     */
    cursor?: string;
    /**
     * (Highcharts, Highstock, Gantt) The line color of the drag handles.
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) The line width for the drag handles.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) Function to define the SVG path to use for
     * the drag handles. Takes the point as argument. Should return an SVG path
     * in array format. The SVG path is automatically positioned on the point.
     */
    pathFormatter?: Function;
    /**
     * (Highcharts, Highstock, Gantt) The z index for the drag handles.
     */
    zIndex?: number;
}
/**
 * Current drag and drop position.
 */
export interface DragDropPositionObject {
    /**
     * Chart x position
     */
    chartX: number;
    /**
     * Chart y position
     */
    chartY: number;
    /**
     * Drag and drop guide box.
     */
    guideBox?: BBoxObject;
    /**
     * Updated point data.
     */
    points: Dictionary<Dictionary<number>>;
    /**
     * Delta of previous x position.
     */
    prevdX?: number;
    /**
     * Delta of previous y position.
     */
    prevdY?: number;
}
/**
 * (Highcharts, Highmaps) Additional styles to apply to the data label of a
 * point that has drilldown data. By default it is underlined and blue to invite
 * to interaction.
 *
 * In styled mode, active data label styles can be applied with the
 * `.highcharts-drilldown-data-label` class.
 */
export interface DrilldownActiveDataLabelStyleOptions {
    color?: string;
    cursor?: string;
    fontWeight?: string;
    textDecoration?: string;
}
/**
 * (Highcharts, Highmaps) Options for the drill up button that appears when
 * drilling down on a series. The text for the button is defined in
 * lang.drillUpText.
 */
export interface DrilldownDrillUpButtonOptions {
    /**
     * (Highcharts, Highmaps) Positioning options for the button within the
     * `relativeTo` box. Available properties are `x`, `y`, `align` and
     * `verticalAlign`.
     */
    position?: (AlignObject|DrilldownDrillUpButtonPositionOptions);
    /**
     * (Highcharts, Highmaps) What box to align the button to. Can be either
     * `plotBox` or `spacingBox`.
     */
    relativeTo?: ButtonRelativeToValue;
    /**
     * (Highcharts, Highmaps) A collection of attributes for the button. The
     * object takes SVG attributes like `fill`, `stroke`, `stroke-width` or `r`,
     * the border radius. The theme also supports `style`, a collection of CSS
     * properties for the text. Equivalent attributes for the hover state are
     * given in `theme.states.hover`.
     *
     * In styled mode, drill-up button styles can be applied with the
     * `.highcharts-drillup-button` class.
     */
    theme?: object;
}
/**
 * (Highcharts, Highmaps) Positioning options for the button within the
 * `relativeTo` box. Available properties are `x`, `y`, `align` and
 * `verticalAlign`.
 */
export interface DrilldownDrillUpButtonPositionOptions {
    /**
     * (Highcharts, Highmaps) Horizontal alignment.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highmaps) Vertical alignment of the button.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highmaps) The X offset of the button.
     */
    x?: number;
    /**
     * (Highcharts, Highmaps) The Y offset of the button.
     */
    y?: number;
}
/**
 * The event arguments when a drilldown point is clicked.
 */
export interface DrilldownEventObject {
    /**
     * If a category label was clicked, which index.
     */
    category?: number;
    /**
     * The original browser event (usually click) that triggered the drilldown.
     */
    originalEvent?: Event;
    /**
     * The originating point.
     */
    point: Point;
    /**
     * If a category label was clicked, this array holds all points
     * corresponding to the category. Otherwise it is set to false.
     */
    points?: (boolean|Array<Point>);
    /**
     * Prevents the default behaviour of the event.
     */
    preventDefault: Function;
    /**
     * Options for the new series. If the event is utilized for async drilldown,
     * the seriesOptions are not added, but rather loaded async.
     */
    seriesOptions?: SeriesOptionsType;
    /**
     * The event target.
     */
    target: Chart;
    /**
     * The event type.
     */
    type: "drilldown";
}
/**
 * (Highcharts, Highmaps) Options for drill down, the concept of inspecting
 * increasingly high resolution data through clicking on chart items like
 * columns or pie slices.
 *
 * The drilldown feature requires the drilldown.js file to be loaded, found in
 * the modules directory of the download package, or online at
 * code.highcharts.com/modules/drilldown.js.
 */
export interface DrilldownOptions {
    /**
     * (Highcharts, Highmaps) Additional styles to apply to the X axis label for
     * a point that has drilldown data. By default it is underlined and blue to
     * invite to interaction.
     *
     * In styled mode, active label styles can be set with the
     * `.highcharts-drilldown-axis-label` class.
     */
    activeAxisLabelStyle?: CSSObject;
    /**
     * (Highcharts, Highmaps) Additional styles to apply to the data label of a
     * point that has drilldown data. By default it is underlined and blue to
     * invite to interaction.
     *
     * In styled mode, active data label styles can be applied with the
     * `.highcharts-drilldown-data-label` class.
     */
    activeDataLabelStyle?: (CSSObject|DrilldownActiveDataLabelStyleOptions);
    /**
     * (Highcharts) When this option is false, clicking a single point will
     * drill down all points in the same category, equivalent to clicking the X
     * axis label.
     */
    allowPointDrilldown?: boolean;
    /**
     * (Highcharts, Highmaps) Set the animation for all drilldown animations.
     * Animation of a drilldown occurs when drilling between a column point and
     * a column series, or a pie slice and a full pie series. Drilldown can
     * still be used between series and points of different types, but animation
     * will not occur.
     *
     * The animation can either be set as a boolean or a configuration object.
     * If `true`, it will use the 'swing' jQuery easing and a duration of 500
     * ms. If used as a configuration object, the following properties are
     * supported:
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: A string reference to an easing function set on the `Math`
     * object. See the easing demo.
     */
    animation?: (boolean|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highmaps) Options for the drill up button that appears when
     * drilling down on a series. The text for the button is defined in
     * lang.drillUpText.
     */
    drillUpButton?: DrilldownDrillUpButtonOptions;
    /**
     * (Highcharts, Highmaps) An array of series configurations for the drill
     * down. Each series configuration uses the same syntax as the series option
     * set. These drilldown series are hidden by default. The drilldown series
     * is linked to the parent series' point by its `id`.
     */
    series?: Array<SeriesOptionsType>;
}
/**
 * The event arguments when all the series have been drilled up.
 */
export interface DrillupAllEventObject {
    /**
     * Prevents the default behaviour of the event.
     */
    preventDefault: Function;
    /**
     * The event target.
     */
    target: Chart;
    /**
     * The event type.
     */
    type: "drillupall";
}
/**
 * The event arguments when drilling up from a drilldown series.
 */
export interface DrillupEventObject {
    /**
     * Prevents the default behaviour of the event.
     */
    preventDefault: Function;
    /**
     * Options for the new series.
     */
    seriesOptions?: SeriesOptionsType;
    /**
     * The event target.
     */
    target: Chart;
    /**
     * The event type.
     */
    type: "drillup";
}
/**
 * The event options for adding function callback.
 */
export interface EventOptionsObject {
    /**
     * The order the event handler should be called. This opens for having one
     * handler be called before another, independent of in which order they were
     * added.
     */
    order: number;
}
/**
 * Contains information about the export data event.
 */
export interface ExportDataEventObject {
    /**
     * Contains the data rows for the current export task and can be modified.
     */
    dataRows: Array<Array<string>>;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for the
 * exporting menu. Requires the Accessibility module.
 */
export interface ExportingAccessibilityOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable accessibility support for
     * the export menu.
     */
    enabled?: boolean;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A configuration object for the
 * button theme. The object accepts SVG properties like `stroke-width`, `stroke`
 * and `fill`. Tri-state button styles are supported by the `states.hover` and
 * `states.select` objects.
 */
export interface ExportingButtonsContextButtonThemeOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The default fill exists only to
     * capture hover events.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Padding for the button.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Default stroke for the buttons.
     */
    stroke?: ColorString;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the export related
 * buttons, print and export. In addition to the default buttons listed here,
 * custom buttons can be added. See navigation.buttonOptions for general
 * options.
 */
export interface ExportingButtonsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the export button.
     *
     * In styled mode, export button styles can be applied with the
     * `.highcharts-contextbutton` class.
     */
    contextButton?: ExportingButtonsOptionsObject;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the export button.
 *
 * In styled mode, export button styles can be applied with the
 * `.highcharts-contextbutton` class.
 */
export interface ExportingButtonsOptionsObject {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Alignment for the buttons.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel spacing between
     * buttons.
     */
    buttonSpacing?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The class name of the context
     * button.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to enable buttons.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Pixel height of the buttons.
     */
    height?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The class name of the menu
     * appearing from the button.
     */
    menuClassName?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A collection of strings pointing
     * to config options for the menu items. The config options are defined in
     * the `menuItemDefinitions` option.
     *
     * By default, there is the "View in full screen" and "Print" menu items,
     * plus one menu item for each of the available export types.
     */
    menuItems?: Array<string>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A click handler callback to use
     * on the button directly instead of the popup menu.
     */
    onclick?: Function;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The symbol for the button.
     * Points to a definition function in the `Highcharts.Renderer.symbols`
     * collection. The default `exportIcon` function is part of the exporting
     * module. Possible values are "circle", "square", "diamond", "triangle",
     * "triangle-down", "menu", "menuball" or custom shape.
     */
    symbol?: ("menu"|"menuball"|"exportIcon"|string|SymbolKeyValue);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) See
     * navigation.buttonOptions.symbolFill.
     */
    symbolFill?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel size of the symbol on
     * the button.
     */
    symbolSize?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the symbol's stroke
     * or line.
     */
    symbolStroke?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel stroke width of the
     * symbol on the button.
     */
    symbolStrokeWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position of the center of
     * the symbol inside the button.
     */
    symbolX?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position of the center of
     * the symbol inside the button.
     */
    symbolY?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A text string to add to the
     * individual button.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A configuration object for the
     * button theme. The object accepts SVG properties like `stroke-width`,
     * `stroke` and `fill`. Tri-state button styles are supported by the
     * `states.hover` and `states.select` objects.
     */
    theme?: ExportingButtonsContextButtonThemeOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The key to a lang option setting
     * that is used for the button's title tooltip. When the key is
     * `contextButtonTitle`, it refers to lang.contextButtonTitle that defaults
     * to "Chart context menu".
     */
    titleKey?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * buttons. Can be one of `"top"`, `"middle"` or `"bottom"`.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of the button.
     */
    width?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The horizontal position of the
     * button relative to the `align` option.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical offset of the
     * button's position relative to its `verticalAlign`.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for annotations in the
 * export-data table.
 */
export interface ExportingCsvAnnotationsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The way to mark the separator
     * for annotations combined in one export-data table cell.
     */
    itemDelimiter?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When several labels are assigned
     * to a specific point, they will be displayed in one field in the table.
     */
    join?: boolean;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for exporting data to CSV or
 * ExCel, or displaying the data in a HTML table or a JavaScript structure.
 *
 * This module adds data export options to the export menu and provides
 * functions like `Chart.getCSV`, `Chart.getTable`, `Chart.getDataRows` and
 * `Chart.viewData`.
 *
 * The XLS converter is limited and only creates a HTML string that is passed
 * for download, which works but creates a warning before opening. The
 * workaround for this is to use a third party XLSX converter, as demonstrated
 * in the sample below.
 */
export interface ExportingCsvOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for annotations in the
     * export-data table.
     */
    annotations?: ExportingCsvAnnotationsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Formatter callback for the
     * column headers. Parameters are:
     *
     * - `item` - The series or axis object)
     *
     * - `key` - The point key, for example y or z
     *
     * - `keyLength` - The amount of value keys for this item, for example a
     * range series has the keys `low` and `high` so the key length is 2.
     *
     * If useMultiLevelHeaders is true, columnHeaderFormatter by default returns
     * an object with columnTitle and topLevelColumnTitle for each key. Columns
     * with the same topLevelColumnTitle have their titles merged into a single
     * cell with colspan for table/Excel export.
     *
     * If `useMultiLevelHeaders` is false, or for CSV export, it returns the
     * series name, followed by the key if there is more than one key.
     *
     * For the axis it returns the axis title or "Category" or "DateTime" by
     * default.
     *
     * Return `false` to use Highcharts' proposed header.
     */
    columnHeaderFormatter?: (Function|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Which date format to use for
     * exported dates on a datetime X axis. See `Highcharts.dateFormat`.
     */
    dateFormat?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Which decimal point to use for
     * exported CSV. Defaults to the same as the browser locale, typically `.`
     * (English) or `,` (German, French etc).
     */
    decimalPoint?: (string|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The item delimiter in the
     * exported data. Use `;` for direct exporting to Excel. Defaults to a best
     * guess based on the browser locale. If the locale _decimal point_ is `,`,
     * the `itemDelimiter` defaults to `;`, otherwise the `itemDelimiter`
     * defaults to `,`.
     */
    itemDelimiter?: (string|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The line delimiter in the
     * exported data, defaults to a newline.
     */
    lineDelimiter?: string;
}
/**
 * Definition for a menu item in the context menu.
 */
export interface ExportingMenuObject {
    /**
     * The click handler for the menu item.
     */
    onclick?: EventCallbackFunction<Chart>;
    /**
     * Indicates a separator line instead of an item.
     */
    separator?: boolean;
    /**
     * The text for the menu item.
     */
    text?: string;
    /**
     * If internationalization is required, the key to a language string.
     */
    textKey?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the exporting module.
 * For an overview on the matter, see the docs.
 */
export interface ExportingOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for the
     * exporting menu. Requires the Accessibility module.
     */
    accessibility?: ExportingAccessibilityOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Experimental setting to allow
     * HTML inside the chart (added through the `useHTML` options), directly in
     * the exported image. This allows you to preserve complicated HTML
     * structures like tables or bi-directional text in exported charts.
     *
     * Disclaimer: The HTML is rendered in a `foreignObject` tag in the
     * generated SVG. The official export server is based on PhantomJS, which
     * supports this, but other SVG clients, like Batik, does not support it.
     * This also applies to downloaded SVG that you want to open in a desktop
     * client.
     */
    allowHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the export related
     * buttons, print and export. In addition to the default buttons listed
     * here, custom buttons can be added. See navigation.buttonOptions for
     * general options.
     */
    buttons?: (ExportingButtonsOptions|Dictionary<ExportingButtonsOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Additional chart options to be
     * merged into the chart before exporting to an image format. This does not
     * apply to printing the chart via the export menu.
     *
     * For example, a common use case is to add data labels to improve
     * readability of the exported chart, or to add a printer-friendly color
     * scheme to exported PDFs.
     */
    chartOptions?: Options;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for exporting data to
     * CSV or ExCel, or displaying the data in a HTML table or a JavaScript
     * structure.
     *
     * This module adds data export options to the export menu and provides
     * functions like `Chart.getCSV`, `Chart.getTable`, `Chart.getDataRows` and
     * `Chart.viewData`.
     *
     * The XLS converter is limited and only creates a HTML string that is
     * passed for download, which works but creates a warning before opening.
     * The workaround for this is to use a third party XLSX converter, as
     * demonstrated in the sample below.
     */
    csv?: ExportingCsvOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to enable the exporting
     * module. Disabling the module will hide the context button, but API
     * methods will still be available.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Function to call if the
     * offline-exporting module fails to export a chart on the client side, and
     * fallbackToExportServer is disabled. If left undefined, an exception is
     * thrown instead. Receives two parameters, the exporting options, and the
     * error from the module.
     */
    error?: ExportingErrorCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether or not to fall back to
     * the export server if the offline-exporting module is unable to export the
     * chart on the client side. This happens for certain browsers, and certain
     * features (e.g. allowHTML), depending on the image type exporting to. For
     * very complex charts, it is possible that export can fail in browsers that
     * don't support Blob objects, due to data URL length limits. It is
     * recommended to define the exporting.error handler if disabling fallback,
     * in order to notify users in case export fails.
     */
    fallbackToExportServer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The filename, without extension,
     * to use for the exported chart.
     */
    filename?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An object containing additional
     * key value data for the POST form that sends the SVG to the export server.
     * For example, a `target` can be set to make sure the generated image is
     * received in another frame, or a custom `enctype` or `encoding` can be
     * set.
     */
    formAttributes?: HTMLAttributes;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Path where Highcharts will look
     * for export module dependencies to load on demand if they don't already
     * exist on `window`. Should currently point to location of CanVG library,
     * RGBColor.js, jsPDF and svg2pdf.js, required for client side export in
     * certain browsers.
     */
    libURL?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An object consisting of
     * definitions for the menu items in the context menu. Each key value pair
     * has a `key` that is referenced in the menuItems setting, and a `value`,
     * which is an object with the following properties:
     *
     * - **onclick:** The click handler for the menu item
     *
     * - **text:** The text for the menu item
     *
     * - **textKey:** If internationalization is required, the key to a language
     * string
     *
     * Custom text for the "exitFullScreen" can be set only in lang options (it
     * is not a separate button).
     */
    menuItemDefinitions?: Dictionary<ExportingMenuObject>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When printing the chart from the
     * menu item in the burger menu, if the on-screen chart exceeds this width,
     * it is resized. After printing or cancelled, it is restored. The default
     * width makes the chart fit into typical paper format. Note that this does
     * not affect the chart when printing the web page as a whole.
     */
    printMaxWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Defines the scale or zoom factor
     * for the exported image compared to the on-screen display. While for
     * instance a 600px wide chart may look good on a website, it will look bad
     * in print. The default scale of 2 makes this chart export to a 1200px PNG
     * or JPG.
     */
    scale?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Show a HTML table below the
     * chart with the chart's current data.
     */
    showTable?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Analogous to sourceWidth.
     */
    sourceHeight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the original chart
     * when exported, unless an explicit chart.width is set, or a pixel width is
     * set on the container. The width exported raster image is then multiplied
     * by scale.
     */
    sourceWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Caption for the data table. Same
     * as chart title by default. Set to `false` to disable.
     */
    tableCaption?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Default MIME type for exporting
     * if `chart.exportChart()` is called without specifying a `type` option.
     * Possible values are `image/png`, `image/jpeg`, `application/pdf` and
     * `image/svg+xml`.
     */
    type?: ExportingMimeTypeValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The URL for the server module
     * converting the SVG string to an image format. By default this points to
     * Highchart's free web service.
     */
    url?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Use multi level headers in data
     * table. If csv.columnHeaderFormatter is defined, it has to return objects
     * in order for multi level headers to work.
     */
    useMultiLevelHeaders?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) If using multi level table
     * headers, use rowspans for headers that have only one level.
     */
    useRowspanHeaders?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of charts
     * exported to PNG or JPG. As of Highcharts 3.0, the default pixel width is
     * a function of the chart.width or exporting.sourceWidth and the
     * exporting.scale.
     */
    width?: number;
}
/**
 * The returned object literal from the Highcharts.Axis#getExtremes function.
 */
export interface ExtremesObject {
    /**
     * The maximum value of the axis' associated series.
     */
    dataMax: number;
    /**
     * The minimum value of the axis' associated series.
     */
    dataMin: number;
    /**
     * The maximum axis value, either automatic or set manually. If the `max`
     * option is not set, `maxPadding` is 0 and `endOnTick` is false, this value
     * will be the same as `dataMax`.
     */
    max: number;
    /**
     * The minimum axis value, either automatic or set manually. If the `min`
     * option is not set, `minPadding` is 0 and `startOnTick` is false, this
     * value will be the same as `dataMin`.
     */
    min: number;
    /**
     * The user defined maximum, either from the `max` option or from a zoom or
     * `setExtremes` action.
     */
    userMax: number;
    /**
     * The user defined minimum, either from the `min` option or from a zoom or
     * `setExtremes` action.
     */
    userMin: number;
}
/**
 * The font metrics.
 */
export interface FontMetricsObject {
    /**
     * The baseline relative to the top of the box.
     */
    b: number;
    /**
     * The font size.
     */
    f: number;
    /**
     * The line height.
     */
    h: number;
}
/**
 * (Gantt) Data for a Gantt series.
 */
export interface GanttPointOptionsObject {
    /**
     * (Gantt) Accessibility options for a data point.
     */
    accessibility?: PointAccessibilityOptionsObject;
    /**
     * (Gantt) Whether the grid node belonging to this point should start as
     * collapsed. Used in axes of type treegrid.
     */
    collapsed?: boolean;
    /**
     * (Gantt) Progress indicator, how much of the task completed. If it is a
     * number, the `fill` will be applied automatically.
     */
    completed?: (number|XrangePointPartialFillOptionsObject);
    /**
     * (Gantt) A reserved subspace to store options and values for customized
     * functionality. Here you can add additional data for your own event
     * callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Gantt) The ID of the point (task) that this point depends on in Gantt
     * charts. Aliases connect. Can also be an object, specifying further
     * connecting options between the points. Multiple connections can be
     * specified by providing an array.
     */
    dependency?: (string|XrangePointConnectorsOptionsObject|Array<(string|XrangePointConnectorsOptionsObject)>);
    /**
     * (Gantt) A description of the point to add to the screen reader
     * information about the point.
     */
    description?: string;
    /**
     * (Gantt) Point specific options for the draggable-points module. Overrides
     * options on `series.dragDrop`.
     */
    dragDrop?: SeriesLineDataDragDropOptions;
    /**
     * (Highcharts) The `id` of a series in the drilldown.series array to use
     * for a drilldown for this point.
     */
    drilldown?: string;
    /**
     * (Gantt) The end time of a task.
     */
    end?: number;
    /**
     * (Highcharts, Highstock, Gantt) An id for the point. This can be used
     * after render time to get a pointer to the point object through
     * `chart.get()`.
     */
    id?: string;
    /**
     * (Gantt) The rank for this point's data label in case of collision. If two
     * data labels are about to overlap, only the one with the highest
     * `labelrank` will be drawn.
     */
    labelrank?: number;
    /**
     * (Highcharts, Highstock) Options for the point markers of line-like
     * series.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Gantt) Whether this point is a milestone. If so, only the `start` option
     * is handled, while `end` is ignored.
     */
    milestone?: boolean;
    /**
     * (Gantt) The name of a task. If a `treegrid` y-axis is used (default in
     * Gantt charts), this will be picked up automatically, and used to
     * calculate the y-value.
     */
    name?: string;
    /**
     * (Gantt) The ID of the parent point (task) of this point in Gantt charts.
     */
    parent?: string;
    /**
     * (Gantt) The start time of a task.
     */
    start?: number;
    /**
     * (Gantt) The Y value of a task.
     */
    y?: number;
}
/**
 * Represents the loose structure of a geographic JSON file.
 */
export interface GeoJSON {
    /**
     * Full copyright note of the geographic data.
     */
    copyright?: string;
    /**
     * Short copyright note of the geographic data suitable for watermarks.
     */
    copyrightShort?: string;
    /**
     * Additional meta information based on the coordinate reference system.
     */
    crs?: Dictionary<any>;
    /**
     * Data sets of geographic features.
     */
    features: Array<GeoJSONFeature>;
    /**
     * Map projections and transformations to be used when calculating between
     * lat/lon and chart values. Required for lat/lon support on maps. Allows
     * resizing, rotating, and moving portions of a map within its projected
     * coordinate system while still retaining lat/lon support. If using lat/lon
     * on a portion of the map that does not match a `hitZone`, the definition
     * with the key `default` is used.
     */
    "hc-transform"?: Dictionary<GeoJSONTranslation>;
    /**
     * Title of the geographic data.
     */
    title?: string;
    /**
     * Type of the geographic data. Type of an optimized map collection is
     * `FeatureCollection`.
     */
    type?: string;
    /**
     * Version of the geographic data.
     */
    version?: string;
}
/**
 * Data set of a geographic feature.
 */
export interface GeoJSONFeature extends Dictionary<any> {
    /**
     * Data type of the geographic feature.
     */
    type: string;
}
/**
 * Describes the map projection and transformations applied to a portion of a
 * map.
 */
export interface GeoJSONTranslation {
    /**
     * The coordinate reference system used to generate this portion of the map.
     */
    crs: string;
    /**
     * Define the portion of the map that this defintion applies to. Defined as
     * a GeoJSON polygon feature object, with `type` and `coordinates`
     * properties.
     */
    hitZone?: Dictionary<any>;
    /**
     * Property for internal use for maps generated by Highsoft.
     */
    jsonmarginX?: number;
    /**
     * Property for internal use for maps generated by Highsoft.
     */
    jsonmarginY?: number;
    /**
     * Property for internal use for maps generated by Highsoft.
     */
    jsonres?: number;
    /**
     * Specifies clockwise rotation of the coordinates after the projection, but
     * before scaling and panning. Defined in radians, relative to the
     * coordinate system origin.
     */
    rotation?: number;
    /**
     * The scaling factor applied to the projected coordinates.
     */
    scale?: number;
    /**
     * Property for internal use for maps generated by Highsoft.
     */
    xoffset?: number;
    /**
     * X offset of projected coordinates after scaling.
     */
    xpan?: number;
    /**
     * Property for internal use for maps generated by Highsoft.
     */
    yoffset?: number;
    /**
     * Y offset of projected coordinates after scaling.
     */
    ypan?: number;
}
export interface GlobalOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Path to the pattern image
     * required by VML browsers in order to draw radial gradients.
     */
    VMLRadialGradientURL?: string;
}
/**
 * Gradient options instead of a solid color.
 */
export interface GradientColorObject {
    /**
     * Holds an object that defines the start position and the end position
     * relative to the shape.
     */
    linearGradient?: LinearGradientColorObject;
    /**
     * Holds an object that defines the center position and the radius.
     */
    radialGradient?: RadialGradientColorObject;
    /**
     * The first item in each tuple is the position in the gradient, where 0 is
     * the start of the gradient and 1 is the end of the gradient. Multiple
     * stops can be applied. The second item is the color for each stop. This
     * color can also be given in the rgba format.
     */
    stops: Array<GradientColorStopObject>;
}
/**
 * Color stop tuple.
 */
export interface GradientColorStopObject {
    0: number;
    1: ColorString;
    color?: Color;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the focus border drawn
 * around elements while navigating through them.
 */
export interface KeyboardNavigationFocusBorderOptionsObject {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable/disable focus border for
     * chart.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Hide the browser's default focus
     * indicator.
     */
    hideBrowserFocusOutline?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Focus border margin around the
     * elements.
     */
    margin?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Style options for the focus
     * border drawn around elements while navigating through them. Note that
     * some browsers in addition draw their own borders for focused elements.
     * These automatic borders can not be styled by Highcharts.
     *
     * In styled mode, the border is given the `.highcharts-focus-border` class.
     */
    style?: CSSObject;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for keyboard navigation.
 */
export interface KeyboardNavigationOptionsObject {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable keyboard navigation for
     * the chart.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the focus border
     * drawn around elements while navigating through them.
     */
    focusBorder?: KeyboardNavigationFocusBorderOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Order of tab navigation in the
     * chart. Determines which elements are tabbed to first. Available elements
     * are: `series`, `zoom`, `rangeSelector`, `chartMenu`, `legend`. In
     * addition, any custom components can be added here.
     */
    order?: Array<string>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the keyboard
     * navigation of data points and series.
     */
    seriesNavigation?: KeyboardNavigationSeriesNavigationOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether or not to wrap around
     * when reaching the end of arrow-key navigation for an element in the
     * chart.
     */
    wrapAround?: boolean;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the keyboard navigation
 * of data points and series.
 */
export interface KeyboardNavigationSeriesNavigationOptionsObject {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Set the keyboard navigation mode
     * for the chart. Can be "normal" or "serialize". In normal mode, left/right
     * arrow keys move between points in a series, while up/down arrow keys move
     * between series. Up/down navigation acts intelligently to figure out which
     * series makes sense to move to from any given point.
     *
     * In "serialize" mode, points are instead navigated as a single list.
     * Left/right behaves as in "normal" mode. Up/down arrow keys will behave
     * like left/right. This can be useful for unifying navigation behavior
     * with/without screen readers enabled.
     */
    mode?: OptionsModeValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When a series contains more
     * points than this, we no longer allow keyboard navigation for it.
     *
     * Set to `false` to disable.
     */
    pointNavigationEnabledThreshold?: (boolean|number);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Skip null points when navigating
     * through points with the keyboard.
     */
    skipNullPoints?: boolean;
}
/**
 * Containing the position of a box that should be avoided by labels.
 */
export interface LabelIntersectBoxObject {
    bottom: number;
    left: number;
    right: number;
    top: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Default announcement for new data in
 * charts. If addPoint or addSeries is used, and only one series/point is added,
 * the `newPointAnnounce` and `newSeriesAnnounce` strings are used. The
 * `...Single` versions will be used if there is only one chart on the page, and
 * the `...Multiple` versions will be used if there are multiple charts on the
 * page. For all other new data events, the `newDataAnnounce` string will be
 * used.
 */
export interface LangAccessibilityAnnounceNewDataOptions {
    newDataAnnounce?: string;
    newPointAnnounceMultiple?: string;
    newPointAnnounceSingle?: string;
    newSeriesAnnounceMultiple?: string;
    newSeriesAnnounceSingle?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Axis description format strings.
 */
export interface LangAccessibilityAxisOptions {
    rangeCategories?: string;
    rangeFromTo?: string;
    timeRangeDays?: string;
    timeRangeHours?: string;
    timeRangeMinutes?: string;
    timeRangeSeconds?: string;
    xAxisDescriptionPlural?: string;
    xAxisDescriptionSingular?: string;
    yAxisDescriptionPlural?: string;
    yAxisDescriptionSingular?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Chart type description strings. This
 * is added to the chart information region.
 *
 * If there is only a single series type used in the chart, we use the format
 * string for the series type, or default if missing. There is one format string
 * for cases where there is only a single series in the chart, and one for
 * multiple series of the same type.
 */
export interface LangAccessibilityChartTypesOptions {
    barMultiple?: string;
    barSingle?: string;
    boxplotMultiple?: string;
    boxplotSingle?: string;
    bubbleMultiple?: string;
    bubbleSingle?: string;
    columnMultiple?: string;
    columnSingle?: string;
    combinationChart?: string;
    defaultMultiple?: string;
    defaultSingle?: string;
    emptyChart?: string;
    lineMultiple?: string;
    lineSingle?: string;
    mapTypeDescription?: string;
    pieMultiple?: string;
    pieSingle?: string;
    scatterMultiple?: string;
    scatterSingle?: string;
    splineMultiple?: string;
    splineSingle?: string;
    unknownMap?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Exporting menu format strings for
 * accessibility module.
 */
export interface LangAccessibilityExportingOptions {
    chartMenuLabel?: string;
    exportRegionLabel?: string;
    menuButtonLabel?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Language options for accessibility
 * of the legend.
 */
export interface LangAccessibilityLegendOptions {
    legendItem?: string;
    legendLabel?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Configure the accessibility strings
 * in the chart. Requires the accessibility module to be loaded. For a
 * description of the module and information on its features, see Highcharts
 * Accessibility.
 *
 * For more dynamic control over the accessibility functionality, see
 * accessibility.pointDescriptionFormatter,
 * accessibility.seriesDescriptionFormatter, and
 * accessibility.screenReaderSectionFormatter.
 */
export interface LangAccessibilityOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Default announcement for new
     * data in charts. If addPoint or addSeries is used, and only one
     * series/point is added, the `newPointAnnounce` and `newSeriesAnnounce`
     * strings are used. The `...Single` versions will be used if there is only
     * one chart on the page, and the `...Multiple` versions will be used if
     * there are multiple charts on the page. For all other new data events, the
     * `newDataAnnounce` string will be used.
     */
    announceNewData?: LangAccessibilityAnnounceNewDataOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Axis description format strings.
     */
    axis?: LangAccessibilityAxisOptions;
    chartContainerLabel?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Chart type description strings.
     * This is added to the chart information region.
     *
     * If there is only a single series type used in the chart, we use the
     * format string for the series type, or default if missing. There is one
     * format string for cases where there is only a single series in the chart,
     * and one for multiple series of the same type.
     */
    chartTypes?: LangAccessibilityChartTypesOptions;
    credits?: string;
    defaultChartTitle?: string;
    drillUpButton?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting menu format strings
     * for accessibility module.
     */
    exporting?: LangAccessibilityExportingOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Set a label on the container
     * wrapping the SVG.
     */
    graphicContainerLabel?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Language options for
     * accessibility of the legend.
     */
    legend?: LangAccessibilityLegendOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Range selector language options
     * for accessibility.
     */
    rangeSelector?: LangAccessibilityRangeSelectorOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Language options for the screen
     * reader information sections added before and after the charts.
     */
    screenReaderSection?: LangAccessibilityScreenReaderSectionOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Lang configuration for different
     * series types. For more dynamic control over the series element
     * descriptions, see accessibility.seriesDescriptionFormatter.
     */
    series?: LangAccessibilitySeriesOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Descriptions of lesser known
     * series types. The relevant description is added to the screen reader
     * information region when these series types are used.
     */
    seriesTypeDescriptions?: LangAccessibilitySeriesTypeDescriptionsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Language options for
     * sonification.
     */
    sonification?: LangAccessibilitySonificationOptions;
    svgContainerLabel?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Title element text for the chart
     * SVG element. Leave this empty to disable adding the title element.
     * Browsers will display this content when hovering over elements in the
     * chart. Assistive technology may use this element to label the chart.
     */
    svgContainerTitle?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Accessibility language options
     * for the data table.
     */
    table?: LangAccessibilityTableOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Thousands separator to use when
     * formatting numbers for screen readers. Note that many screen readers will
     * not handle space as a thousands separator, and will consider "11 700" as
     * two numbers.
     *
     * Set to `null` to use the separator defined in lang.thousandsSep.
     */
    thousandsSep?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Chart and map zoom accessibility
     * language options.
     */
    zoom?: LangAccessibilityZoomOptions;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Range selector language options for
 * accessibility.
 */
export interface LangAccessibilityRangeSelectorOptions {
    buttonText?: string;
    maxInputLabel?: string;
    minInputLabel?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Language options for annotation
 * descriptions.
 */
export interface LangAccessibilityScreenReaderSectionAnnotationsOptions {
    descriptionMultiplePoints?: string;
    descriptionNoPoints?: string;
    descriptionSinglePoint?: string;
    heading?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Language options for the screen
 * reader information sections added before and after the charts.
 */
export interface LangAccessibilityScreenReaderSectionOptions {
    afterRegionLabel?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Language options for annotation
     * descriptions.
     */
    annotations?: LangAccessibilityScreenReaderSectionAnnotationsOptions;
    beforeRegionLabel?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Label for the end of the chart.
     * Announced by screen readers.
     */
    endOfChartMarker?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Lang configuration for different
 * series types. For more dynamic control over the series element descriptions,
 * see accessibility.seriesDescriptionFormatter.
 */
export interface LangAccessibilitySeriesOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) User supplied description text.
     * This is added in the point comment description by default if present.
     */
    description?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Description for the value of
     * null points.
     */
    nullPointValue?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Description for annotations on a
     * point, as it is made available to assistive technology.
     */
    pointAnnotationsDescription?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Lang configuration for the
     * series main summary. Each series type has two modes:
     *
     * 1. This series type is the only series type used in the chart
     *
     * 2. This is a combination chart with multiple series types
     *
     * If a definition does not exist for the specific series type and mode, the
     * 'default' lang definitions are used.
     */
    summary?: LangAccessibilitySeriesSummaryOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) xAxis description for series if
     * there are multiple xAxes in the chart.
     */
    xAxisDescription?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) yAxis description for series if
     * there are multiple yAxes in the chart.
     */
    yAxisDescription?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Lang configuration for the series
 * main summary. Each series type has two modes:
 *
 * 1. This series type is the only series type used in the chart
 *
 * 2. This is a combination chart with multiple series types
 *
 * If a definition does not exist for the specific series type and mode, the
 * 'default' lang definitions are used.
 */
export interface LangAccessibilitySeriesSummaryOptions {
    bar?: string;
    barCombination?: string;
    boxplot?: string;
    boxplotCombination?: string;
    bubble?: string;
    bubbleCombination?: string;
    column?: string;
    columnCombination?: string;
    default?: string;
    defaultCombination?: string;
    line?: string;
    lineCombination?: string;
    map?: string;
    mapbubble?: string;
    mapbubbleCombination?: string;
    mapCombination?: string;
    mapline?: string;
    maplineCombination?: string;
    pie?: string;
    pieCombination?: string;
    scatter?: string;
    scatterCombination?: string;
    spline?: string;
    splineCombination?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Descriptions of lesser known series
 * types. The relevant description is added to the screen reader information
 * region when these series types are used.
 */
export interface LangAccessibilitySeriesTypeDescriptionsOptions {
    arearange?: string;
    areasplinerange?: string;
    boxplot?: string;
    bubble?: string;
    columnrange?: string;
    errorbar?: string;
    funnel?: string;
    pyramid?: string;
    waterfall?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Language options for sonification.
 */
export interface LangAccessibilitySonificationOptions {
    playAsSoundButtonText?: string;
    playAsSoundClickAnnouncement?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Accessibility language options for
 * the data table.
 */
export interface LangAccessibilityTableOptions {
    tableSummary?: string;
    viewAsDataTableButtonText?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Chart and map zoom accessibility
 * language options.
 */
export interface LangAccessibilityZoomOptions {
    mapZoomIn?: string;
    mapZoomOut?: string;
    resetZoomButton?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The text for exported table.
 */
export interface LangExportDataOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The annotation column title.
     */
    annotationHeader?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The category column title when
     * axis type set to "datetime".
     */
    categoryDatetimeHeader?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The category column title.
     */
    categoryHeader?: string;
}
/**
 * (Highcharts, Highstock) Configure the Popup strings in the chart. Requires
 * the `annotations.js` or `annotations-advanced.src.js` module to be loaded.
 */
export interface LangNavigationOptions {
    /**
     * (Highcharts, Highstock) Translations for all field names used in popup.
     */
    popup?: LangNavigationPopupOptions;
}
/**
 * (Highcharts, Highstock) Translations for all field names used in popup.
 */
export interface LangNavigationPopupOptions {
    addButton?: string;
    arrowLine?: string;
    arrowRay?: string;
    arrowSegment?: string;
    background?: string;
    backgroundColor?: string;
    backgroundColors?: string;
    borderColor?: string;
    borderRadius?: string;
    borderWidth?: string;
    circle?: string;
    color?: string;
    connector?: string;
    crooked3?: string;
    crooked5?: string;
    crosshairX?: string;
    crosshairY?: string;
    editButton?: string;
    elliott3?: string;
    elliott5?: string;
    fibonacci?: string;
    fill?: string;
    flags?: string;
    fontSize?: string;
    format?: string;
    height?: string;
    horizontalLine?: string;
    infinityLine?: string;
    innerBackground?: string;
    label?: string;
    labelOptions?: string;
    labels?: string;
    line?: string;
    lines?: string;
    measure?: string;
    measureX?: string;
    measureXY?: string;
    measureY?: string;
    name?: string;
    outerBackground?: string;
    padding?: string;
    parallelChannel?: string;
    pitchfork?: string;
    ray?: string;
    rectangle?: string;
    removeButton?: string;
    saveButton?: string;
    segment?: string;
    series?: string;
    shapeOptions?: string;
    shapes?: string;
    simpleShapes?: string;
    stroke?: string;
    strokeWidth?: string;
    style?: string;
    title?: string;
    tunnel?: string;
    typeOptions?: string;
    verticalArrow?: string;
    verticalCounter?: string;
    verticalLabel?: string;
    verticalLine?: string;
    volume?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Language object. The language object
 * is global and it can't be set on each chart initialization. Instead, use
 * `Highcharts.setOptions` to set it before any chart is initialized. (see
 * online documentation for example)
 */
export interface LangOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Configure the accessibility
     * strings in the chart. Requires the accessibility module to be loaded. For
     * a description of the module and information on its features, see
     * Highcharts Accessibility.
     *
     * For more dynamic control over the accessibility functionality, see
     * accessibility.pointDescriptionFormatter,
     * accessibility.seriesDescriptionFormatter, and
     * accessibility.screenReaderSectionFormatter.
     */
    accessibility?: LangAccessibilityOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module menu. The
     * tooltip title for the context menu holding print and export menu items.
     */
    contextButtonTitle?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The default decimal point used
     * in the `Highcharts.numberFormat` method unless otherwise specified in the
     * function arguments.
     */
    decimalPoint?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text for the menu item.
     */
    downloadCSV?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the JPEG download menu item.
     */
    downloadJPEG?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the PDF download menu item.
     */
    downloadPDF?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the PNG download menu item.
     */
    downloadPNG?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the SVG download menu item.
     */
    downloadSVG?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text for the menu item.
     */
    downloadXLS?: string;
    /**
     * (Highcharts, Highmaps) The text for the button that appears when drilling
     * down, linking back to the parent series. The parent series' name is
     * inserted for `{series.name}`.
     */
    drillUpText?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the menu item to exit the chart from full screen.
     */
    exitFullscreen?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text for exported table.
     */
    exportData?: LangExportDataOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text for the menu item.
     */
    hideData?: string;
    /**
     * (Highcharts, Highstock) What to show in a date field for invalid dates.
     * Defaults to an empty string.
     */
    invalidDate?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The loading text that appears
     * when the chart is set into the loading state following a call to
     * `chart.showLoading`.
     */
    loading?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array containing the months
     * names. Corresponds to the `%B` format in `Highcharts.dateFormat()`.
     */
    months?: Array<string>;
    /**
     * (Highcharts, Highstock) Configure the Popup strings in the chart.
     * Requires the `annotations.js` or `annotations-advanced.src.js` module to
     * be loaded.
     */
    navigation?: LangNavigationOptions;
    /**
     * (Highcharts, Highstock) The text to display when the chart contains no
     * data.
     */
    noData?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The magnitude of numericSymbols
     * replacements. Use 10000 for Japanese, Korean and various Chinese locales,
     * which use symbols for 10^4, 10^8 and 10^12.
     */
    numericSymbolMagnitude?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Metric prefixes used to shorten
     * high numbers in axis labels. Replacing any of the positions with `null`
     * causes the full number to be written. Setting `numericSymbols` to `null`
     * disables shortening altogether.
     */
    numericSymbols?: Array<string>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the menu item to print the chart.
     */
    printChart?: string;
    /**
     * (Highstock, Gantt) The text for the label for the "from" input box in the
     * range selector.
     */
    rangeSelectorFrom?: string;
    /**
     * (Highstock, Gantt) The text for the label for the "to" input box in the
     * range selector.
     */
    rangeSelectorTo?: string;
    /**
     * (Highstock, Gantt) The text for the label for the range selector buttons.
     */
    rangeSelectorZoom?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text for the label appearing
     * when a chart is zoomed.
     */
    resetZoom?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The tooltip title for the label
     * appearing when a chart is zoomed.
     */
    resetZoomTitle?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array containing the months
     * names in abbreviated form. Corresponds to the `%b` format in
     * `Highcharts.dateFormat()`.
     */
    shortMonths?: Array<string>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Short week days, starting
     * Sunday. If not specified, Highcharts uses the first three letters of the
     * `lang.weekdays` option.
     */
    shortWeekdays?: Array<string>;
    /**
     * (Highstock) Configure the stockTools GUI titles(hints) in the chart.
     * Requires the `stock-tools.js` module to be loaded.
     */
    stockTools?: LangStockToolsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The default thousands separator
     * used in the `Highcharts.numberFormat` method unless otherwise specified
     * in the function arguments. Defaults to a single space character, which is
     * recommended in ISO 31-0 and works across Anglo-American and continental
     * European languages.
     */
    thousandsSep?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text for the menu item.
     */
    viewData?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the menu item to view the chart in full screen.
     */
    viewFullscreen?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array containing the weekday
     * names.
     */
    weekdays?: Array<string>;
    /**
     * (Highmaps) The title appearing on hovering the zoom in button. The text
     * itself defaults to "+" and can be changed in the button options.
     */
    zoomIn?: string;
    /**
     * (Highmaps) The title appearing on hovering the zoom out button. The text
     * itself defaults to "-" and can be changed in the button options.
     */
    zoomOut?: string;
}
export interface LangStockToolsGuiOptions {
    advanced?: string;
    arrowLine?: string;
    arrowRay?: string;
    arrowSegment?: string;
    circle?: string;
    crooked3?: string;
    crooked5?: string;
    crookedLines?: string;
    currentPriceIndicator?: string;
    elliott3?: string;
    elliott5?: string;
    fibonacci?: string;
    flagCirclepin?: string;
    flagDiamondpin?: string;
    flags?: string;
    flagSimplepin?: string;
    flagSquarepin?: string;
    fullScreen?: string;
    horizontalLine?: string;
    indicators?: string;
    infinityLine?: string;
    label?: string;
    line?: string;
    lines?: string;
    measure?: string;
    measureX?: string;
    measureXY?: string;
    measureY?: string;
    parallelChannel?: string;
    pitchfork?: string;
    ray?: string;
    rectangle?: string;
    saveChart?: string;
    segment?: string;
    simpleShapes?: string;
    toggleAnnotations?: string;
    typeCandlestick?: string;
    typeChange?: string;
    typeLine?: string;
    typeOHLC?: string;
    verticalArrow?: string;
    verticalCounter?: string;
    verticalLabel?: string;
    verticalLabels?: string;
    verticalLine?: string;
    zoomChange?: string;
    zoomX?: string;
    zoomXY?: string;
    zoomY?: string;
}
/**
 * (Highstock) Configure the stockTools GUI titles(hints) in the chart. Requires
 * the `stock-tools.js` module to be loaded.
 */
export interface LangStockToolsOptions {
    gui?: LangStockToolsGuiOptions;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for keyboard navigation for
 * the legend.
 */
export interface LegendAccessibilityKeyboardNavigationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable keyboard navigation for
     * the legend.
     */
    enabled?: boolean;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for the
 * legend. Requires the Accessibility module.
 */
export interface LegendAccessibilityOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable accessibility support for
     * the legend.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for keyboard navigation
     * for the legend.
     */
    keyboardNavigation?: LegendAccessibilityKeyboardNavigationOptions;
}
/**
 * (Highcharts, Highstock, Highmaps) Options for the bubble legend labels.
 */
export interface LegendBubbleLegendLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps) The alignment of the labels compared to
     * the bubble legend. Can be one of `left`, `center` or `right`.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps) Whether to allow data labels to
     * overlap.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) An additional class name to apply to
     * the bubble legend label graphical elements. This option does not replace
     * default class names of the graphical element.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps) A format string for the bubble legend
     * labels. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps) Available `this` properties are:
     *
     * - `this.value`: The bubble value.
     *
     * - `this.radius`: The radius of the bubble range.
     *
     * - `this.center`: The center y position of the range.
     */
    formatter?: FormatterCallbackFunction<BubbleLegendFormatterContextObject>;
    /**
     * (Highcharts, Highstock, Highmaps) CSS styles for the labels.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps) The x position offset of the label
     * relative to the connector.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The y position offset of the label
     * relative to the connector.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps) The bubble legend is an additional element
 * in legend which presents the scale of the bubble series. Individual bubble
 * ranges can be defined by user or calculated from series. In the case of
 * automatically calculated ranges, a 1px margin of error is permitted.
 */
export interface LegendBubbleLegendOptions {
    /**
     * (Highcharts, Highstock, Highmaps) The color of the ranges borders, can be
     * also defined for an individual range.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) The width of the ranges borders in
     * pixels, can be also defined for an individual range.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps) An additional class name to apply to
     * the bubble legend' circle graphical elements. This option does not
     * replace default class names of the graphical element.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps) The main color of the bubble legend.
     * Applies to ranges, if individual color is not defined.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) An additional class name to apply to
     * the bubble legend's connector graphical elements. This option does not
     * replace default class names of the graphical element.
     */
    connectorClassName?: string;
    /**
     * (Highcharts, Highstock, Highmaps) The color of the connector, can be also
     * defined for an individual range.
     */
    connectorColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) The length of the connectors in pixels.
     * If labels are centered, the distance is reduced to 0.
     */
    connectorDistance?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The width of the connectors in pixels.
     */
    connectorWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Enable or disable the bubble legend.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) Options for the bubble legend labels.
     */
    labels?: LegendBubbleLegendLabelsOptions;
    /**
     * (Highcharts, Highstock, Highmaps) The position of the bubble legend in
     * the legend.
     */
    legendIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Miximum bubble legend range size. If
     * values for ranges are not specified, the `minSize` and the `maxSize` are
     * calculated from bubble series.
     */
    maxSize?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Minimum bubble legend range size. If
     * values for ranges are not specified, the `minSize` and the `maxSize` are
     * calculated from bubble series.
     */
    minSize?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Options for specific range. One range
     * consists of bubble, label and connector.
     */
    ranges?: Array<LegendBubbleLegendRangesOptions>;
    /**
     * (Highcharts, Highstock, Highmaps) Whether the bubble legend range value
     * should be represented by the area or the width of the bubble. The
     * default, area, corresponds best to the human perception of the size of
     * each bubble.
     */
    sizeBy?: BubbleSizeByValue;
    /**
     * (Highcharts, Highstock, Highmaps) When this is true, the absolute value
     * of z determines the size of the bubble. This means that with the default
     * zThreshold of 0, a bubble of value -1 will have the same size as a bubble
     * of value 1, while a bubble of value 0 will have a smaller size according
     * to minSize.
     */
    sizeByAbsoluteValue?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) Define the visual z index of the bubble
     * legend.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Ranges with with lower value than
     * zThreshold, are skipped.
     */
    zThreshold?: number;
}
/**
 * (Highcharts, Highstock, Highmaps) Options for specific range. One range
 * consists of bubble, label and connector.
 */
export interface LegendBubbleLegendRangesOptions {
    /**
     * (Highcharts, Highstock, Highmaps) The color of the border for individual
     * range.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) The color of the bubble for individual
     * range.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) The color of the connector for
     * individual range.
     */
    connectorColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) Range size value, similar to bubble Z
     * data.
     */
    value?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the paging or navigation
 * appearing when the legend is overflown. Navigation works well on screen, but
 * not in static exported images. One way of working around that is to increase
 * the chart height in export.
 */
export interface LegendNavigationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color for the active up or
     * down arrow in the legend page navigation.
     */
    activeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to animate the pages when
     * navigating up or down. A value of `true` applies the default navigation
     * given in the `chart.animation` option. Additional options can be given as
     * an object containing values for easing and duration.
     */
    animation?: (boolean|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel size of the up and
     * down arrows in the legend paging navigation.
     */
    arrowSize?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to enable the legend
     * navigation. In most cases, disabling the navigation results in an
     * unwanted overflow.
     *
     * See also the adapt chart to legend plugin for a solution to extend the
     * chart height to make room for the legend, optionally in exported charts
     * only.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the inactive up or
     * down arrow in the legend page navigation. .
     */
    inactiveColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text styles for the legend page
     * navigation.
     */
    style?: CSSObject;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The legend is a box containing a
 * symbol and name for each series item or point item in the chart. Each series
 * (or points in case of pie charts) is represented by a symbol and its name in
 * the legend.
 *
 * It is possible to override the symbol creator function and create custom
 * legend symbols.
 */
export interface LegendOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for the
     * legend. Requires the Accessibility module.
     */
    accessibility?: LegendAccessibilityOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The horizontal alignment of the
     * legend box within the chart area. Valid values are `left`, `center` and
     * `right`.
     *
     * In the case that the legend is aligned in a corner position, the `layout`
     * option will determine whether to place it above/below or on the side of
     * the plot area.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) If the layout is `horizontal`
     * and the legend items span over two lines or more, whether to align the
     * items into vertical columns. Setting this to `false` makes room for more
     * items, but will look more messy.
     */
    alignColumns?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color of the
     * legend.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the drawn border
     * around the legend.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border corner radius of the
     * legend.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the drawn border
     * around the legend.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The bubble legend is an additional
     * element in legend which presents the scale of the bubble series.
     * Individual bubble ranges can be defined by user or calculated from
     * series. In the case of automatically calculated ranges, a 1px margin of
     * error is permitted.
     */
    bubbleLegend?: LegendBubbleLegendOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the legend.
     * There is also a series-specific option, showInLegend, that can hide the
     * series from the legend. In some series types this is `false` by default,
     * so it must set to `true` in order to show the legend for the series.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When the legend is floating, the
     * plot area ignores it and is allowed to be placed below it.
     */
    floating?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Default styling for the checkbox
     * next to a legend item when `showCheckbox` is true.
     */
    itemCheckboxStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In a legend with horizontal
     * layout, the itemDistance defines the pixel distance between each item.
     */
    itemDistance?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for each legend item
     * when the corresponding series or point is hidden. Only a subset of CSS is
     * supported, notably those options related to text. Properties are
     * inherited from `style` unless overridden here.
     */
    itemHiddenStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for each legend item
     * in hover mode. Only a subset of CSS is supported, notably those options
     * related to text. Properties are inherited from `style` unless overridden
     * here.
     */
    itemHoverStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel bottom margin for each
     * legend item.
     */
    itemMarginBottom?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel top margin for each
     * legend item.
     */
    itemMarginTop?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for each legend item.
     * Only a subset of CSS is supported, notably those options related to text.
     * The default `textOverflow` property makes long texts truncate. Set it to
     * `undefined` to wrap text instead. A `width` property can be added to
     * control the text width.
     */
    itemStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width for each legend item.
     * By default the items are laid out successively. In a horizontal layout,
     * if the items are laid out across two rows or more, they will be
     * vertically aligned depending on the legend.alignColumns option.
     */
    itemWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for each legend
     * label. Available variables relates to properties on the series, or the
     * point in case of pies.
     */
    labelFormat?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback function to format each
     * of the series' labels. The `this` keyword refers to the series object, or
     * the point object in case of pie charts. By default the series or point
     * name is printed.
     */
    labelFormatter?: FormatterCallbackFunction<(Point|Series)>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The layout of the legend items.
     * Can be one of `horizontal` or `vertical` or `proximate`. When
     * `proximate`, the legend items will be placed as close as possible to the
     * graphs they're representing, except in inverted charts or when the legend
     * position doesn't allow it.
     */
    layout?: OptionsLayoutValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) If the plot area sized is
     * calculated automatically and the legend is not floating, the legend
     * margin is the space between the legend and the axis labels or plot area.
     */
    margin?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Maximum pixel height for the
     * legend. When the maximum height is extended, navigation will show.
     */
    maxHeight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the paging or
     * navigation appearing when the legend is overflown. Navigation works well
     * on screen, but not in static exported images. One way of working around
     * that is to increase the chart height in export.
     */
    navigation?: LegendNavigationOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The inner padding of the legend
     * box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to reverse the order of
     * the legend items compared to the order of the series or points as defined
     * in the configuration object.
     */
    reversed?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to show the symbol on
     * the right side of the text rather than the left side. This is common in
     * Arabic and Hebraic.
     */
    rtl?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to apply a drop shadow
     * to the legend. A `backgroundColor` also needs to be applied for this to
     * take effect. The shadow can be an object configuration containing
     * `color`, `offsetX`, `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|CSSObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When this is true, the legend
     * symbol width will be the same as the symbol height, which in turn
     * defaults to the font size of the legend items.
     */
    squareSymbol?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel height of the symbol
     * for series types that use a rectangle in the legend. Defaults to the font
     * size of legend items.
     */
    symbolHeight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel padding between the
     * legend item symbol and the legend item text.
     */
    symbolPadding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius of the symbol
     * for series types that use a rectangle in the legend. Defaults to half the
     * `symbolHeight`.
     */
    symbolRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of the legend
     * item symbol. When the `squareSymbol` option is set, this defaults to the
     * `symbolHeight`, otherwise 16.
     */
    symbolWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A title to be added on top of
     * the legend.
     */
    title?: LegendTitleOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the legend item texts.
     *
     * Prior to 4.1.7, when using HTML, legend.navigation was disabled.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * legend box. Can be one of `top`, `middle` or `bottom`. Vertical position
     * can be further determined by the `y` option.
     *
     * In the case that the legend is aligned in a corner position, the `layout`
     * option will determine whether to place it above/below or on the side of
     * the plot area.
     *
     * When the layout option is `proximate`, the `verticalAlign` option doesn't
     * apply.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the legend box. If
     * a number is set, it translates to pixels. Since v7.0.2 it allows setting
     * a percent string of the full chart width, for example `40%`.
     *
     * Defaults to the full chart width for legends below or above the chart,
     * half the chart width for legends to the left and right.
     */
    width?: (number|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x offset of the legend
     * relative to its horizontal alignment `align` within chart.spacingLeft and
     * chart.spacingRight. Negative x moves it to the left, positive x moves it
     * to the right.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical offset of the
     * legend relative to it's vertical alignment `verticalAlign` within
     * chart.spacingTop and chart.spacingBottom. Negative y moves it up,
     * positive y moves it down.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A title to be added on top of the
 * legend.
 */
export interface LegendTitleOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Generic CSS styles for the
     * legend title.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A text or HTML string for the
     * title.
     */
    text?: string;
}
/**
 * Defines the start position and the end position for a gradient relative to
 * the shape. Start position (x1, y1) and end position (x2, y2) are relative to
 * the shape, where 0 means top/left and 1 is bottom/right.
 */
export interface LinearGradientColorObject {
    /**
     * Start horizontal position of the gradient. Float ranges 0-1.
     */
    x1: number;
    /**
     * End horizontal position of the gradient. Float ranges 0-1.
     */
    x2: number;
    /**
     * Start vertical position of the gradient. Float ranges 0-1.
     */
    y1: number;
    /**
     * End vertical position of the gradient. Float ranges 0-1.
     */
    y2: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The loading options control the
 * appearance of the loading screen that covers the plot area on chart
 * operations. This screen only appears after an explicit call to
 * `chart.showLoading()`. It is a utility for developers to communicate to the
 * end user that something is going on, for example while retrieving new data
 * via an XHR connection. The "Loading..." text itself is not part of this
 * configuration object, but part of the `lang` object.
 */
export interface LoadingOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The duration in milliseconds of
     * the fade out effect.
     */
    hideDuration?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the loading label
     * `span`.
     */
    labelStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The duration in milliseconds of
     * the fade in effect.
     */
    showDuration?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the loading
     * screen that covers the plot area.
     *
     * In styled mode, the loading label is styled with the
     * `.highcharts-loading` class.
     */
    style?: CSSObject;
}
/**
 * (Highmaps) General options for the map navigation buttons. Individual options
 * can be given from the mapNavigation.buttons option set.
 */
export interface MapNavigationButtonOptions {
    /**
     * (Highmaps) The alignment of the navigation buttons.
     */
    align?: AlignValue;
    /**
     * (Highmaps) What box to align the buttons to. Possible values are
     * `plotBox` and `spacingBox`.
     */
    alignTo?: ButtonRelativeToValue;
    /**
     * (Highmaps) The pixel height of the map navigation buttons.
     */
    height?: number;
    /**
     * (Highmaps) Padding for the navigation buttons.
     */
    padding?: number;
    /**
     * (Highmaps) Text styles for the map navigation buttons.
     */
    style?: CSSObject;
    /**
     * (Highmaps) A configuration object for the button theme. The object
     * accepts SVG properties like `stroke-width`, `stroke` and `fill`.
     * Tri-state button styles are supported by the `states.hover` and
     * `states.select` objects.
     */
    theme?: SVGAttributes;
    /**
     * (Highmaps) The vertical alignment of the buttons. Individual alignment
     * can be adjusted by each button's `y` offset.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highmaps) The width of the map navigation buttons.
     */
    width?: number;
    /**
     * (Highmaps) The X offset of the buttons relative to its `align` setting.
     */
    x?: number;
}
/**
 * (Highmaps) The individual buttons for the map navigation. This usually
 * includes the zoom in and zoom out buttons. Properties for each button is
 * inherited from mapNavigation.buttonOptions, while individual options can be
 * overridden. But default, the `onclick`, `text` and `y` options are
 * individual.
 */
export interface MapNavigationButtonsOptions {
    /**
     * (Highmaps) Options for the zoom in button. Properties for the zoom in and
     * zoom out buttons are inherited from mapNavigation.buttonOptions, while
     * individual options can be overridden. By default, the `onclick`, `text`
     * and `y` options are individual.
     */
    zoomIn?: MapNavigationButtonsZoomInOptions;
    /**
     * (Highmaps) Options for the zoom out button. Properties for the zoom in
     * and zoom out buttons are inherited from mapNavigation.buttonOptions,
     * while individual options can be overridden. By default, the `onclick`,
     * `text` and `y` options are individual.
     */
    zoomOut?: MapNavigationButtonsZoomOutOptions;
}
/**
 * (Highmaps) Options for the zoom in button. Properties for the zoom in and
 * zoom out buttons are inherited from mapNavigation.buttonOptions, while
 * individual options can be overridden. By default, the `onclick`, `text` and
 * `y` options are individual.
 */
export interface MapNavigationButtonsZoomInOptions {
    /**
     * (Highmaps) The alignment of the navigation buttons.
     */
    align?: AlignValue;
    /**
     * (Highmaps) What box to align the buttons to. Possible values are
     * `plotBox` and `spacingBox`.
     */
    alignTo?: ButtonRelativeToValue;
    /**
     * (Highmaps) The pixel height of the map navigation buttons.
     */
    height?: number;
    /**
     * (Highmaps) Click handler for the button.
     */
    onclick?: Function;
    /**
     * (Highmaps) Padding for the navigation buttons.
     */
    padding?: number;
    /**
     * (Highmaps) Text styles for the map navigation buttons.
     */
    style?: CSSObject;
    /**
     * (Highmaps) The text for the button. The tooltip (title) is a language
     * option given by lang.zoomIn.
     */
    text?: string;
    /**
     * (Highmaps) A configuration object for the button theme. The object
     * accepts SVG properties like `stroke-width`, `stroke` and `fill`.
     * Tri-state button styles are supported by the `states.hover` and
     * `states.select` objects.
     */
    theme?: SVGAttributes;
    /**
     * (Highmaps) The vertical alignment of the buttons. Individual alignment
     * can be adjusted by each button's `y` offset.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highmaps) The width of the map navigation buttons.
     */
    width?: number;
    /**
     * (Highmaps) The X offset of the buttons relative to its `align` setting.
     */
    x?: number;
    /**
     * (Highmaps) The position of the zoomIn button relative to the vertical
     * alignment.
     */
    y?: number;
}
/**
 * (Highmaps) Options for the zoom out button. Properties for the zoom in and
 * zoom out buttons are inherited from mapNavigation.buttonOptions, while
 * individual options can be overridden. By default, the `onclick`, `text` and
 * `y` options are individual.
 */
export interface MapNavigationButtonsZoomOutOptions {
    /**
     * (Highmaps) The alignment of the navigation buttons.
     */
    align?: AlignValue;
    /**
     * (Highmaps) What box to align the buttons to. Possible values are
     * `plotBox` and `spacingBox`.
     */
    alignTo?: ButtonRelativeToValue;
    /**
     * (Highmaps) The pixel height of the map navigation buttons.
     */
    height?: number;
    /**
     * (Highmaps) Click handler for the button.
     */
    onclick?: Function;
    /**
     * (Highmaps) Padding for the navigation buttons.
     */
    padding?: number;
    /**
     * (Highmaps) Text styles for the map navigation buttons.
     */
    style?: CSSObject;
    /**
     * (Highmaps) The text for the button. The tooltip (title) is a language
     * option given by lang.zoomOut.
     */
    text?: string;
    /**
     * (Highmaps) A configuration object for the button theme. The object
     * accepts SVG properties like `stroke-width`, `stroke` and `fill`.
     * Tri-state button styles are supported by the `states.hover` and
     * `states.select` objects.
     */
    theme?: SVGAttributes;
    /**
     * (Highmaps) The vertical alignment of the buttons. Individual alignment
     * can be adjusted by each button's `y` offset.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highmaps) The width of the map navigation buttons.
     */
    width?: number;
    /**
     * (Highmaps) The X offset of the buttons relative to its `align` setting.
     */
    x?: number;
    /**
     * (Highmaps) The position of the zoomOut button relative to the vertical
     * alignment.
     */
    y?: number;
}
export interface MapNavigationOptions {
    /**
     * (Highmaps) General options for the map navigation buttons. Individual
     * options can be given from the mapNavigation.buttons option set.
     */
    buttonOptions?: MapNavigationButtonOptions;
    /**
     * (Highmaps) The individual buttons for the map navigation. This usually
     * includes the zoom in and zoom out buttons. Properties for each button is
     * inherited from mapNavigation.buttonOptions, while individual options can
     * be overridden. But default, the `onclick`, `text` and `y` options are
     * individual.
     */
    buttons?: MapNavigationButtonsOptions;
    /**
     * (Highmaps) Whether to enable navigation buttons. By default it inherits
     * the enabled setting.
     */
    enableButtons?: boolean;
    /**
     * (Highmaps) Whether to enable map navigation. The default is not to enable
     * navigation, as many choropleth maps are simple and don't need it.
     * Additionally, when touch zoom and mousewheel zoom is enabled, it breaks
     * the default behaviour of these interactions in the website, and the
     * implementer should be aware of this.
     *
     * Individual interactions can be enabled separately, namely buttons,
     * multitouch zoom, double click zoom, double click zoom to element and
     * mousewheel zoom.
     */
    enabled?: boolean;
    /**
     * (Highmaps) Enables zooming in on an area on double clicking in the map.
     * By default it inherits the enabled setting.
     */
    enableDoubleClickZoom?: boolean;
    /**
     * (Highmaps) Whether to zoom in on an area when that area is double
     * clicked.
     */
    enableDoubleClickZoomTo?: boolean;
    /**
     * (Highmaps) Enables zooming by mouse wheel. By default it inherits the
     * enabled setting.
     */
    enableMouseWheelZoom?: boolean;
    /**
     * (Highmaps) Whether to enable multitouch zooming. Note that if the chart
     * covers the viewport, this prevents the user from using multitouch and
     * touchdrag on the web page, so you should make sure the user is not
     * trapped inside the chart. By default it inherits the enabled setting.
     */
    enableTouchZoom?: boolean;
    /**
     * (Highmaps) Sensitivity of mouse wheel or trackpad scrolling. 1 is no
     * sensitivity, while with 2, one mousewheel delta will zoom in 50%.
     */
    mouseWheelSensitivity?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `annotation`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface NavigationAnnotationsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders annotation immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Events available in annotations.
 */
export interface NavigationAnnotationsEventsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Event callback when annotation
     * is added to the chart.
     */
    add?: EventCallbackFunction<Annotation>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Event callback when annotation
     * is updated (e.g. drag and droppped or resized by control points).
     */
    afterUpdate?: EventCallbackFunction<Annotation>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Event callback when annotation
     * is removed from the chart.
     */
    remove?: EventCallbackFunction<Annotation>;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's labels.
 * Each label inherits options from the labelOptions object. An option from the
 * labelOptions can be overwritten by config for a specific label.
 */
export interface NavigationAnnotationsLabelOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for an
     * annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the
     * annotation's label. If right, the right side of the label should be
     * touching the point.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow the
     * annotation's labels to overlap. To make the labels less sensitive for
     * overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the
     * annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide the annotation's
     * label that is outside the plot area.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The label's pixel distance from
     * the point.
     */
    distance?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the annotation's label. Note that if a `format` or `text` are
     * defined, the format or text take precedence and the formatter is ignored.
     * `This` refers to a point object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether the annotation is
     * visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle the annotation's
     * label that flow outside the plot area. The justify option aligns the
     * label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the borderWidth or
     * the backgroundColor is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. The
     * shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the annotation's
     * label.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Alias for the format option.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) An array of labels for the
 * annotation. For options that apply to multiple labels, they can be added to
 * the labelOptions.
 */
export interface NavigationAnnotationsLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Accessibility options for an
     * annotation label.
     */
    accessibility?: AnnotationLabelAccessibilityOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the
     * annotation's label. If right, the right side of the label should be
     * touching the point.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow the
     * annotation's labels to overlap. To make the labels less sensitive for
     * overlapping, the can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the annotation's label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the
     * annotation's label.
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the annotaiton's label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the annotation's label
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for styling by CSS.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide the annotation's
     * label that is outside the plot area.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The label's pixel distance from
     * the point.
     */
    distance?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the annotation's label. Note that if a `format` or `text` are
     * defined, the format or text take precedence and the formatter is ignored.
     * `This` refers to a point object.
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether the annotation is
     * visible in the exported data table.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle the annotation's
     * label that flow outside the plot area. The justify option aligns the
     * label inside the plot area.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the borderWidth or
     * the backgroundColor is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This option defines the point to
     * which the label will be connected. It can be either the point which
     * exists in the series - it is referenced by the point's id - or a new
     * point with defined x, y properties and optionally axes.
     */
    point?: (string|AnnotationMockPointOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. The
     * shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the annotation's
     * label.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Alias for the format option.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the annotation's label.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * annotation's label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Additional options to be merged into
 * all annotations.
 */
export interface NavigationAnnotationsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `annotation`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|NavigationAnnotationsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's control
     * points. Each control point inherits options from controlPointOptions
     * object. Options from the controlPointOptions can be overwritten by
     * options in a specific control point.
     */
    controlPointOptions?: AnnotationControlPointOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Allow an annotation to be
     * draggable by a user. Possible values are `'x'`, `'xy'`, `'y'` and `''`
     * (disabled).
     */
    draggable?: AnnotationDraggableValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Events available in annotations.
     */
    events?: NavigationAnnotationsEventsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Sets an ID for an annotation.
     * Can be user later when removing an annotation in
     * Chart.removeAnnotation(id) method.
     */
    id?: (number|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's labels.
     * Each label inherits options from the labelOptions object. An option from
     * the labelOptions can be overwritten by config for a specific label.
     */
    labelOptions?: NavigationAnnotationsLabelOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array of labels for the
     * annotation. For options that apply to multiple labels, they can be added
     * to the labelOptions.
     */
    labels?: Array<NavigationAnnotationsLabelsOptions>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's shapes.
     * Each shape inherits options from the shapeOptions object. An option from
     * the shapeOptions can be overwritten by config for a specific shape.
     */
    shapeOptions?: NavigationAnnotationsShapeOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array of shapes for the
     * annotation. For options that apply to multiple shapes, then can be added
     * to the shapeOptions.
     */
    shapes?: Array<NavigationAnnotationsShapesOptions>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether the annotation is
     * visible.
     */
    visible?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the annotation.
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's shapes.
 * Each shape inherits options from the shapeOptions object. An option from the
 * shapeOptions can be overwritten by config for a specific shape.
 */
export interface NavigationAnnotationsShapeOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Name of the dash style to use
     * for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The height of the shape.
     */
    height?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The radius of the shape.
     */
    r?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Defines additional snapping area
     * around an annotation making this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The URL for an image to use as
     * the annotation shape. Note, type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel stroke width of the
     * shape.
     */
    strokeWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The type of the shape, e.g.
     * circle or rectangle.
     */
    type?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the shape.
     */
    width?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) An array of shapes for the
 * annotation. For options that apply to multiple shapes, then can be added to
 * the shapeOptions.
 */
export interface NavigationAnnotationsShapesOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Name of the dash style to use
     * for the shape's stroke.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's fill.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The height of the shape.
     */
    height?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Id of the marker which will be
     * drawn at the final vertex of the path. Custom markers can be defined in
     * defs property.
     */
    markerEnd?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Id of the marker which will be
     * drawn at the first vertex of the path. Custom markers can be defined in
     * defs property.
     */
    markerStart?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This option defines the point to
     * which the shape will be connected. It can be either the point which
     * exists in the series - it is referenced by the point's id - or a new
     * point with defined x, y properties and optionally axes.
     */
    point?: (string|AnnotationMockPointOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array of points for the
     * shape. This option is available for shapes which can use multiple points
     * such as path. A point can be either a point object or a point's id.
     */
    points?: Array<(string|AnnotationMockPointOptionsObject)>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The radius of the shape.
     */
    r?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Defines additional snapping area
     * around an annotation making this annotation to focus. Defined in pixels.
     */
    snap?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The URL for an image to use as
     * the annotation shape. Note, type has to be set to `'image'`.
     */
    src?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's stroke.
     */
    stroke?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel stroke width of the
     * shape.
     */
    strokeWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The type of the shape, e.g.
     * circle or rectangle.
     */
    type?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the shape.
     */
    width?: number;
}
/**
 * (Highcharts, Highstock) Bindings definitions for custom HTML buttons. Each
 * binding implements simple event-driven interface:
 *
 * - `className`: classname used to bind event to
 *
 * - `init`: initial event, fired on button click
 *
 * - `start`: fired on first click on a chart
 *
 * - `steps`: array of sequential events fired one after another on each of
 * users clicks
 *
 * - `end`: last event to be called after last step event
 */
export interface NavigationBindingsOptions {
    /**
     * (Highstock) A line with arrow annotation. Includes `start` and one event
     * in `steps` array.
     */
    arrowInfinityLine?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A ray with an arrow annotation bindings. Includes `start` and
     * one event in `steps` array.
     */
    arrowRay?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A segment with an arrow annotation bindings. Includes `start`
     * and one event in `steps` array.
     */
    arrowSegment?: NavigationBindingsOptionsObject;
    /**
     * (Highcharts, Highstock) A circle annotation bindings. Includes `start`
     * and one event in `steps` array.
     */
    circleAnnotation?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Crooked line (three points) annotation bindings. Includes
     * `start` and two events in `steps` (for second and third points in crooked
     * line) array.
     */
    crooked3?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Crooked line (five points) annotation bindings. Includes
     * `start` and four events in `steps` (for all consequent points in crooked
     * line) array.
     */
    crooked5?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Hides/shows two price indicators:
     *
     * - last price in the dataset
     *
     * - last price in the selected range
     */
    currentPriceIndicator?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Elliott wave (three points) annotation bindings. Includes
     * `start` and two events in `steps` (for second and third points) array.
     */
    elliott3?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Elliott wave (five points) annotation bindings. Includes
     * `start` and four event in `steps` (for all consequent points in Elliott
     * wave) array.
     */
    elliott5?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A fibonacci annotation bindings. Includes `start` and two
     * events in `steps` array (updates second point, then height).
     */
    fibonacci?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A flag series bindings. Includes `start` event. On click,
     * finds the closest point and marks it with a flag with `'circlepin'`
     * shape.
     */
    flagCirclepin?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A flag series bindings. Includes `start` event. On click,
     * finds the closest point and marks it with a flag with `'diamondpin'`
     * shape.
     */
    flagDiamondpin?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A flag series bindings. Includes `start` event. On click,
     * finds the closest point and marks it with a flag without pin shape.
     */
    flagSimplepin?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A flag series bindings. Includes `start` event. On click,
     * finds the closest point and marks it with a flag with `'squarepin'`
     * shape.
     */
    flagSquarepin?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Displays chart in fullscreen.
     *
     * **Note**: Fullscreen is not supported on iPhone due to iOS limitations.
     */
    fullScreen?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A horizontal line annotation. Includes `start` event.
     */
    horizontalLine?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Indicators bindings. Includes `init` event to show a popup.
     *
     * Note: In order to show base series from the chart in the popup's dropdown
     * each series requires series.id to be defined.
     */
    indicators?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A line annotation. Includes `start` and one event in `steps`
     * array.
     */
    infinityLine?: NavigationBindingsOptionsObject;
    /**
     * (Highcharts, Highstock) A label annotation bindings. Includes `start`
     * event only.
     */
    labelAnnotation?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A measure (x-dimension) annotation bindings. Includes `start`
     * and one event in `steps` array.
     */
    measureX?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A measure (xy-dimension) annotation bindings. Includes
     * `start` and one event in `steps` array.
     */
    measureXY?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A measure (y-dimension) annotation bindings. Includes `start`
     * and one event in `steps` array.
     */
    measureY?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A parallel channel (tunnel) annotation bindings. Includes
     * `start` and two events in `steps` array (updates second point, then
     * height).
     */
    parallelChannel?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) An Andrew's pitchfork annotation bindings. Includes `start`
     * and two events in `steps` array (sets second and third control points).
     */
    pitchfork?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A ray annotation bindings. Includes `start` and one event in
     * `steps` array.
     */
    ray?: NavigationBindingsOptionsObject;
    /**
     * (Highcharts, Highstock) A rectangle annotation bindings. Includes `start`
     * and one event in `steps` array.
     */
    rectangleAnnotation?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Save a chart in localStorage under `highcharts-chart` key.
     * Stored items:
     *
     * - annotations
     *
     * - indicators (with yAxes)
     *
     * - flags
     */
    saveChart?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A segment annotation bindings. Includes `start` and one event
     * in `steps` array.
     */
    segment?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Changes main series to `'candlestick'` type.
     */
    seriesTypeCandlestick?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Changes main series to `'line'` type.
     */
    seriesTypeLine?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Changes main series to `'ohlc'` type.
     */
    seriesTypeOhlc?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Hides/shows all annotations on a chart.
     */
    toggleAnnotations?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A vertical arrow annotation bindings. Includes `start` event.
     * On click, finds the closest point and marks it with an arrow. Green arrow
     * when pointing from above, red when pointing from below the point.
     */
    verticalArrow?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A vertical counter annotation bindings. Includes `start`
     * event. On click, finds the closest point and marks it with a numeric
     * annotation - incrementing counter on each add.
     */
    verticalCounter?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A vertical arrow annotation bindings. Includes `start` event.
     * On click, finds the closest point and marks it with an arrow and a label
     * with value.
     */
    verticalLabel?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) A vertical line annotation. Includes `start` event.
     */
    verticalLine?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Enables zooming in xAxis on a chart. Includes `start` event
     * which changes chart.zoomType.
     */
    zoomX?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Enables zooming in xAxis and yAxis on a chart. Includes
     * `start` event which changes chart.zoomType.
     */
    zoomXY?: NavigationBindingsOptionsObject;
    /**
     * (Highstock) Enables zooming in yAxis on a chart. Includes `start` event
     * which changes chart.zoomType.
     */
    zoomY?: NavigationBindingsOptionsObject;
}
/**
 * A config object for navigation bindings in annotations.
 */
export interface NavigationBindingsOptionsObject {
    /**
     * ClassName of the element for a binding.
     */
    className?: string;
    /**
     * Last event to be fired after last step event.
     */
    end?: Function;
    /**
     * Initial event, fired on a button click.
     */
    init?: Function;
    /**
     * Event fired on first click on a chart.
     */
    start?: Function;
    /**
     * Last event to be fired after last step event. Array of step events to be
     * called sequentially after each user click.
     */
    steps?: Array<Function>;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A collection of options for buttons
 * appearing in the exporting module.
 *
 * In styled mode, the buttons are styled with the `.highcharts-contextbutton`
 * and `.highcharts-button-symbol` classes.
 */
export interface NavigationButtonOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Alignment for the buttons.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel spacing between
     * buttons.
     */
    buttonSpacing?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to enable buttons.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Pixel height of the buttons.
     */
    height?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fill color for the symbol within
     * the button.
     */
    symbolFill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel size of the symbol on
     * the button.
     */
    symbolSize?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the symbol's stroke
     * or line.
     */
    symbolStroke?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel stroke width of the
     * symbol on the button.
     */
    symbolStrokeWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position of the center of
     * the symbol inside the button.
     */
    symbolX?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position of the center of
     * the symbol inside the button.
     */
    symbolY?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A text string to add to the
     * individual button.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A configuration object for the
     * button theme. The object accepts SVG properties like `stroke-width`,
     * `stroke` and `fill`. Tri-state button styles are supported by the
     * `states.hover` and `states.select` objects.
     */
    theme?: NavigationButtonThemeOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * buttons. Can be one of `"top"`, `"middle"` or `"bottom"`.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of the button.
     */
    width?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical offset of the
     * button's position relative to its `verticalAlign`.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A configuration object for the
 * button theme. The object accepts SVG properties like `stroke-width`, `stroke`
 * and `fill`. Tri-state button styles are supported by the `states.hover` and
 * `states.select` objects.
 */
export interface NavigationButtonThemeOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The default fill exists only to
     * capture hover events.
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Padding for the button.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Default stroke for the buttons.
     */
    stroke?: ColorString;
}
/**
 * (Highcharts, Highstock) Events to communicate between Stock Tools and custom
 * GUI.
 */
export interface NavigationEventsOptions {
    /**
     * (Highcharts, Highstock) A `closePopup` event. Fired when Popup should be
     * hidden, for example when clicking on an annotation again.
     */
    closePopup?: Function;
    /**
     * (Highcharts, Highstock) Event fired when button state should change, for
     * example after adding an annotation.
     */
    deselectButton?: Function;
    /**
     * (Highcharts, Highstock) Event fired on a button click.
     */
    selectButton?: Function;
    /**
     * (Highcharts, Highstock) A `showPopup` event. Fired when selecting for
     * example an annotation.
     */
    showPopup?: Function;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A collection of options for buttons
 * and menus appearing in the exporting module.
 */
export interface NavigationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Additional options to be merged
     * into all annotations.
     */
    annotationsOptions?: (AnnotationsOptions|NavigationAnnotationsOptions);
    /**
     * (Highcharts, Highstock) Bindings definitions for custom HTML buttons.
     * Each binding implements simple event-driven interface:
     *
     * - `className`: classname used to bind event to
     *
     * - `init`: initial event, fired on button click
     *
     * - `start`: fired on first click on a chart
     *
     * - `steps`: array of sequential events fired one after another on each of
     * users clicks
     *
     * - `end`: last event to be called after last step event
     */
    bindings?: (NavigationBindingsOptions|Dictionary<NavigationBindingsOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A CSS class name where all
     * bindings will be attached to. Multiple charts on the same page should
     * have separate class names to prevent duplicating events.
     *
     * Default value of versions < 7.0.4 `highcharts-bindings-wrapper`
     */
    bindingsClassName?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A collection of options for
     * buttons appearing in the exporting module.
     *
     * In styled mode, the buttons are styled with the
     * `.highcharts-contextbutton` and `.highcharts-button-symbol` classes.
     */
    buttonOptions?: NavigationButtonOptions;
    /**
     * (Highcharts, Highstock) Events to communicate between Stock Tools and
     * custom GUI.
     */
    events?: NavigationEventsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Path where Highcharts will look
     * for icons. Change this to use icons from a different server.
     */
    iconsURL?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the hover state
     * of the individual items within the popup menu appearing by default when
     * the export icon is clicked. The menu items are rendered in HTML.
     */
    menuItemHoverStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the individual
     * items within the popup menu appearing by default when the export icon is
     * clicked. The menu items are rendered in HTML. Font size defaults to
     * `11px` on desktop and `14px` on touch devices.
     */
    menuItemStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the popup menu
     * appearing by default when the export icon is clicked. This menu is
     * rendered in HTML.
     */
    menuStyle?: CSSObject;
}
/**
 * (Highstock, Gantt) Options for the handles for dragging the zoomed area.
 */
export interface NavigatorHandlesOptions {
    /**
     * (Highstock, Gantt) The fill for the handle.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock, Gantt) The stroke for the handle border and the stripes
     * inside.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock, Gantt) Allows to enable/disable handles.
     */
    enabled?: boolean;
    /**
     * (Highstock, Gantt) Height for handles.
     */
    height?: number;
    /**
     * (Highstock, Gantt) The width for the handle border and the stripes
     * inside.
     */
    lineWidth?: object;
    /**
     * (Highstock, Gantt) Array to define shapes of handles. 0-index for left,
     * 1-index for right.
     *
     * Additionally, the URL to a graphic can be given on this form:
     * `url(graphic.png)`. Note that for the image to be applied to exported
     * charts, its URL needs to be accessible by the export server.
     *
     * Custom callbacks for symbol path generation can also be added to
     * `Highcharts.SVGRenderer.prototype.symbols`. The callback is then used by
     * its method name, as shown in the demo.
     */
    symbols?: Array<string>;
    /**
     * (Highstock, Gantt) Width for handles.
     */
    width?: number;
}
/**
 * (Highstock, Gantt) The navigator is a small series below the main series,
 * displaying a view of the entire data set. It provides tools to zoom in and
 * out on parts of the data as well as panning across the dataset.
 */
export interface NavigatorOptions {
    /**
     * (Highstock, Gantt) Whether the navigator and scrollbar should adapt to
     * updated data in the base X axis. When loading data async, as in the demo
     * below, this should be `false`. Otherwise new data will trigger navigator
     * redraw, which will cause unwanted looping. In the demo below, the data in
     * the navigator is set only once. On navigating, only the main chart
     * content is updated.
     */
    adaptToUpdatedData?: boolean;
    /**
     * (Highstock, Gantt) Enable or disable the navigator.
     */
    enabled?: boolean;
    /**
     * (Highstock, Gantt) Options for the handles for dragging the zoomed area.
     */
    handles?: NavigatorHandlesOptions;
    /**
     * (Highstock, Gantt) The height of the navigator.
     */
    height?: number;
    /**
     * (Highstock, Gantt) The distance from the nearest element, the X axis or X
     * axis labels.
     */
    margin?: number;
    /**
     * (Highstock, Gantt) The color of the mask covering the areas of the
     * navigator series that are currently not visible in the main series. The
     * default color is bluish with an opacity of 0.3 to see the series below.
     */
    maskFill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock, Gantt) Whether the mask should be inside the range marking
     * the zoomed range, or outside. In Highstock 1.x it was always `false`.
     */
    maskInside?: boolean;
    /**
     * (Highstock, Gantt) When the chart is inverted, whether to draw the
     * navigator on the opposite side.
     */
    opposite?: boolean;
    /**
     * (Highstock, Gantt) The color of the line marking the currently zoomed
     * area in the navigator.
     */
    outlineColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock, Gantt) The width of the line marking the currently zoomed
     * area in the navigator.
     */
    outlineWidth?: number;
    /**
     * (Highstock, Gantt) Options for the navigator series. Available options
     * are the same as any series, documented at plotOptions and series.
     *
     * Unless data is explicitly defined on navigator.series, the data is
     * borrowed from the first series in the chart.
     *
     * Default series options for the navigator series are: (see online
     * documentation for example)
     */
    series?: (NavigatorSeriesOptions|SeriesOptionsType|Array<NavigatorSeriesOptions>|Array<SeriesOptionsType>);
    /**
     * (Highstock, Gantt) Options for the navigator X axis. Default series
     * options for the navigator xAxis are: (see online documentation for
     * example)
     */
    xAxis?: (NavigatorXAxisOptions|Array<NavigatorXAxisOptions>);
    /**
     * (Highstock, Gantt) Options for the navigator Y axis. Default series
     * options for the navigator yAxis are: (see online documentation for
     * example)
     */
    yAxis?: (NavigatorYAxisOptions|Array<NavigatorYAxisOptions>);
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface NavigatorSeriesDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Data label options for the navigator
 * series. Data labels are disabled by default on the navigator series.
 */
export interface NavigatorSeriesDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|NavigatorSeriesDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
    zIndex?: number;
}
export interface NavigatorSeriesMarkerOptions {
    enabled?: boolean;
}
/**
 * (Highstock, Gantt) Options for the navigator series. Available options are
 * the same as any series, documented at plotOptions and series.
 *
 * Unless data is explicitly defined on navigator.series, the data is borrowed
 * from the first series in the chart.
 *
 * Default series options for the navigator series are: (see online
 * documentation for example)
 */
export interface NavigatorSeriesOptions {
    className?: string;
    /**
     * (Highstock, Gantt) Sets the fill color of the navigator series.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Unless data is explicitly defined, the data is borrowed from
     * the first series in the chart.
     */
    data?: Array<(number|Array<(number|string|null)>|object|null)>;
    /**
     * (Highstock) Data grouping options for the navigator series.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Data label options for the
     * navigator series. Data labels are disabled by default on the navigator
     * series.
     */
    dataLabels?: (NavigatorSeriesDataLabelsOptions|Array<NavigatorSeriesDataLabelsOptions>);
    /**
     * (Highstock, Gantt) The fill opacity of the navigator series.
     */
    fillOpacity?: number;
    id?: string;
    /**
     * (Highstock, Gantt) Line color for the navigator series. Allows setting
     * the color while disallowing the default candlestick setting.
     */
    lineColor?: (ColorString|null);
    /**
     * (Highstock, Gantt) The pixel line width of the navigator series.
     */
    lineWidth?: number;
    marker?: NavigatorSeriesMarkerOptions;
    /**
     * (Highstock) Since Highstock v8, default value is the same as default
     * `pointRange` defined for a specific type (e.g. `null` for column type).
     *
     * In Highstock version < 8, defaults to 0.
     */
    pointRange?: (number|null);
    /**
     * (Highstock, Gantt) The threshold option. Setting it to 0 will make the
     * default navigator area series draw its area from the 0 value and up.
     */
    threshold?: (number|null);
    /**
     * (Highstock, Gantt) The type of the navigator series.
     *
     * Heads up: In column-type navigator, zooming is limited to at least one
     * point with its `pointRange`.
     */
    type?: string;
}
/**
 * (Highcharts, Highstock, Gantt) An array defining breaks in the axis, the
 * sections defined will be left out and all the points shifted closer to each
 * other.
 */
export interface NavigatorXAxisBreaksOptions {
    /**
     * (Highcharts, Highstock, Gantt) A number indicating how much space should
     * be left between the start and the end of the break. The break size is
     * given in axis units, so for instance on a `datetime` axis, a break size
     * of 3600000 would indicate the equivalent of an hour.
     */
    breakSize?: number;
    /**
     * (Highcharts, Highstock, Gantt) The point where the break starts.
     */
    from?: number;
    /**
     * (Highcharts, Highstock, Gantt) Defines an interval after which the break
     * appears again. By default the breaks do not repeat.
     */
    repeat?: number;
    /**
     * (Highcharts, Highstock, Gantt) The point where the break ends.
     */
    to?: number;
}
/**
 * (Gantt) An object defining mouse events for the plot line. Supported
 * properties are `click`, `mouseover`, `mouseout`, `mousemove`.
 */
export interface NavigatorXAxisCurrentDateIndicatorEventsOptions {
    /**
     * (Gantt) Click event on a plot band.
     */
    click?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Gantt) Mouse move event on a plot band.
     */
    mousemove?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Gantt) Mouse out event on the corner of a plot band.
     */
    mouseout?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Gantt) Mouse over event on a plot band.
     */
    mouseover?: EventCallbackFunction<PlotLineOrBand>;
}
/**
 * (Gantt) Text labels for the plot bands
 */
export interface NavigatorXAxisCurrentDateIndicatorLabelOptions {
    /**
     * (Gantt) Horizontal alignment of the label. Can be one of "left", "center"
     * or "right".
     */
    align?: AlignValue;
    /**
     * (Gantt) Format of the label. This options is passed as the fist argument
     * to dateFormat function.
     */
    format?: string;
    /**
     * (Gantt) Callback JavaScript function to format the label. Useful
     * properties like the value of plot line or the range of plot band (`from`
     * & `to` properties) can be found in `this.options` object.
     */
    formatter?: FormatterCallbackFunction<PlotLineOrBand>;
    /**
     * (Gantt) Rotation of the text label in degrees. Defaults to 0 for
     * horizontal plot lines and 90 for vertical lines.
     */
    rotation?: number;
    /**
     * (Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-line-label` class.
     */
    style?: CSSObject;
    /**
     * (Gantt) The text itself. A subset of HTML is supported.
     */
    text?: string;
    /**
     * (Gantt) The text alignment for the label. While `align` determines where
     * the texts anchor point is placed within the plot band, `textAlign`
     * determines how the text is aligned against its anchor point. Possible
     * values are "left", "center" and "right". Defaults to the same as the
     * `align` option.
     */
    textAlign?: AlignValue;
    /**
     * (Gantt) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Gantt) Vertical alignment of the label relative to the plot line. Can be
     * one of "top", "middle" or "bottom".
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Gantt) Horizontal position relative the alignment. Default varies by
     * orientation.
     */
    x?: number;
    /**
     * (Gantt) Vertical position of the text baseline relative to the alignment.
     * Default varies by orientation.
     */
    y?: number;
}
/**
 * (Highstock, Gantt) Event handlers for the axis.
 */
export interface NavigatorXAxisEventsOptions {
    /**
     * (Highcharts, Gantt) An event fired after the breaks have rendered.
     */
    afterBreaks?: AxisEventCallbackFunction;
    /**
     * (Highstock, Gantt) As opposed to the `setExtremes` event, this event
     * fires after the final min and max values are computed and corrected for
     * `minRange`.
     *
     * Fires when the minimum and maximum is set for the axis, either by calling
     * the `.setExtremes()` method or by selecting an area in the chart. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in axis
     * values. The actual data extremes are found in `event.dataMin` and
     * `event.dataMax`.
     */
    afterSetExtremes?: AxisSetExtremesEventCallbackFunction;
    /**
     * (Highcharts, Gantt) An event fired when a break from this axis occurs on
     * a point.
     */
    pointBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highcharts, Highstock, Gantt) An event fired when a point falls inside a
     * break from this axis.
     */
    pointInBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highstock, Gantt) Fires when the minimum and maximum is set for the
     * axis, either by calling the `.setExtremes()` method or by selecting an
     * area in the chart. One parameter, `event`, is passed to the function,
     * containing common event information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in data
     * values. When an axis is zoomed all the way out from the "Reset zoom"
     * button, `event.min` and `event.max` are null, and the new extremes are
     * set based on `this.dataMin` and `this.dataMax`.
     */
    setExtremes?: AxisSetExtremesEventCallbackFunction;
}
/**
 * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
 */
export interface NavigatorXAxisGridOptions {
    /**
     * (Gantt) Set border color for the label grid lines.
     */
    borderColor?: ColorString;
    /**
     * (Gantt) Set border width of the label grid lines.
     */
    borderWidth?: number;
    /**
     * (Gantt) Set cell height for grid axis labels. By default this is
     * calculated from font size. This option only applies to horizontal axes.
     */
    cellHeight?: number;
    /**
     * (Gantt) Set specific options for each column (or row for horizontal axes)
     * in the grid. Each extra column/row is its own axis, and the axis options
     * can be set here.
     */
    columns?: Array<XAxisOptions>;
    /**
     * (Gantt) Enable grid on the axis labels. Defaults to true for Gantt
     * charts.
     */
    enabled?: boolean;
}
/**
 * (Highstock, Gantt) The axis labels show the number or category for each tick.
 *
 * Since v8.0.0: Labels are animated in categorized x-axis with updating data if
 * `tickInterval` and `step` is set to 1.
 */
export interface NavigatorXAxisLabelsOptions {
    /**
     * (Highstock, Gantt) What part of the string the given position is anchored
     * to. If `left`, the left side of the string is at the axis position. Can
     * be one of `"left"`, `"center"` or `"right"`. Defaults to an intelligent
     * guess based on which side of the chart the axis is on and the rotation of
     * the label.
     */
    align?: string;
    /**
     * (Highcharts, Highstock, Gantt) For horizontal axes, the allowed degrees
     * of label rotation to prevent overlapping labels. If there is enough
     * space, labels are not rotated. As the chart gets narrower, it will start
     * rotating the labels -45 degrees, then remove every second label and try
     * again with rotations 0 and -45 etc. Set it to `false` to disable
     * rotation, which will cause the labels to word-wrap if possible.
     */
    autoRotation?: (false|Array<number>);
    /**
     * (Highcharts, Gantt) When each category width is more than this many
     * pixels, we don't apply auto rotation. Instead, we lay out the axis label
     * with word wrap. A lower limit makes sense when the label contains
     * multiple short words that don't extend the available horizontal space for
     * each label.
     */
    autoRotationLimit?: number;
    /**
     * (Highcharts, Gantt) Polar charts only. The label's pixel distance from
     * the perimeter of the plot area.
     */
    distance?: number;
    /**
     * (Highstock, Gantt) Enable or disable the axis labels.
     */
    enabled?: boolean;
    /**
     * (Highstock, Gantt) A format string for the axis label.
     */
    format?: string;
    /**
     * (Highstock, Gantt) Callback JavaScript function to format the label. The
     * value is given by `this.value`. Additional properties for `this` are
     * `axis`, `chart`, `isFirst` and `isLast`. The value of the default label
     * formatter can be retrieved by calling
     * `this.axis.defaultLabelFormatter.call(this)` within the function.
     *
     * Defaults to: (see online documentation for example)
     */
    formatter?: AxisLabelsFormatterCallbackFunction;
    /**
     * (Gantt) The number of pixels to indent the labels per level in a treegrid
     * axis.
     */
    indentation?: number;
    /**
     * (Highstock, Gantt) How to handle overflowing labels on horizontal axis.
     * If set to `"allow"`, it will not be aligned at all. By default it
     * `"justify"` labels inside the chart area. If there is room to move it, it
     * will be aligned to the edge, else it will be removed.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highcharts, Gantt) The pixel padding for axis labels, to ensure white
     * space between them.
     */
    padding?: number;
    /**
     * (Highcharts) Defines how the labels are be repositioned according to the
     * 3D chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     */
    position3d?: OptionsPosition3dValue;
    /**
     * (Highcharts, Gantt) Whether to reserve space for the labels. By default,
     * space is reserved for the labels in these cases:
     *
     * * On all horizontal axes.
     *
     * * On vertical axes if `label.align` is `right` on a left-side axis or
     * `left` on a right-side axis.
     *
     * * On vertical axes if `label.align` is `center`.
     *
     * This can be turned off when for example the labels are rendered inside
     * the plot area instead of outside.
     */
    reserveSpace?: boolean;
    /**
     * (Highstock, Gantt) Rotation of the labels in degrees.
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis labels will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `labels.position3d`.
     */
    skew3d?: boolean;
    /**
     * (Highstock, Gantt) Horizontal axes only. The number of lines to spread
     * the labels over to make room or tighter labels.
     */
    staggerLines?: number;
    /**
     * (Highstock, Gantt) To show only every _n_'th label on the axis, set the
     * step to _n_. Setting the step to 2 shows every other label.
     *
     * By default, the step is calculated automatically to avoid overlap. To
     * prevent this, set it to 1\. This usually only happens on a category axis,
     * and is often a sign that you have chosen the wrong axis type.
     *
     * Read more at Axis docs => What axis should I use?
     */
    step?: number;
    /**
     * (Highstock, Gantt) CSS styles for the label. Use `whiteSpace: 'nowrap'`
     * to prevent wrapping of category labels. Use `textOverflow: 'none'` to
     * prevent ellipsis (dots).
     *
     * In styled mode, the labels are styled with the `.highcharts-axis-labels`
     * class.
     */
    style?: CSSObject;
    /**
     * (Highstock, Gantt) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Highstock, Gantt) The x position offset of all labels relative to the
     * tick positions on the axis.
     */
    x?: number;
    /**
     * (Highstock, Gantt) The y position offset of all labels relative to the
     * tick positions on the axis. The default makes it adapt to the font size
     * of the bottom axis.
     */
    y?: number;
    /**
     * (Highstock, Gantt) The Z index for the axis labels.
     */
    zIndex?: number;
}
/**
 * (Highstock, Gantt) Options for the navigator X axis. Default series options
 * for the navigator xAxis are: (see online documentation for example)
 */
export interface NavigatorXAxisOptions {
    /**
     * (Highstock, Gantt) Accessibility options for an axis. Requires the
     * accessibility module.
     */
    accessibility?: AxisAccessibilityOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When using multiple axis, the ticks of two
     * or more opposite axes will automatically be aligned by adding ticks to
     * the axis or axes with the least ticks, as if `tickAmount` were specified.
     *
     * This can be prevented by setting `alignTicks` to false. If the grid lines
     * look messy, it's a good idea to hide them for the secondary axis by
     * setting `gridLineWidth` to 0.
     *
     * If `startOnTick` or `endOnTick` in an Axis options are set to false, then
     * the `alignTicks ` will be disabled for the Axis.
     *
     * Disabled for logarithmic axes.
     */
    alignTicks?: boolean;
    /**
     * (Highstock, Gantt) Whether to allow decimals in this axis' ticks. When
     * counting integers, like persons or hits on a web page, decimals should be
     * avoided in the labels.
     */
    allowDecimals?: boolean;
    /**
     * (Highstock, Gantt) When using an alternate grid color, a band is painted
     * across the plot area between every other grid line.
     */
    alternateGridColor?: ColorType;
    /**
     * (Highcharts) In a polar chart, this is the angle of the Y axis in
     * degrees, where 0 is up and 90 is right. The angle determines the position
     * of the axis line and the labels, though the coordinate system is
     * unaffected. Since v8.0.0 this option is also applicable for X axis
     * (inverted polar).
     */
    angle?: number;
    /**
     * (Highcharts, Highstock, Gantt) An array defining breaks in the axis, the
     * sections defined will be left out and all the points shifted closer to
     * each other.
     */
    breaks?: Array<NavigatorXAxisBreaksOptions>;
    /**
     * (Highcharts, Gantt) If categories are present for the xAxis, names are
     * used instead of numbers for that axis.
     *
     * Since Highcharts 3.0, categories can also be extracted by giving each
     * point a name and setting axis type to `category`. However, if you have
     * multiple series, best practice remains defining the `categories` array.
     *
     * Example: `categories: ['Apples', 'Bananas', 'Oranges']`
     */
    categories?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) The highest allowed value for
     * automatically computed axis extremes.
     */
    ceiling?: number;
    /**
     * (Highstock, Gantt) A class name that opens for styling the axis by CSS,
     * especially in Highcharts styled mode. The class name is applied to group
     * elements for the grid, axis elements and labels.
     */
    className?: string;
    /**
     * (Highstock, Gantt) Configure a crosshair that follows either the mouse
     * pointer or the hovered point.
     *
     * In styled mode, the crosshairs are styled in the `.highcharts-crosshair`,
     * `.highcharts-crosshair-thin` or `.highcharts-xaxis-category` classes.
     */
    crosshair?: (boolean|AxisCrosshairOptions);
    /**
     * (Gantt) Show an indicator on the axis for the current date and time. Can
     * be a boolean or a configuration object similar to xAxis.plotLines.
     */
    currentDateIndicator?: (boolean|AxisCurrentDateIndicatorOptions);
    /**
     * (Highcharts, Highstock, Gantt) For a datetime axis, the scale will
     * automatically adjust to the appropriate unit. This member gives the
     * default string representations used for each unit. For intermediate
     * values, different units may be used, for example the `day` unit can be
     * used on midnight and `hour` unit be used for intermediate values on the
     * same axis.
     *
     * For an overview of the replacement codes, see dateFormat.
     *
     * Defaults to: (see online documentation for example)
     */
    dateTimeLabelFormats?: AxisDateTimeLabelFormatsOptions;
    /**
     * (Highstock, Gantt) Whether to force the axis to end on a tick. Use this
     * option with the `maxPadding` option to control the axis end.
     */
    endOnTick?: boolean;
    /**
     * (Highstock, Gantt) Event handlers for the axis.
     */
    events?: NavigatorXAxisEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) The lowest allowed value for automatically
     * computed axis extremes.
     */
    floor?: number;
    /**
     * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
     */
    grid?: NavigatorXAxisGridOptions;
    /**
     * (Highstock, Gantt) Color of the grid lines extending the ticks across the
     * plot area.
     *
     * In styled mode, the stroke is given in the `.highcharts-grid-line` class.
     */
    gridLineColor?: string;
    /**
     * (Highstock, Gantt) The dash or dot style of the grid lines. For possible
     * values, see this demonstration.
     */
    gridLineDashStyle?: DashStyleValue;
    /**
     * (Highcharts) Polar charts only. Whether the grid lines should draw as a
     * polygon with straight lines between categories, or as circles. Can be
     * either `circle` or `polygon`. Since v8.0.0 this option is also applicable
     * for X axis (inverted polar).
     */
    gridLineInterpolation?: OptionsGridLineInterpolationValue;
    /**
     * (Highstock, Gantt) The width of the grid lines extending the ticks across
     * the plot area.
     *
     * In styled mode, the stroke width is given in the `.highcharts-grid-line`
     * class.
     */
    gridLineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The Z index of the grid lines.
     */
    gridZIndex?: number;
    /**
     * (Highcharts, Highstock) The height as the vertical axis. If it's a
     * number, it is interpreted as pixels.
     *
     * Since Highcharts 2: If it's a percentage string, it is interpreted as
     * percentages of the total plot height.
     */
    height?: (number|string);
    /**
     * (Highstock, Gantt) An id for the axis. This can be used after render time
     * to get a pointer to the axis object through `chart.get()`.
     */
    id?: string;
    /**
     * (Highstock, Gantt) The axis labels show the number or category for each
     * tick.
     *
     * Since v8.0.0: Labels are animated in categorized x-axis with updating
     * data if `tickInterval` and `step` is set to 1.
     */
    labels?: NavigatorXAxisLabelsOptions;
    /**
     * (Highcharts, Highstock) The left position as the horizontal axis. If it's
     * a number, it is interpreted as pixel position relative to the chart.
     *
     * Since Highcharts v5.0.13: If it's a percentage string, it is interpreted
     * as percentages of the plot width, offset from plot area left.
     */
    left?: (number|string);
    /**
     * (Highstock, Gantt) The color of the line marking the axis itself.
     *
     * In styled mode, the line stroke is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     */
    lineColor?: ColorType;
    /**
     * (Highstock, Gantt) The width of the line marking the axis itself.
     *
     * In styled mode, the stroke width is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     */
    lineWidth?: number;
    /**
     * (Highstock, Gantt) If there are multiple axes on the same side of the
     * chart, the pixel margin between the axes. Defaults to 0 on vertical axes,
     * 15 on horizontal axes.
     */
    margin?: number;
    /**
     * (Highstock, Gantt) The maximum value of the axis. If `null`, the max
     * value is automatically calculated.
     *
     * If the endOnTick option is true, the `max` value might be rounded up.
     *
     * If a tickAmount is set, the axis may be extended beyond the set max in
     * order to reach the given number of ticks. The same may happen in a chart
     * with multiple axes, determined by chart. alignTicks, where a `tickAmount`
     * is applied internally.
     */
    max?: (number|null);
    /**
     * (Highstock, Gantt) Padding of the max value relative to the length of the
     * axis. A padding of 0.05 will make a 100px axis 5px longer. This is useful
     * when you don't want the highest data value to appear on the edge of the
     * plot area. When the axis' `max` option is set or a max extreme is set
     * using `axis.setExtremes()`, the maxPadding will be ignored.
     */
    maxPadding?: number;
    /**
     * (Highstock, Gantt) The minimum value of the axis. If `null` the min value
     * is automatically calculated.
     *
     * If the startOnTick option is true (default), the `min` value might be
     * rounded down.
     *
     * The automatically calculated minimum value is also affected by floor,
     * softMin, minPadding, minRange as well as series.threshold and
     * series.softThreshold.
     */
    min?: (number|null);
    /**
     * (Highstock, Gantt) Color of the minor, secondary grid lines.
     *
     * In styled mode, the stroke width is given in the
     * `.highcharts-minor-grid-line` class.
     */
    minorGridLineColor?: ColorType;
    /**
     * (Highstock, Gantt) The dash or dot style of the minor grid lines. For
     * possible values, see this demonstration.
     */
    minorGridLineDashStyle?: DashStyleValue;
    /**
     * (Highstock, Gantt) Width of the minor, secondary grid lines.
     *
     * In styled mode, the stroke width is given in the `.highcharts-grid-line`
     * class.
     */
    minorGridLineWidth?: number;
    /**
     * (Highstock, Gantt) Color for the minor tick marks.
     */
    minorTickColor?: ColorType;
    /**
     * (Highstock, Gantt) Specific tick interval in axis units for the minor
     * ticks. On a linear axis, if `"auto"`, the minor tick interval is
     * calculated as a fifth of the tickInterval. If `null` or `undefined`,
     * minor ticks are not shown.
     *
     * On logarithmic axes, the unit is the power of the value. For example,
     * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
     * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
     * and 10, 10 and 100 etc.
     *
     * If user settings dictate minor ticks to become too dense, they don't make
     * sense, and will be ignored to prevent performance problems.
     */
    minorTickInterval?: (number|string|null);
    /**
     * (Highstock, Gantt) The pixel length of the minor tick marks.
     */
    minorTickLength?: number;
    /**
     * (Highstock, Gantt) The position of the minor tick marks relative to the
     * axis line. Can be one of `inside` and `outside`.
     */
    minorTickPosition?: OptionsMinorTickPositionValue;
    /**
     * (Highstock, Gantt) Enable or disable minor ticks. Unless
     * minorTickInterval is set, the tick interval is calculated as a fifth of
     * the `tickInterval`.
     *
     * On a logarithmic axis, minor ticks are laid out based on a best guess,
     * attempting to enter approximately 5 minor ticks between each major tick.
     *
     * Prior to v6.0.0, ticks were unabled in auto layout by setting
     * `minorTickInterval` to `"auto"`.
     */
    minorTicks?: boolean;
    /**
     * (Highstock, Gantt) The pixel width of the minor tick mark.
     */
    minorTickWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) Padding of the min value relative to the
     * length of the axis. A padding of 0.05 will make a 100px axis 5px longer.
     * This is useful when you don't want the lowest data value to appear on the
     * edge of the plot area. When the axis' `min` option is set or a min
     * extreme is set using `axis.setExtremes()`, the minPadding will be
     * ignored.
     */
    minPadding?: number;
    /**
     * (Highstock, Gantt) The minimum tick interval allowed in axis values. For
     * example on zooming in on an axis with daily data, this can be used to
     * prevent the axis from showing hours. Defaults to the closest distance
     * between two points on the axis.
     */
    minTickInterval?: number;
    /**
     * (Highstock, Gantt) The distance in pixels from the plot area to the axis
     * line. A positive offset moves the axis with it's line, labels and ticks
     * away from the plot area. This is typically used when two or more axes are
     * displayed on the same side of the plot. With multiple axes the offset is
     * dynamically adjusted to avoid collision, this can be overridden by
     * setting offset explicitly.
     */
    offset?: number;
    /**
     * (Highstock) In an ordinal axis, the points are equally spaced in the
     * chart regardless of the actual time or x distance between them. This
     * means that missing data periods (e.g. nights or weekends for a stock
     * chart) will not take up space in the chart. Having `ordinal: false` will
     * show any gaps created by the `gapSize` setting proportionate to their
     * duration.
     *
     * In stock charts the X axis is ordinal by default, unless the boost module
     * is used and at least one of the series' data length exceeds the
     * boostThreshold.
     */
    ordinal?: boolean;
    /**
     * (Highstock) Additional range on the right side of the xAxis. Works
     * similar to xAxis.maxPadding, but value is set in milliseconds. Can be set
     * for both, main xAxis and navigator's xAxis.
     */
    overscroll?: number;
    /**
     * (Highcharts) Refers to the index in the panes array. Used for circular
     * gauges and polar charts. When the option is not set then first pane will
     * be used.
     */
    pane?: number;
    /**
     * (Highcharts, Highstock, Gantt) An array of colored bands stretching
     * across the plot area marking an interval on the axis.
     *
     * In styled mode, the plot bands are styled by the `.highcharts-plot-band`
     * class in addition to the `className` option.
     */
    plotBands?: Array<NavigatorXAxisPlotBandsOptions>;
    /**
     * (Highcharts, Highstock, Gantt) An array of lines stretching across the
     * plot area, marking a specific value on one of the axes.
     *
     * In styled mode, the plot lines are styled by the `.highcharts-plot-line`
     * class in addition to the `className` option.
     */
    plotLines?: Array<NavigatorXAxisPlotLinesOptions>;
    /**
     * (Highstock, Gantt) Whether to reverse the axis so that the highest number
     * is closest to the origin. If the chart is inverted, the x axis is
     * reversed by default.
     */
    reversed?: boolean;
    /**
     * (Highcharts, Highstock) This option determines how stacks should be
     * ordered within a group. For example reversed xAxis also reverses stacks,
     * so first series comes last in a group. To keep order like for
     * non-reversed xAxis enable this option.
     */
    reversedStacks?: boolean;
    /**
     * (Highstock, Gantt) Whether to show the first tick label.
     */
    showFirstLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to show the last tick label.
     * Defaults to `true` on cartesian charts, and `false` on polar charts.
     */
    showLastLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A soft maximum for the axis. If the series
     * data maximum is less than this, the axis will stay at this maximum, but
     * if the series data maximum is higher, the axis will flex to show all
     * data.
     */
    softMax?: number;
    /**
     * (Highcharts, Highstock, Gantt) A soft minimum for the axis. If the series
     * data minimum is greater than this, the axis will stay at this minimum,
     * but if the series data minimum is lower, the axis will flex to show all
     * data.
     */
    softMin?: number;
    /**
     * (Highcharts, Highstock, Gantt) For datetime axes, this decides where to
     * put the tick between weeks. 0 = Sunday, 1 = Monday.
     */
    startOfWeek?: number;
    /**
     * (Highstock, Gantt) Whether to force the axis to start on a tick. Use this
     * option with the `minPadding` option to control the axis start.
     */
    startOnTick?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) The amount of ticks to draw on the axis.
     * This opens up for aligning the ticks of multiple charts or panes within a
     * chart. This option overrides the `tickPixelInterval` option.
     *
     * This option only has an effect on linear axes. Datetime, logarithmic or
     * category axes are not affected.
     */
    tickAmount?: number;
    /**
     * (Highstock, Gantt) Color for the main tick marks.
     *
     * In styled mode, the stroke is given in the `.highcharts-tick` class.
     */
    tickColor?: ColorType;
    /**
     * (Highstock, Gantt) The interval of the tick marks in axis units. When
     * `undefined`, the tick interval is computed to approximately follow the
     * tickPixelInterval on linear and datetime axes. On categorized axes, a
     * `undefined` tickInterval will default to 1, one category. Note that
     * datetime axes are based on milliseconds, so for example an interval of
     * one day is expressed as `24 * 3600 * 1000`.
     *
     * On logarithmic axes, the tickInterval is based on powers, so a
     * tickInterval of 1 means one tick on each of 0.1, 1, 10, 100 etc. A
     * tickInterval of 2 means a tick of 0.1, 10, 1000 etc. A tickInterval of
     * 0.2 puts a tick on 0.1, 0.2, 0.4, 0.6, 0.8, 1, 2, 4, 6, 8, 10, 20, 40
     * etc.
     *
     * If the tickInterval is too dense for labels to be drawn, Highcharts may
     * remove ticks.
     *
     * If the chart has multiple axes, the alignTicks option may interfere with
     * the `tickInterval` setting.
     */
    tickInterval?: number;
    /**
     * (Highstock, Gantt) The pixel length of the main tick marks.
     */
    tickLength?: number;
    /**
     * (Highcharts, Gantt) For categorized axes only. If `on` the tick mark is
     * placed in the center of the category, if `between` the tick mark is
     * placed between categories. The default is `between` if the `tickInterval`
     * is 1, else `on`.
     */
    tickmarkPlacement?: OptionsTickmarkPlacementValue;
    /**
     * (Highstock, Gantt) If tickInterval is `null` this option sets the
     * approximate pixel interval of the tick marks. Not applicable to
     * categorized axis.
     *
     * The tick interval is also influenced by the minTickInterval option, that,
     * by default prevents ticks from being denser than the data points.
     */
    tickPixelInterval?: number;
    /**
     * (Highstock, Gantt) The position of the major tick marks relative to the
     * axis line. Can be one of `inside` and `outside`.
     */
    tickPosition?: OptionsTickPositionValue;
    /**
     * (Highstock, Gantt) A callback function returning array defining where the
     * ticks are laid out on the axis. This overrides the default behaviour of
     * tickPixelInterval and tickInterval. The automatic tick positions are
     * accessible through `this.tickPositions` and can be modified by the
     * callback.
     */
    tickPositioner?: AxisTickPositionerCallbackFunction;
    /**
     * (Highstock, Gantt) An array defining where the ticks are laid out on the
     * axis. This overrides the default behaviour of tickPixelInterval and
     * tickInterval.
     */
    tickPositions?: Array<number>;
    /**
     * (Highstock, Gantt) The pixel width of the major tick marks. Defaults to 0
     * on category axes, otherwise 1.
     *
     * In styled mode, the stroke width is given in the `.highcharts-tick`
     * class, but in order for the element to be generated on category axes, the
     * option must be explicitly set to 1.
     */
    tickWidth?: (number|undefined);
    /**
     * (Highstock, Gantt) The axis title, showing next to the axis line.
     */
    title?: NavigatorXAxisTitleOptions;
    /**
     * (Highcharts, Highstock) The top position as the vertical axis. If it's a
     * number, it is interpreted as pixel position relative to the chart.
     *
     * Since Highcharts 2: If it's a percentage string, it is interpreted as
     * percentages of the plot height, offset from plot area top.
     */
    top?: (number|string);
    /**
     * (Highcharts, Gantt) The type of axis. Can be one of `linear`,
     * `logarithmic`, `datetime` or `category`. In a datetime axis, the numbers
     * are given in milliseconds, and tick marks are placed on appropriate
     * values like full hours or days. In a category axis, the point names of
     * the chart's series are used for categories, if not a categories array is
     * defined.
     */
    type?: AxisTypeValue;
    /**
     * (Highcharts, Gantt) Applies only when the axis `type` is `category`. When
     * `uniqueNames` is true, points are placed on the X axis according to their
     * names. If the same point name is repeated in the same or another series,
     * the point is placed on the same X position as other points of the same
     * name. When `uniqueNames` is false, the points are laid out in increasing
     * X positions regardless of their names, and the X axis category will take
     * the name of the last point in each position.
     */
    uniqueNames?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Datetime axis only. An array determining
     * what time intervals the ticks are allowed to fall on. Each array item is
     * an array where the first value is the time unit and the second value
     * another array of allowed multiples.
     *
     * Defaults to: (see online documentation for example)
     */
    units?: Array<[string, (Array<number>|null)]>;
    /**
     * (Highcharts, Highstock, Gantt) Whether axis, including axis title, line,
     * ticks and labels, should be visible.
     */
    visible?: boolean;
    /**
     * (Highcharts, Highstock) The width as the horizontal axis. If it's a
     * number, it is interpreted as pixels.
     *
     * Since Highcharts v5.0.13: If it's a percentage string, it is interpreted
     * as percentages of the total plot width.
     */
    width?: (number|string);
    /**
     * (Highstock, Gantt) Whether to zoom axis. If `chart.zoomType` is set, the
     * option allows to disable zooming on an individual axis.
     */
    zoomEnabled?: boolean;
}
/**
 * (Highcharts, Highstock, Gantt) An object defining mouse events for the plot
 * band. Supported properties are `click`, `mouseover`, `mouseout`, `mousemove`.
 */
export interface NavigatorXAxisPlotBandsEventsOptions {
    /**
     * (Highcharts, Highstock, Gantt) Click event on a plot band.
     */
    click?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse move event on a plot band.
     */
    mousemove?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse out event on the corner of a plot
     * band.
     */
    mouseout?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse over event on a plot band.
     */
    mouseover?: EventCallbackFunction<PlotLineOrBand>;
}
/**
 * (Highcharts, Highstock, Gantt) Text labels for the plot bands
 */
export interface NavigatorXAxisPlotBandsLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) Horizontal alignment of the label. Can be
     * one of "left", "center" or "right".
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Rotation of the text label in degrees .
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-band-label` class.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) The string text itself. A subset of HTML
     * is supported.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Gantt) The text alignment for the label. While
     * `align` determines where the texts anchor point is placed within the plot
     * band, `textAlign` determines how the text is aligned against its anchor
     * point. Possible values are "left", "center" and "right". Defaults to the
     * same as the `align` option.
     */
    textAlign?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Vertical alignment of the label relative
     * to the plot band. Can be one of "top", "middle" or "bottom".
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal position relative the
     * alignment. Default varies by orientation.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical position of the text baseline
     * relative to the alignment. Default varies by orientation.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An array of colored bands stretching across
 * the plot area marking an interval on the axis.
 *
 * In styled mode, the plot bands are styled by the `.highcharts-plot-band`
 * class in addition to the `className` option.
 */
export interface NavigatorXAxisPlotBandsOptions {
    /**
     * (Highstock) Flag to decide if plotBand should be rendered across all
     * panes.
     */
    acrossPanes?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Border color for the plot band. Also
     * requires `borderWidth` to be set.
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) Border width for the plot band. Also
     * requires `borderColor` to be set.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) A custom class name, in addition to the
     * default `highcharts-plot-band`, to apply to each individual band.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) The color of the plot band.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Gantt) An object defining mouse events for the
     * plot band. Supported properties are `click`, `mouseover`, `mouseout`,
     * `mousemove`.
     */
    events?: NavigatorXAxisPlotBandsEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) The start position of the plot band in
     * axis units.
     */
    from?: number;
    /**
     * (Highcharts, Highstock, Gantt) An id used for identifying the plot band
     * in Axis.removePlotBand.
     */
    id?: string;
    /**
     * (Highcharts, Highstock, Gantt) Text labels for the plot bands
     */
    label?: NavigatorXAxisPlotBandsLabelOptions;
    /**
     * (Highcharts, Highstock, Gantt) The end position of the plot band in axis
     * units.
     */
    to?: number;
    /**
     * (Highcharts, Highstock, Gantt) The z index of the plot band within the
     * chart, relative to other elements. Using the same z index as another
     * element may give unpredictable results, as the last rendered element will
     * be on top. Values from 0 to 20 make sense.
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An object defining mouse events for the plot
 * line. Supported properties are `click`, `mouseover`, `mouseout`, `mousemove`.
 */
export interface NavigatorXAxisPlotLinesEventsOptions {
    /**
     * (Highcharts, Highstock, Gantt) Click event on a plot band.
     */
    click?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse move event on a plot band.
     */
    mousemove?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse out event on the corner of a plot
     * band.
     */
    mouseout?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse over event on a plot band.
     */
    mouseover?: EventCallbackFunction<PlotLineOrBand>;
}
/**
 * (Highcharts, Highstock, Gantt) Text labels for the plot bands
 */
export interface NavigatorXAxisPlotLinesLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) Horizontal alignment of the label. Can be
     * one of "left", "center" or "right".
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Callback JavaScript function to format the
     * label. Useful properties like the value of plot line or the range of plot
     * band (`from` & `to` properties) can be found in `this.options` object.
     */
    formatter?: FormatterCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Rotation of the text label in degrees.
     * Defaults to 0 for horizontal plot lines and 90 for vertical lines.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-line-label` class.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) The text itself. A subset of HTML is
     * supported.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Gantt) The text alignment for the label. While
     * `align` determines where the texts anchor point is placed within the plot
     * band, `textAlign` determines how the text is aligned against its anchor
     * point. Possible values are "left", "center" and "right". Defaults to the
     * same as the `align` option.
     */
    textAlign?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Vertical alignment of the label relative
     * to the plot line. Can be one of "top", "middle" or "bottom".
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal position relative the
     * alignment. Default varies by orientation.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical position of the text baseline
     * relative to the alignment. Default varies by orientation.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An array of lines stretching across the plot
 * area, marking a specific value on one of the axes.
 *
 * In styled mode, the plot lines are styled by the `.highcharts-plot-line`
 * class in addition to the `className` option.
 */
export interface NavigatorXAxisPlotLinesOptions {
    /**
     * (Highstock) Flag to decide if plotLine should be rendered across all
     * panes.
     */
    acrossPanes?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A custom class name, in addition to the
     * default `highcharts-plot-line`, to apply to each individual line.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) The color of the line.
     */
    color?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) The dashing or dot style for the plot
     * line. For possible values see this overview.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highcharts, Highstock, Gantt) An object defining mouse events for the
     * plot line. Supported properties are `click`, `mouseover`, `mouseout`,
     * `mousemove`.
     */
    events?: NavigatorXAxisPlotLinesEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) An id used for identifying the plot line
     * in Axis.removePlotLine.
     */
    id?: string;
    /**
     * (Highcharts, Highstock, Gantt) Text labels for the plot bands
     */
    label?: NavigatorXAxisPlotLinesLabelOptions;
    /**
     * (Highcharts, Highstock, Gantt) The position of the line in axis units.
     */
    value?: number;
    /**
     * (Highcharts, Highstock, Gantt) The width or thickness of the plot line.
     */
    width?: number;
    /**
     * (Highcharts, Highstock, Gantt) The z index of the plot line within the
     * chart.
     */
    zIndex?: number;
}
/**
 * (Highstock, Gantt) The axis title, showing next to the axis line.
 */
export interface NavigatorXAxisTitleOptions {
    /**
     * (Highstock, Gantt) Alignment of the title relative to the axis values.
     * Possible values are "low", "middle" or "high".
     */
    align?: AxisTitleAlignValue;
    /**
     * (Highstock, Gantt) The pixel distance between the axis labels or line and
     * the title. Defaults to 0 for horizontal axes, 10 for vertical
     */
    margin?: number;
    /**
     * (Highstock, Gantt) The distance of the axis title from the axis line. By
     * default, this distance is computed from the offset width of the labels,
     * the labels' distance from the axis and the title's margin. However when
     * the offset option is set, it overrides all this.
     */
    offset?: number;
    /**
     * (Highcharts) Defines how the title is repositioned according to the 3D
     * chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     *
     * - `undefined`: Will use the config from `labels.position3d`
     */
    position3d?: ("chart"|"flap"|"offset"|"ortho"|null);
    /**
     * (Highcharts, Highstock, Gantt) Whether to reserve space for the title
     * when laying out the axis.
     */
    reserveSpace?: boolean;
    /**
     * (Highstock, Gantt) The rotation of the text in degrees. 0 is horizontal,
     * 270 is vertical reading from bottom to top.
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis title will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `title.position3d`.
     *
     * A `null` value will use the config from `labels.skew3d`.
     */
    skew3d?: (boolean|null);
    /**
     * (Highstock, Gantt) CSS styles for the title. If the title text is longer
     * than the axis length, it will wrap to multiple lines by default. This can
     * be customized by setting `textOverflow: 'ellipsis'`, by setting a
     * specific `width` or by setting `whiteSpace: 'nowrap'`.
     *
     * In styled mode, the stroke width is given in the `.highcharts-axis-title`
     * class.
     */
    style?: CSSObject;
    /**
     * (Highstock, Gantt) The actual text of the axis title. It can contain
     * basic HTML tags like `b`, `i` and `span` with style.
     */
    text?: (string|null);
    /**
     * (Highstock, Gantt) Alignment of the text, can be `"left"`, `"right"` or
     * `"center"`. Default alignment depends on the title.align:
     *
     * Horizontal axes:
     *
     * - for `align` = `"low"`, `textAlign` is set to `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"`, `textAlign` is set to `right`
     *
     * Vertical axes:
     *
     * - for `align` = `"low"` and `opposite` = `true`, `textAlign` is set to
     * `right`
     *
     * - for `align` = `"low"` and `opposite` = `false`, `textAlign` is set to
     * `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"` and `opposite` = `true` `textAlign` is set to
     * `left`
     *
     * - for `align` = `"high"` and `opposite` = `false` `textAlign` is set to
     * `right`
     */
    textAlign?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the axis
     * title.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal pixel offset of the title
     * position.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical pixel offset of the title
     * position.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An array defining breaks in the axis, the
 * sections defined will be left out and all the points shifted closer to each
 * other.
 */
export interface NavigatorYAxisBreaksOptions {
    /**
     * (Highcharts, Highstock, Gantt) A number indicating how much space should
     * be left between the start and the end of the break. The break size is
     * given in axis units, so for instance on a `datetime` axis, a break size
     * of 3600000 would indicate the equivalent of an hour.
     */
    breakSize?: number;
    /**
     * (Highcharts, Highstock, Gantt) The point where the break starts.
     */
    from?: number;
    /**
     * (Highcharts, Highstock, Gantt) Defines an interval after which the break
     * appears again. By default the breaks do not repeat.
     */
    repeat?: number;
    /**
     * (Highcharts, Highstock, Gantt) The point where the break ends.
     */
    to?: number;
}
/**
 * (Highstock, Gantt) Event handlers for the axis.
 */
export interface NavigatorYAxisEventsOptions {
    /**
     * (Highcharts, Gantt) An event fired after the breaks have rendered.
     */
    afterBreaks?: AxisEventCallbackFunction;
    /**
     * (Highstock, Gantt) As opposed to the `setExtremes` event, this event
     * fires after the final min and max values are computed and corrected for
     * `minRange`.
     *
     * Fires when the minimum and maximum is set for the axis, either by calling
     * the `.setExtremes()` method or by selecting an area in the chart. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in axis
     * values. The actual data extremes are found in `event.dataMin` and
     * `event.dataMax`.
     */
    afterSetExtremes?: AxisSetExtremesEventCallbackFunction;
    /**
     * (Highcharts, Gantt) An event fired when a break from this axis occurs on
     * a point.
     */
    pointBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highcharts, Highstock, Gantt) An event fired when a point falls inside a
     * break from this axis.
     */
    pointInBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highstock, Gantt) Fires when the minimum and maximum is set for the
     * axis, either by calling the `.setExtremes()` method or by selecting an
     * area in the chart. One parameter, `event`, is passed to the function,
     * containing common event information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in data
     * values. When an axis is zoomed all the way out from the "Reset zoom"
     * button, `event.min` and `event.max` are null, and the new extremes are
     * set based on `this.dataMin` and `this.dataMax`.
     */
    setExtremes?: AxisSetExtremesEventCallbackFunction;
}
/**
 * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
 */
export interface NavigatorYAxisGridOptions {
    /**
     * (Gantt) Set border color for the label grid lines.
     */
    borderColor?: ColorString;
    /**
     * (Gantt) Set border width of the label grid lines.
     */
    borderWidth?: number;
    /**
     * (Gantt) Set cell height for grid axis labels. By default this is
     * calculated from font size. This option only applies to horizontal axes.
     */
    cellHeight?: number;
    /**
     * (Gantt) Set specific options for each column (or row for horizontal axes)
     * in the grid. Each extra column/row is its own axis, and the axis options
     * can be set here.
     */
    columns?: Array<XAxisOptions>;
    /**
     * (Gantt) Enable grid on the axis labels. Defaults to true for Gantt
     * charts.
     */
    enabled?: boolean;
}
/**
 * (Gantt) Set options on specific levels in a tree grid axis. Takes precedence
 * over labels options.
 */
export interface NavigatorYAxisLabelsLevelsOptions {
    /**
     * (Gantt) Specify the level which the options within this object applies
     * to.
     */
    level?: number;
    style?: CSSObject;
}
/**
 * (Highstock, Gantt) The axis labels show the number or category for each tick.
 *
 * Since v8.0.0: Labels are animated in categorized x-axis with updating data if
 * `tickInterval` and `step` is set to 1.
 */
export interface NavigatorYAxisLabelsOptions {
    /**
     * (Highstock, Gantt) What part of the string the given position is anchored
     * to. Can be one of `"left"`, `"center"` or `"right"`. The exact position
     * also depends on the `labels.x` setting.
     *
     * Angular gauges and solid gauges defaults to `"center"`. Solid gauges with
     * two labels have additional option `"auto"` for automatic horizontal and
     * vertical alignment.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) For horizontal axes, the allowed degrees
     * of label rotation to prevent overlapping labels. If there is enough
     * space, labels are not rotated. As the chart gets narrower, it will start
     * rotating the labels -45 degrees, then remove every second label and try
     * again with rotations 0 and -45 etc. Set it to `false` to disable
     * rotation, which will cause the labels to word-wrap if possible.
     */
    autoRotation?: (false|Array<number>);
    /**
     * (Highcharts, Gantt) When each category width is more than this many
     * pixels, we don't apply auto rotation. Instead, we lay out the axis label
     * with word wrap. A lower limit makes sense when the label contains
     * multiple short words that don't extend the available horizontal space for
     * each label.
     */
    autoRotationLimit?: number;
    /**
     * (Highcharts) Angular gauges and solid gauges only. The label's pixel
     * distance from the perimeter of the plot area.
     *
     * Since v7.1.2: If it's a percentage string, it is interpreted the same as
     * series.radius, so label can be aligned under the gauge's shape.
     */
    distance?: (number|string);
    /**
     * (Highstock, Gantt) Enable or disable the axis labels.
     */
    enabled?: boolean;
    /**
     * (Highstock, Gantt) A format string for the axis label.
     */
    format?: string;
    /**
     * (Highstock, Gantt) Callback JavaScript function to format the label. The
     * value is given by `this.value`. Additional properties for `this` are
     * `axis`, `chart`, `isFirst` and `isLast`. The value of the default label
     * formatter can be retrieved by calling
     * `this.axis.defaultLabelFormatter.call(this)` within the function.
     *
     * Defaults to: (see online documentation for example)
     */
    formatter?: AxisLabelsFormatterCallbackFunction;
    /**
     * (Gantt) The number of pixels to indent the labels per level in a treegrid
     * axis.
     */
    indentation?: number;
    /**
     * (Gantt) Set options on specific levels in a tree grid axis. Takes
     * precedence over labels options.
     */
    levels?: Array<NavigatorYAxisLabelsLevelsOptions>;
    /**
     * (Highstock, Gantt) How to handle overflowing labels on horizontal axis.
     * If set to `"allow"`, it will not be aligned at all. By default it
     * `"justify"` labels inside the chart area. If there is room to move it, it
     * will be aligned to the edge, else it will be removed.
     */
    overflow?: OptionsOverflowValue;
    /**
     * (Highcharts, Gantt) The pixel padding for axis labels, to ensure white
     * space between them.
     */
    padding?: number;
    /**
     * (Highcharts) Defines how the labels are be repositioned according to the
     * 3D chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     */
    position3d?: OptionsPosition3dValue;
    /**
     * (Highcharts, Gantt) Whether to reserve space for the labels. By default,
     * space is reserved for the labels in these cases:
     *
     * * On all horizontal axes.
     *
     * * On vertical axes if `label.align` is `right` on a left-side axis or
     * `left` on a right-side axis.
     *
     * * On vertical axes if `label.align` is `center`.
     *
     * This can be turned off when for example the labels are rendered inside
     * the plot area instead of outside.
     */
    reserveSpace?: boolean;
    /**
     * (Highstock, Gantt) Rotation of the labels in degrees.
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis labels will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `labels.position3d`.
     */
    skew3d?: boolean;
    /**
     * (Highstock, Gantt) Horizontal axes only. The number of lines to spread
     * the labels over to make room or tighter labels.
     */
    staggerLines?: number;
    /**
     * (Highstock, Gantt) To show only every _n_'th label on the axis, set the
     * step to _n_. Setting the step to 2 shows every other label.
     *
     * By default, the step is calculated automatically to avoid overlap. To
     * prevent this, set it to 1\. This usually only happens on a category axis,
     * and is often a sign that you have chosen the wrong axis type.
     *
     * Read more at Axis docs => What axis should I use?
     */
    step?: number;
    /**
     * (Highstock, Gantt) CSS styles for the label. Use `whiteSpace: 'nowrap'`
     * to prevent wrapping of category labels. Use `textOverflow: 'none'` to
     * prevent ellipsis (dots).
     *
     * In styled mode, the labels are styled with the `.highcharts-axis-labels`
     * class.
     */
    style?: CSSObject;
    /**
     * (Gantt) The symbol for the collapse and expand icon in a treegrid.
     */
    symbol?: NavigatorYAxisLabelsSymbolOptions;
    /**
     * (Highstock, Gantt) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Highstock, Gantt) The x position offset of all labels relative to the
     * tick positions on the axis. Defaults to -15 for left axis, 15 for right
     * axis.
     */
    x?: number;
    /**
     * (Highstock, Gantt) The y position offset of all labels relative to the
     * tick positions on the axis. For polar and radial axis consider the use of
     * the distance option.
     */
    y?: number;
    /**
     * (Highstock, Gantt) The Z index for the axis labels.
     */
    zIndex?: number;
}
/**
 * (Gantt) The symbol for the collapse and expand icon in a treegrid.
 */
export interface NavigatorYAxisLabelsSymbolOptions {
    height?: number;
    padding?: number;
    /**
     * (Gantt) The symbol type. Points to a definition function in the
     * `Highcharts.Renderer.symbols` collection.
     */
    type?: SymbolKeyValue;
    width?: number;
    x?: number;
    y?: number;
}
/**
 * (Highstock, Gantt) Options for the navigator Y axis. Default series options
 * for the navigator yAxis are: (see online documentation for example)
 */
export interface NavigatorYAxisOptions {
    /**
     * (Highstock, Gantt) Accessibility options for an axis. Requires the
     * accessibility module.
     */
    accessibility?: AxisAccessibilityOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When using multiple axis, the ticks of two
     * or more opposite axes will automatically be aligned by adding ticks to
     * the axis or axes with the least ticks, as if `tickAmount` were specified.
     *
     * This can be prevented by setting `alignTicks` to false. If the grid lines
     * look messy, it's a good idea to hide them for the secondary axis by
     * setting `gridLineWidth` to 0.
     *
     * If `startOnTick` or `endOnTick` in an Axis options are set to false, then
     * the `alignTicks ` will be disabled for the Axis.
     *
     * Disabled for logarithmic axes.
     */
    alignTicks?: boolean;
    /**
     * (Highstock, Gantt) Whether to allow decimals in this axis' ticks. When
     * counting integers, like persons or hits on a web page, decimals should be
     * avoided in the labels.
     */
    allowDecimals?: boolean;
    /**
     * (Highstock, Gantt) When using an alternate grid color, a band is painted
     * across the plot area between every other grid line.
     */
    alternateGridColor?: ColorType;
    /**
     * (Highcharts) In a polar chart, this is the angle of the Y axis in
     * degrees, where 0 is up and 90 is right. The angle determines the position
     * of the axis line and the labels, though the coordinate system is
     * unaffected. Since v8.0.0 this option is also applicable for X axis
     * (inverted polar).
     */
    angle?: number;
    /**
     * (Highcharts, Highstock, Gantt) An array defining breaks in the axis, the
     * sections defined will be left out and all the points shifted closer to
     * each other.
     */
    breaks?: Array<NavigatorYAxisBreaksOptions>;
    /**
     * (Highcharts, Gantt) If categories are present for the xAxis, names are
     * used instead of numbers for that axis.
     *
     * Since Highcharts 3.0, categories can also be extracted by giving each
     * point a name and setting axis type to `category`. However, if you have
     * multiple series, best practice remains defining the `categories` array.
     *
     * Example: `categories: ['Apples', 'Bananas', 'Oranges']`
     */
    categories?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) The highest allowed value for
     * automatically computed axis extremes.
     */
    ceiling?: number;
    /**
     * (Highstock, Gantt) A class name that opens for styling the axis by CSS,
     * especially in Highcharts styled mode. The class name is applied to group
     * elements for the grid, axis elements and labels.
     */
    className?: string;
    /**
     * (Highstock, Gantt) Configure a crosshair that follows either the mouse
     * pointer or the hovered point.
     *
     * In styled mode, the crosshairs are styled in the `.highcharts-crosshair`,
     * `.highcharts-crosshair-thin` or `.highcharts-xaxis-category` classes.
     */
    crosshair?: (boolean|AxisCrosshairOptions);
    /**
     * (Highcharts, Highstock, Gantt) For a datetime axis, the scale will
     * automatically adjust to the appropriate unit. This member gives the
     * default string representations used for each unit. For intermediate
     * values, different units may be used, for example the `day` unit can be
     * used on midnight and `hour` unit be used for intermediate values on the
     * same axis.
     *
     * For an overview of the replacement codes, see dateFormat.
     *
     * Defaults to: (see online documentation for example)
     */
    dateTimeLabelFormats?: AxisDateTimeLabelFormatsOptions;
    /**
     * (Highstock, Gantt) Whether to force the axis to end on a tick. Use this
     * option with the `maxPadding` option to control the axis end.
     *
     * This option is always disabled, when panning type is either `y` or `xy`.
     */
    endOnTick?: boolean;
    /**
     * (Highstock, Gantt) Event handlers for the axis.
     */
    events?: NavigatorYAxisEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) The lowest allowed value for automatically
     * computed axis extremes.
     */
    floor?: number;
    /**
     * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
     */
    grid?: NavigatorYAxisGridOptions;
    /**
     * (Highstock, Gantt) Color of the grid lines extending the ticks across the
     * plot area.
     *
     * In styled mode, the stroke is given in the `.highcharts-grid-line` class.
     */
    gridLineColor?: ColorType;
    /**
     * (Highstock, Gantt) The dash or dot style of the grid lines. For possible
     * values, see this demonstration.
     */
    gridLineDashStyle?: DashStyleValue;
    /**
     * (Highcharts) Polar charts only. Whether the grid lines should draw as a
     * polygon with straight lines between categories, or as circles. Can be
     * either `circle` or `polygon`. Since v8.0.0 this option is also applicable
     * for X axis (inverted polar).
     */
    gridLineInterpolation?: OptionsGridLineInterpolationValue;
    /**
     * (Highstock, Gantt) The width of the grid lines extending the ticks across
     * the plot area.
     *
     * In styled mode, the stroke width is given in the `.highcharts-grid-line`
     * class.
     */
    gridLineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The Z index of the grid lines.
     */
    gridZIndex?: number;
    /**
     * (Highstock, Gantt) An id for the axis. This can be used after render time
     * to get a pointer to the axis object through `chart.get()`.
     */
    id?: string;
    /**
     * (Highstock, Gantt) The axis labels show the number or category for each
     * tick.
     *
     * Since v8.0.0: Labels are animated in categorized x-axis with updating
     * data if `tickInterval` and `step` is set to 1.
     */
    labels?: NavigatorYAxisLabelsOptions;
    /**
     * (Highcharts, Highstock) The left position as the horizontal axis. If it's
     * a number, it is interpreted as pixel position relative to the chart.
     *
     * Since Highcharts v5.0.13: If it's a percentage string, it is interpreted
     * as percentages of the plot width, offset from plot area left.
     */
    left?: (number|string);
    /**
     * (Highstock, Gantt) The color of the line marking the axis itself.
     *
     * In styled mode, the line stroke is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     */
    lineColor?: ColorType;
    /**
     * (Highstock, Gantt) The width of the line marking the axis itself.
     *
     * In styled mode, the stroke width is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     */
    lineWidth?: number;
    /**
     * (Highstock, Gantt) If there are multiple axes on the same side of the
     * chart, the pixel margin between the axes. Defaults to 0 on vertical axes,
     * 15 on horizontal axes.
     */
    margin?: number;
    /**
     * (Highstock, Gantt) The maximum value of the axis. If `null`, the max
     * value is automatically calculated.
     *
     * If the endOnTick option is true, the `max` value might be rounded up.
     *
     * If a tickAmount is set, the axis may be extended beyond the set max in
     * order to reach the given number of ticks. The same may happen in a chart
     * with multiple axes, determined by chart. alignTicks, where a `tickAmount`
     * is applied internally.
     */
    max?: (number|null);
    /**
     * (Highcharts) Solid gauge only. Unless stops are set, the color to
     * represent the maximum value of the Y axis.
     */
    maxColor?: ColorType;
    /**
     * (Highcharts, Highstock, Gantt) Padding of the max value relative to the
     * length of the axis. A padding of 0.05 will make a 100px axis 5px longer.
     * This is useful when you don't want the highest data value to appear on
     * the edge of the plot area. When the axis' `max` option is set or a max
     * extreme is set using `axis.setExtremes()`, the maxPadding will be
     * ignored.
     *
     * Also the `softThreshold` option takes precedence over `maxPadding`, so if
     * the data is tangent to the threshold, `maxPadding` may not apply unless
     * `softThreshold` is set to false.
     */
    maxPadding?: number;
    /**
     * (Highstock, Gantt) The minimum value of the axis. If `null` the min value
     * is automatically calculated.
     *
     * If the startOnTick option is true (default), the `min` value might be
     * rounded down.
     *
     * The automatically calculated minimum value is also affected by floor,
     * softMin, minPadding, minRange as well as series.threshold and
     * series.softThreshold.
     */
    min?: (number|null);
    /**
     * (Highcharts) Solid gauge only. Unless stops are set, the color to
     * represent the minimum value of the Y axis.
     */
    minColor?: ColorType;
    /**
     * (Highstock, Gantt) Color of the minor, secondary grid lines.
     *
     * In styled mode, the stroke width is given in the
     * `.highcharts-minor-grid-line` class.
     */
    minorGridLineColor?: ColorType;
    /**
     * (Highstock, Gantt) The dash or dot style of the minor grid lines. For
     * possible values, see this demonstration.
     */
    minorGridLineDashStyle?: DashStyleValue;
    /**
     * (Highstock, Gantt) Width of the minor, secondary grid lines.
     *
     * In styled mode, the stroke width is given in the `.highcharts-grid-line`
     * class.
     */
    minorGridLineWidth?: number;
    /**
     * (Highstock, Gantt) Color for the minor tick marks.
     */
    minorTickColor?: ColorType;
    /**
     * (Highstock, Gantt) Specific tick interval in axis units for the minor
     * ticks. On a linear axis, if `"auto"`, the minor tick interval is
     * calculated as a fifth of the tickInterval. If `null` or `undefined`,
     * minor ticks are not shown.
     *
     * On logarithmic axes, the unit is the power of the value. For example,
     * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
     * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
     * and 10, 10 and 100 etc.
     *
     * If user settings dictate minor ticks to become too dense, they don't make
     * sense, and will be ignored to prevent performance problems.
     */
    minorTickInterval?: (number|string|null);
    /**
     * (Highstock, Gantt) The pixel length of the minor tick marks.
     */
    minorTickLength?: number;
    /**
     * (Highstock, Gantt) The position of the minor tick marks relative to the
     * axis line. Can be one of `inside` and `outside`.
     */
    minorTickPosition?: OptionsMinorTickPositionValue;
    /**
     * (Highstock, Gantt) Enable or disable minor ticks. Unless
     * minorTickInterval is set, the tick interval is calculated as a fifth of
     * the `tickInterval`.
     *
     * On a logarithmic axis, minor ticks are laid out based on a best guess,
     * attempting to enter approximately 5 minor ticks between each major tick.
     *
     * Prior to v6.0.0, ticks were unabled in auto layout by setting
     * `minorTickInterval` to `"auto"`.
     */
    minorTicks?: boolean;
    /**
     * (Highstock, Gantt) The pixel width of the minor tick mark.
     */
    minorTickWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) Padding of the min value relative to the
     * length of the axis. A padding of 0.05 will make a 100px axis 5px longer.
     * This is useful when you don't want the lowest data value to appear on the
     * edge of the plot area. When the axis' `min` option is set or a max
     * extreme is set using `axis.setExtremes()`, the maxPadding will be
     * ignored.
     *
     * Also the `softThreshold` option takes precedence over `minPadding`, so if
     * the data is tangent to the threshold, `minPadding` may not apply unless
     * `softThreshold` is set to false.
     */
    minPadding?: number;
    /**
     * (Highstock, Gantt) The minimum tick interval allowed in axis values. For
     * example on zooming in on an axis with daily data, this can be used to
     * prevent the axis from showing hours. Defaults to the closest distance
     * between two points on the axis.
     */
    minTickInterval?: number;
    /**
     * (Highstock, Gantt) The distance in pixels from the plot area to the axis
     * line. A positive offset moves the axis with it's line, labels and ticks
     * away from the plot area. This is typically used when two or more axes are
     * displayed on the same side of the plot. With multiple axes the offset is
     * dynamically adjusted to avoid collision, this can be overridden by
     * setting offset explicitly.
     */
    offset?: number;
    /**
     * (Highstock, Highcharts, Gantt) Whether to display the axis on the
     * opposite side of the normal. The normal is on the left side for vertical
     * axes and bottom for horizontal, so the opposite sides will be right and
     * top respectively. This is typically used with dual or multiple axes.
     */
    opposite?: boolean;
    /**
     * (Highcharts) Refers to the index in the panes array. Used for circular
     * gauges and polar charts. When the option is not set then first pane will
     * be used.
     */
    pane?: number;
    /**
     * (Highcharts, Highstock, Gantt) An array of colored bands stretching
     * across the plot area marking an interval on the axis.
     *
     * In styled mode, the plot bands are styled by the `.highcharts-plot-band`
     * class in addition to the `className` option.
     */
    plotBands?: Array<NavigatorYAxisPlotBandsOptions>;
    /**
     * (Highcharts, Highstock, Gantt) An array of lines stretching across the
     * plot area, marking a specific value on one of the axes.
     *
     * In styled mode, the plot lines are styled by the `.highcharts-plot-line`
     * class in addition to the `className` option.
     */
    plotLines?: Array<NavigatorYAxisPlotLinesOptions>;
    /**
     * (Highstock, Gantt) Whether to reverse the axis so that the highest number
     * is closest to the origin.
     */
    reversed?: boolean;
    /**
     * (Highcharts, Highstock) If `true`, the first series in a stack will be
     * drawn on top in a positive, non-reversed Y axis. If `false`, the first
     * series is in the base of the stack.
     */
    reversedStacks?: boolean;
    /**
     * (Highstock, Gantt) Whether to show the first tick label.
     */
    showFirstLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to show the last tick label.
     * Defaults to `true` on cartesian charts, and `false` on polar charts.
     */
    showLastLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A soft maximum for the axis. If the series
     * data maximum is less than this, the axis will stay at this maximum, but
     * if the series data maximum is higher, the axis will flex to show all
     * data.
     *
     * **Note**: The series.softThreshold option takes precedence over this
     * option.
     */
    softMax?: number;
    /**
     * (Highcharts, Highstock, Gantt) A soft minimum for the axis. If the series
     * data minimum is greater than this, the axis will stay at this minimum,
     * but if the series data minimum is lower, the axis will flex to show all
     * data.
     *
     * **Note**: The series.softThreshold option takes precedence over this
     * option.
     */
    softMin?: number;
    /**
     * (Highcharts) The stack labels show the total value for each bar in a
     * stacked column or bar chart. The label will be placed on top of positive
     * columns and below negative columns. In case of an inverted column chart
     * or a bar chart the label is placed to the right of positive bars and to
     * the left of negative bars.
     */
    stackLabels?: NavigatorYAxisStackLabelsOptions;
    /**
     * (Highcharts, Highstock, Gantt) For datetime axes, this decides where to
     * put the tick between weeks. 0 = Sunday, 1 = Monday.
     */
    startOfWeek?: number;
    /**
     * (Highcharts, Highstock, Gantt) Whether to force the axis to start on a
     * tick. Use this option with the `maxPadding` option to control the axis
     * start.
     *
     * This option is always disabled, when panning type is either `y` or `xy`.
     */
    startOnTick?: boolean;
    /**
     * (Gantt) For vertical axes only. Setting the static scale ensures that
     * each tick unit is translated into a fixed pixel height. For example,
     * setting the static scale to 24 results in each Y axis category taking up
     * 24 pixels, and the height of the chart adjusts. Adding or removing items
     * will make the chart resize.
     */
    staticScale?: number;
    /**
     * (Highcharts) Solid gauge series only. Color stops for the solid gauge.
     * Use this in cases where a linear gradient between a `minColor` and
     * `maxColor` is not sufficient. The stops is an array of tuples, where the
     * first item is a float between 0 and 1 assigning the relative position in
     * the gradient, and the second item is the color.
     *
     * For solid gauges, the Y axis also inherits the concept of data classes
     * from the Highmaps color axis.
     */
    stops?: Array<[number, ColorType]>;
    /**
     * (Highcharts, Highstock, Gantt) The amount of ticks to draw on the axis.
     * This opens up for aligning the ticks of multiple charts or panes within a
     * chart. This option overrides the `tickPixelInterval` option.
     *
     * This option only has an effect on linear axes. Datetime, logarithmic or
     * category axes are not affected.
     */
    tickAmount?: number;
    /**
     * (Highstock, Gantt) Color for the main tick marks.
     *
     * In styled mode, the stroke is given in the `.highcharts-tick` class.
     */
    tickColor?: ColorType;
    /**
     * (Highstock, Gantt) The interval of the tick marks in axis units. When
     * `undefined`, the tick interval is computed to approximately follow the
     * tickPixelInterval on linear and datetime axes. On categorized axes, a
     * `undefined` tickInterval will default to 1, one category. Note that
     * datetime axes are based on milliseconds, so for example an interval of
     * one day is expressed as `24 * 3600 * 1000`.
     *
     * On logarithmic axes, the tickInterval is based on powers, so a
     * tickInterval of 1 means one tick on each of 0.1, 1, 10, 100 etc. A
     * tickInterval of 2 means a tick of 0.1, 10, 1000 etc. A tickInterval of
     * 0.2 puts a tick on 0.1, 0.2, 0.4, 0.6, 0.8, 1, 2, 4, 6, 8, 10, 20, 40
     * etc.
     *
     * If the tickInterval is too dense for labels to be drawn, Highcharts may
     * remove ticks.
     *
     * If the chart has multiple axes, the alignTicks option may interfere with
     * the `tickInterval` setting.
     */
    tickInterval?: number;
    /**
     * (Highstock, Gantt) The pixel length of the main tick marks.
     */
    tickLength?: number;
    /**
     * (Highcharts, Gantt) For categorized axes only. If `on` the tick mark is
     * placed in the center of the category, if `between` the tick mark is
     * placed between categories. The default is `between` if the `tickInterval`
     * is 1, else `on`.
     */
    tickmarkPlacement?: OptionsTickmarkPlacementValue;
    /**
     * (Highstock, Gantt) If tickInterval is `null` this option sets the
     * approximate pixel interval of the tick marks. Not applicable to
     * categorized axis.
     *
     * The tick interval is also influenced by the minTickInterval option, that,
     * by default prevents ticks from being denser than the data points.
     */
    tickPixelInterval?: number;
    /**
     * (Highstock, Gantt) The position of the major tick marks relative to the
     * axis line. Can be one of `inside` and `outside`.
     */
    tickPosition?: OptionsTickPositionValue;
    /**
     * (Highstock, Gantt) A callback function returning array defining where the
     * ticks are laid out on the axis. This overrides the default behaviour of
     * tickPixelInterval and tickInterval. The automatic tick positions are
     * accessible through `this.tickPositions` and can be modified by the
     * callback.
     */
    tickPositioner?: AxisTickPositionerCallbackFunction;
    /**
     * (Highstock, Gantt) An array defining where the ticks are laid out on the
     * axis. This overrides the default behaviour of tickPixelInterval and
     * tickInterval.
     */
    tickPositions?: Array<number>;
    /**
     * (Highcharts, Highstock, Gantt) The pixel width of the major tick marks.
     */
    tickWidth?: number;
    /**
     * (Highstock, Gantt) The axis title, showing next to the axis line.
     */
    title?: NavigatorYAxisTitleOptions;
    /**
     * (Highcharts) Parallel coordinates only. Format that will be used for
     * point.y and available in tooltip.pointFormat as `{point.formattedValue}`.
     * If not set, `{point.formattedValue}` will use other options, in this
     * order:
     *
     * 1. yAxis.labels.format will be used if set
     *
     * 2. If yAxis is a category, then category name will be displayed
     *
     * 3. If yAxis is a datetime, then value will use the same format as yAxis
     * labels
     *
     * 4. If yAxis is linear/logarithmic type, then simple value will be used
     */
    tooltipValueFormat?: string;
    /**
     * (Highcharts, Gantt) The type of axis. Can be one of `linear`,
     * `logarithmic`, `datetime`, `category` or `treegrid`. Defaults to
     * `treegrid` for Gantt charts, `linear` for other chart types.
     *
     * In a datetime axis, the numbers are given in milliseconds, and tick marks
     * are placed on appropriate values, like full hours or days. In a category
     * or treegrid axis, the point names of the chart's series are used for
     * categories, if a categories array is not defined.
     */
    type?: AxisTypeValue;
    /**
     * (Highcharts, Gantt) Applies only when the axis `type` is `category`. When
     * `uniqueNames` is true, points are placed on the X axis according to their
     * names. If the same point name is repeated in the same or another series,
     * the point is placed on the same X position as other points of the same
     * name. When `uniqueNames` is false, the points are laid out in increasing
     * X positions regardless of their names, and the X axis category will take
     * the name of the last point in each position.
     */
    uniqueNames?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether axis, including axis title, line,
     * ticks and labels, should be visible.
     */
    visible?: boolean;
    /**
     * (Highcharts, Highstock) The width as the horizontal axis. If it's a
     * number, it is interpreted as pixels.
     *
     * Since Highcharts v5.0.13: If it's a percentage string, it is interpreted
     * as percentages of the total plot width.
     */
    width?: (number|string);
    /**
     * (Highstock, Gantt) Whether to zoom axis. If `chart.zoomType` is set, the
     * option allows to disable zooming on an individual axis.
     */
    zoomEnabled?: boolean;
}
/**
 * (Highcharts, Highstock, Gantt) An object defining mouse events for the plot
 * band. Supported properties are `click`, `mouseover`, `mouseout`, `mousemove`.
 */
export interface NavigatorYAxisPlotBandsEventsOptions {
    /**
     * (Highcharts, Highstock, Gantt) Click event on a plot band.
     */
    click?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse move event on a plot band.
     */
    mousemove?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse out event on the corner of a plot
     * band.
     */
    mouseout?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse over event on a plot band.
     */
    mouseover?: EventCallbackFunction<PlotLineOrBand>;
}
/**
 * (Highcharts, Highstock, Gantt) Text labels for the plot bands
 */
export interface NavigatorYAxisPlotBandsLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) Horizontal alignment of the label. Can be
     * one of "left", "center" or "right".
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Rotation of the text label in degrees .
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-band-label` class.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) The string text itself. A subset of HTML
     * is supported.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Gantt) The text alignment for the label. While
     * `align` determines where the texts anchor point is placed within the plot
     * band, `textAlign` determines how the text is aligned against its anchor
     * point. Possible values are "left", "center" and "right". Defaults to the
     * same as the `align` option.
     */
    textAlign?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Vertical alignment of the label relative
     * to the plot band. Can be one of "top", "middle" or "bottom".
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal position relative the
     * alignment. Default varies by orientation.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical position of the text baseline
     * relative to the alignment. Default varies by orientation.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An array of colored bands stretching across
 * the plot area marking an interval on the axis.
 *
 * In styled mode, the plot bands are styled by the `.highcharts-plot-band`
 * class in addition to the `className` option.
 */
export interface NavigatorYAxisPlotBandsOptions {
    /**
     * (Highstock) Flag to decide if plotBand should be rendered across all
     * panes.
     */
    acrossPanes?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Border color for the plot band. Also
     * requires `borderWidth` to be set.
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) Border width for the plot band. Also
     * requires `borderColor` to be set.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) A custom class name, in addition to the
     * default `highcharts-plot-band`, to apply to each individual band.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) The color of the plot band.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Gantt) An object defining mouse events for the
     * plot band. Supported properties are `click`, `mouseover`, `mouseout`,
     * `mousemove`.
     */
    events?: NavigatorYAxisPlotBandsEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) The start position of the plot band in
     * axis units.
     */
    from?: number;
    /**
     * (Highcharts, Highstock, Gantt) An id used for identifying the plot band
     * in Axis.removePlotBand.
     */
    id?: string;
    /**
     * (Highcharts) In a gauge chart, this option determines the inner radius of
     * the plot band that stretches along the perimeter. It can be given as a
     * percentage string, like `"100%"`, or as a pixel number, like `100`. By
     * default, the inner radius is controlled by the thickness option.
     */
    innerRadius?: (number|string);
    /**
     * (Highcharts, Highstock, Gantt) Text labels for the plot bands
     */
    label?: NavigatorYAxisPlotBandsLabelOptions;
    /**
     * (Highcharts) In a gauge chart, this option determines the outer radius of
     * the plot band that stretches along the perimeter. It can be given as a
     * percentage string, like `"100%"`, or as a pixel number, like `100`.
     */
    outerRadius?: (number|string);
    /**
     * (Highcharts) In a gauge chart, this option sets the width of the plot
     * band stretching along the perimeter. It can be given as a percentage
     * string, like `"10%"`, or as a pixel number, like `10`. The default value
     * 10 is the same as the default tickLength, thus making the plot band act
     * as a background for the tick markers.
     */
    thickness?: (number|string);
    /**
     * (Highcharts, Highstock, Gantt) The end position of the plot band in axis
     * units.
     */
    to?: number;
    /**
     * (Highcharts, Highstock, Gantt) The z index of the plot band within the
     * chart, relative to other elements. Using the same z index as another
     * element may give unpredictable results, as the last rendered element will
     * be on top. Values from 0 to 20 make sense.
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An object defining mouse events for the plot
 * line. Supported properties are `click`, `mouseover`, `mouseout`, `mousemove`.
 */
export interface NavigatorYAxisPlotLinesEventsOptions {
    /**
     * (Highcharts, Highstock, Gantt) Click event on a plot band.
     */
    click?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse move event on a plot band.
     */
    mousemove?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse out event on the corner of a plot
     * band.
     */
    mouseout?: EventCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Mouse over event on a plot band.
     */
    mouseover?: EventCallbackFunction<PlotLineOrBand>;
}
/**
 * (Highcharts, Highstock, Gantt) Text labels for the plot bands
 */
export interface NavigatorYAxisPlotLinesLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) Horizontal alignment of the label. Can be
     * one of "left", "center" or "right".
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Callback JavaScript function to format the
     * label. Useful properties like the value of plot line or the range of plot
     * band (`from` & `to` properties) can be found in `this.options` object.
     */
    formatter?: FormatterCallbackFunction<PlotLineOrBand>;
    /**
     * (Highcharts, Highstock, Gantt) Rotation of the text label in degrees.
     * Defaults to 0 for horizontal plot lines and 90 for vertical lines.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-line-label` class.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) The text itself. A subset of HTML is
     * supported.
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Gantt) The text alignment for the label. While
     * `align` determines where the texts anchor point is placed within the plot
     * band, `textAlign` determines how the text is aligned against its anchor
     * point. Possible values are "left", "center" and "right". Defaults to the
     * same as the `align` option.
     */
    textAlign?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Vertical alignment of the label relative
     * to the plot line. Can be one of "top", "middle" or "bottom".
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal position relative the
     * alignment. Default varies by orientation.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical position of the text baseline
     * relative to the alignment. Default varies by orientation.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An array of lines stretching across the plot
 * area, marking a specific value on one of the axes.
 *
 * In styled mode, the plot lines are styled by the `.highcharts-plot-line`
 * class in addition to the `className` option.
 */
export interface NavigatorYAxisPlotLinesOptions {
    /**
     * (Highstock) Flag to decide if plotLine should be rendered across all
     * panes.
     */
    acrossPanes?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A custom class name, in addition to the
     * default `highcharts-plot-line`, to apply to each individual line.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) The color of the line.
     */
    color?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) The dashing or dot style for the plot
     * line. For possible values see this overview.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highcharts, Highstock, Gantt) An object defining mouse events for the
     * plot line. Supported properties are `click`, `mouseover`, `mouseout`,
     * `mousemove`.
     */
    events?: NavigatorYAxisPlotLinesEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) An id used for identifying the plot line
     * in Axis.removePlotLine.
     */
    id?: string;
    /**
     * (Highcharts, Highstock, Gantt) Text labels for the plot bands
     */
    label?: NavigatorYAxisPlotLinesLabelOptions;
    /**
     * (Highcharts, Highstock, Gantt) The position of the line in axis units.
     */
    value?: number;
    /**
     * (Highcharts, Highstock, Gantt) The width or thickness of the plot line.
     */
    width?: number;
    /**
     * (Highcharts, Highstock, Gantt) The z index of the plot line within the
     * chart.
     */
    zIndex?: number;
}
/**
 * (Highcharts) Enable or disable the initial animation when a series is
 * displayed for the `stackLabels`. The animation can also be set as a
 * configuration object. Please note that this option only applies to the
 * initial animation. For other animations, see chart.animation and the
 * animation parameter under the API methods. The following properties are
 * supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface NavigatorYAxisStackLabelsAnimationOptions {
    /**
     * (Highcharts) The animation delay time in milliseconds. Set to `0` renders
     * stackLabel immediately. As `undefined` inherits defer time from the
     * series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts) The stack labels show the total value for each bar in a stacked
 * column or bar chart. The label will be placed on top of positive columns and
 * below negative columns. In case of an inverted column chart or a bar chart
 * the label is placed to the right of positive bars and to the left of negative
 * bars.
 */
export interface NavigatorYAxisStackLabelsOptions {
    /**
     * (Highcharts) Defines the horizontal alignment of the stack total label.
     * Can be one of `"left"`, `"center"` or `"right"`. The default value is
     * calculated at runtime and depends on orientation and whether the stack is
     * positive or negative.
     */
    align?: AlignValue;
    /**
     * (Highcharts) Allow the stack labels to overlap.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts) Enable or disable the initial animation when a series is
     * displayed for the `stackLabels`. The animation can also be set as a
     * configuration object. Please note that this option only applies to the
     * initial animation. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|NavigatorYAxisStackLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts) The background color or gradient for the stack label.
     */
    backgroundColor?: ColorType;
    /**
     * (Highcharts) The border color for the stack label. Defaults to
     * `undefined`.
     */
    borderColor?: ColorType;
    /**
     * (Highcharts) The border radius in pixels for the stack label.
     */
    borderRadius?: number;
    /**
     * (Highcharts) The border width in pixels for the stack label.
     */
    borderWidth?: number;
    /**
     * (Highcharts) Whether to hide stack labels that are outside the plot area.
     * By default, the stack label is moved inside the plot area according to
     * the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts) Enable or disable the stack total labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) A format string for the data label. Available
     * variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts) Callback JavaScript function to format the label. The value
     * is given by `this.total`.
     */
    formatter?: FormatterCallbackFunction<StackItemObject>;
    /**
     * (Highcharts) How to handle stack total labels that flow outside the plot
     * area. The default is set to `"justify"`, which aligns them inside the
     * plot area. For columns and bars, this means it will be moved inside the
     * bar. To display stack labels outside the plot area, set `crop` to `false`
     * and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts) Rotation of the labels in degrees.
     */
    rotation?: number;
    /**
     * (Highcharts) CSS styles for the label.
     *
     * In styled mode, the styles are set in the `.highcharts-stack-label`
     * class.
     */
    style?: CSSObject;
    /**
     * (Highcharts) The text alignment for the label. While `align` determines
     * where the texts anchor point is placed with regards to the stack,
     * `textAlign` determines how the text is aligned against its anchor point.
     * Possible values are `"left"`, `"center"` and `"right"`. The default value
     * is calculated at runtime and depends on orientation and whether the stack
     * is positive or negative.
     */
    textAlign?: AlignValue;
    /**
     * (Highcharts, Highstock) Whether to use HTML to render the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts) Defines the vertical alignment of the stack total label. Can
     * be one of `"top"`, `"middle"` or `"bottom"`. The default value is
     * calculated at runtime and depends on orientation and whether the stack is
     * positive or negative.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts) The x position offset of the label relative to the left of
     * the stacked bar. The default value is calculated at runtime and depends
     * on orientation and whether the stack is positive or negative.
     */
    x?: number;
    /**
     * (Highcharts) The y position offset of the label relative to the tick
     * position on the axis. The default value is calculated at runtime and
     * depends on orientation and whether the stack is positive or negative.
     */
    y?: number;
}
/**
 * (Highstock, Gantt) The axis title, showing next to the axis line.
 */
export interface NavigatorYAxisTitleOptions {
    /**
     * (Highstock, Gantt) Alignment of the title relative to the axis values.
     * Possible values are "low", "middle" or "high".
     */
    align?: AxisTitleAlignValue;
    /**
     * (Highstock, Gantt) The pixel distance between the axis labels and the
     * title. Positive values are outside the axis line, negative are inside.
     */
    margin?: number;
    /**
     * (Highstock, Gantt) The distance of the axis title from the axis line. By
     * default, this distance is computed from the offset width of the labels,
     * the labels' distance from the axis and the title's margin. However when
     * the offset option is set, it overrides all this.
     */
    offset?: number;
    /**
     * (Highcharts) Defines how the title is repositioned according to the 3D
     * chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     *
     * - `undefined`: Will use the config from `labels.position3d`
     */
    position3d?: ("chart"|"flap"|"offset"|"ortho"|null);
    /**
     * (Highcharts, Highstock, Gantt) Whether to reserve space for the title
     * when laying out the axis.
     */
    reserveSpace?: boolean;
    /**
     * (Highstock, Gantt) The rotation of the text in degrees. 0 is horizontal,
     * 270 is vertical reading from bottom to top.
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis title will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `title.position3d`.
     *
     * A `null` value will use the config from `labels.skew3d`.
     */
    skew3d?: (boolean|null);
    /**
     * (Highstock, Gantt) CSS styles for the title. If the title text is longer
     * than the axis length, it will wrap to multiple lines by default. This can
     * be customized by setting `textOverflow: 'ellipsis'`, by setting a
     * specific `width` or by setting `whiteSpace: 'nowrap'`.
     *
     * In styled mode, the stroke width is given in the `.highcharts-axis-title`
     * class.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) The actual text of the axis title.
     * Horizontal texts can contain HTML, but rotated texts are painted using
     * vector techniques and must be clean text. The Y axis title is disabled by
     * setting the `text` option to `undefined`.
     */
    text?: (string|null);
    /**
     * (Highstock, Gantt) Alignment of the text, can be `"left"`, `"right"` or
     * `"center"`. Default alignment depends on the title.align:
     *
     * Horizontal axes:
     *
     * - for `align` = `"low"`, `textAlign` is set to `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"`, `textAlign` is set to `right`
     *
     * Vertical axes:
     *
     * - for `align` = `"low"` and `opposite` = `true`, `textAlign` is set to
     * `right`
     *
     * - for `align` = `"low"` and `opposite` = `false`, `textAlign` is set to
     * `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"` and `opposite` = `true` `textAlign` is set to
     * `left`
     *
     * - for `align` = `"high"` and `opposite` = `false` `textAlign` is set to
     * `right`
     */
    textAlign?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the axis
     * title.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal pixel offset of the title
     * position.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical pixel offset of the title
     * position.
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) Options for displaying a message like "No data
 * to display". This feature requires the file no-data-to-display.js to be
 * loaded in the page. The actual text to display is set in the lang.noData
 * option.
 */
export interface NoDataOptions {
    /**
     * (Highcharts, Highstock, Gantt) An object of additional SVG attributes for
     * the no-data label.
     */
    attr?: SVGAttributes;
    /**
     * (Highcharts, Highstock, Gantt) The position of the no-data label,
     * relative to the plot area.
     */
    position?: (AlignObject|NoDataPositionOptions);
    /**
     * (Highcharts, Highstock, Gantt) CSS styles for the no-data label.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) Whether to insert the label as HTML, or as
     * pseudo-HTML rendered with SVG.
     */
    useHTML?: boolean;
}
/**
 * (Highcharts, Highstock, Gantt) The position of the no-data label, relative to
 * the plot area.
 */
export interface NoDataPositionOptions {
    /**
     * (Highcharts, Highstock, Gantt) Horizontal alignment of the label.
     */
    align?: AlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Vertical alignment of the label.
     */
    verticalAlign?: VerticalAlignValue;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal offset of the label, in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical offset of the label, in pixels.
     */
    y?: number;
}
/**
 * An object containing `left` and `top` properties for the position in the
 * page.
 */
export interface OffsetObject {
    /**
     * Left distance to the page border.
     */
    left: number;
    /**
     * Top distance to the page border.
     */
    top: number;
}
export interface Options {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for configuring
     * accessibility for the chart. Requires the accessibility module to be
     * loaded. For a description of the module and information on its features,
     * see Highcharts Accessibility.
     */
    accessibility?: AccessibilityOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A basic type of an annotation.
     * It allows to add custom labels or shapes. The items can be tied to
     * points, axis coordinates or chart pixel coordinates.
     */
    annotations?: Array<AnnotationsOptions>;
    boost?: BoostOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The chart's caption, which will
     * render below the chart and will be part of exported charts. The caption
     * can be updated after chart initialization through the `Chart.update` or
     * `Chart.caption.update` methods.
     */
    caption?: CaptionOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) General options for the chart.
     */
    chart?: ChartOptions;
    /**
     * (Highcharts, Highstock, Highmaps) A color axis for series. Visually, the
     * color axis will appear as a gradient or as separate items inside the
     * legend, depending on whether the axis is scalar or based on data classes.
     *
     * For supported color formats, see the docs article about colors.
     *
     * A scalar color axis is represented by a gradient. The colors either range
     * between the minColor and the maxColor, or for more fine grained control
     * the colors can be defined in stops. Often times, the color axis needs to
     * be adjusted to get the right color spread for the data. In addition to
     * stops, consider using a logarithmic axis type, or setting min and max to
     * avoid the colors being determined by outliers.
     *
     * When dataClasses are used, the ranges are subdivided into separate
     * classes like categories based on their values. This can be used for
     * ranges between two values, but also for a true category. However, when
     * your data is categorized, it may be as convenient to add each category to
     * a separate series.
     *
     * Color axis does not work with: `sankey`, `sunburst`, `dependencywheel`,
     * `networkgraph`, `wordcloud`, `venn`, `gauge` and `solidgauge` series
     * types.
     *
     * Since v7.2.0 `colorAxis` can also be an array of options objects.
     *
     * See the Axis object for programmatic access to the axis.
     */
    colorAxis?: (ColorAxisOptions|Array<ColorAxisOptions>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array containing the default
     * colors for the chart's series. When all colors are used, new colors are
     * pulled from the start again.
     *
     * Default colors can also be set on a series or series.type basis, see
     * column.colors, pie.colors.
     *
     * In styled mode, the colors option doesn't exist. Instead, colors are
     * defined in CSS and applied either through series or point class names, or
     * through the chart.colorCount option.
     *
     * ### Legacy
     *
     * In Highcharts 3.x, the default colors were: (see online documentation for
     * example)
     *
     * In Highcharts 2.x, the default colors were: (see online documentation for
     * example)
     */
    colors?: Array<ColorString>;
    /**
     * (Gantt) The Pathfinder module allows you to define connections between
     * any two points, represented as lines - optionally with markers for the
     * start and/or end points. Multiple algorithms are available for
     * calculating how the connecting lines are drawn.
     *
     * Connector functionality requires Highcharts Gantt to be loaded. In Gantt
     * charts, the connectors are used to draw dependencies between tasks.
     */
    connectors?: ConnectorsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Highchart by default puts a
     * credits label in the lower right corner of the chart. This can be changed
     * using these options.
     */
    credits?: CreditsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Data module provides a
     * simplified interface for adding data to a chart from sources like CVS,
     * HTML tables or grid views. See also the tutorial article on the Data
     * module.
     *
     * It requires the `modules/data.js` file to be loaded.
     *
     * Please note that the default way of adding data in Highcharts, without
     * the need of a module, is through the series._type_.data option.
     */
    data?: DataOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for configuring markers
     * for annotations.
     *
     * An example of the arrow marker: (see online documentation for example)
     */
    defs?: (DefsOptions|Dictionary<SVGDefinitionObject>);
    /**
     * (Highcharts, Highmaps) Options for drill down, the concept of inspecting
     * increasingly high resolution data through clicking on chart items like
     * columns or pie slices.
     *
     * The drilldown feature requires the drilldown.js file to be loaded, found
     * in the modules directory of the download package, or online at
     * code.highcharts.com/modules/drilldown.js.
     */
    drilldown?: DrilldownOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the exporting
     * module. For an overview on the matter, see the docs.
     */
    exporting?: ExportingOptions;
    global?: GlobalOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Language object. The language
     * object is global and it can't be set on each chart initialization.
     * Instead, use `Highcharts.setOptions` to set it before any chart is
     * initialized. (see online documentation for example)
     */
    lang?: LangOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The legend is a box containing a
     * symbol and name for each series item or point item in the chart. Each
     * series (or points in case of pie charts) is represented by a symbol and
     * its name in the legend.
     *
     * It is possible to override the symbol creator function and create custom
     * legend symbols.
     */
    legend?: LegendOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The loading options control the
     * appearance of the loading screen that covers the plot area on chart
     * operations. This screen only appears after an explicit call to
     * `chart.showLoading()`. It is a utility for developers to communicate to
     * the end user that something is going on, for example while retrieving new
     * data via an XHR connection. The "Loading..." text itself is not part of
     * this configuration object, but part of the `lang` object.
     */
    loading?: LoadingOptions;
    mapNavigation?: MapNavigationOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A collection of options for
     * buttons and menus appearing in the exporting module.
     */
    navigation?: NavigationOptions;
    /**
     * (Highstock, Gantt) The navigator is a small series below the main series,
     * displaying a view of the entire data set. It provides tools to zoom in
     * and out on parts of the data as well as panning across the dataset.
     */
    navigator?: NavigatorOptions;
    /**
     * (Highcharts, Highstock, Gantt) Options for displaying a message like "No
     * data to display". This feature requires the file no-data-to-display.js to
     * be loaded in the page. The actual text to display is set in the
     * lang.noData option.
     */
    noData?: NoDataOptions;
    /**
     * (Highcharts) The pane serves as a container for axes and backgrounds for
     * circular gauges and polar charts.
     */
    pane?: PaneOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The plotOptions is a wrapper
     * object for config objects for each series type. The config objects for
     * each series can also be overridden for each series item as given in the
     * series array.
     *
     * Configuration options for the series are given in three levels. Options
     * for all series in a chart are given in the plotOptions.series object.
     * Then options for all series of a specific type are given in the
     * plotOptions of that type, for example `plotOptions.line`. Next, options
     * for one single series are given in the series array.
     */
    plotOptions?: PlotOptions;
    /**
     * (Highstock, Gantt) The range selector is a tool for selecting ranges to
     * display within the chart. It provides buttons to select preconfigured
     * ranges in the chart, like 1 day, 1 week, 1 month etc. It also provides
     * input boxes where min and max dates can be manually input.
     */
    rangeSelector?: RangeSelectorOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Allows setting a set of rules to
     * apply for different screen or chart sizes. Each rule specifies additional
     * chart options.
     */
    responsive?: ResponsiveOptions;
    /**
     * (Highstock, Gantt) The scrollbar is a means of panning over the X axis of
     * a stock chart. Scrollbars can also be applied to other types of axes.
     *
     * Another approach to scrollable charts is the chart.scrollablePlotArea
     * option that is especially suitable for simpler cartesian charts on
     * mobile.
     *
     * In styled mode, all the presentational options for the scrollbar are
     * replaced by the classes `.highcharts-scrollbar-thumb`,
     * `.highcharts-scrollbar-arrow`, `.highcharts-scrollbar-button`,
     * `.highcharts-scrollbar-rifles` and `.highcharts-scrollbar-track`.
     */
    scrollbar?: ScrollbarOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Series options for specific data
     * and the data itself. In TypeScript you have to cast the series options to
     * specific series types, to get all possible options for a series.
     */
    series?: Array<SeriesOptionsType>;
    /**
     * (Highstock) Configure the stockTools gui strings in the chart. Requires
     * the [stockTools module]() to be loaded. For a description of the module
     * and information on its features, see [Highcharts StockTools]().
     */
    stockTools?: StockToolsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The chart's subtitle. This can
     * be used both to display a subtitle below the main title, and to display
     * random text anywhere in the chart. The subtitle can be updated after
     * chart initialization through the `Chart.setTitle` method.
     */
    subtitle?: SubtitleOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Time options that can apply
     * globally or to individual charts. These settings affect how `datetime`
     * axes are laid out, how tooltips are formatted, how series
     * pointIntervalUnit works and how the Highstock range selector handles
     * time.
     *
     * The common use case is that all charts in the same Highcharts object
     * share the same time settings, in which case the global settings are set
     * using `setOptions`. (see online documentation for example)
     *
     * Since v6.0.5, the time options were moved from the `global` obect to the
     * `time` object, and time options can be set on each individual chart.
     */
    time?: TimeOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The chart's main title.
     */
    title?: TitleOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the tooltip that
     * appears when the user hovers over a series or point.
     */
    tooltip?: TooltipOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The X axis or category axis.
     * Normally this is the horizontal axis, though if the chart is inverted
     * this is the vertical axis. In case of multiple axes, the xAxis node is an
     * array of configuration objects.
     *
     * See the Axis class for programmatic access to the axis.
     */
    xAxis?: (XAxisOptions|Array<XAxisOptions>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Y axis or value axis.
     * Normally this is the vertical axis, though if the chart is inverted this
     * is the horizontal axis. In case of multiple axes, the yAxis node is an
     * array of configuration objects.
     *
     * See the Axis object for programmatic access to the axis.
     */
    yAxis?: (YAxisOptions|Array<YAxisOptions>);
    /**
     * (Highcharts) The Z axis or depth axis for 3D plots.
     *
     * See the Axis class for programmatic access to the axis.
     */
    zAxis?: (ZAxisOptions|Array<ZAxisOptions>);
}
/**
 * (Highcharts) An array of background items for the pane.
 */
export interface PaneBackgroundOptions {
    /**
     * (Highcharts) The background color or gradient for the pane.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The pane background border color.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The pixel border width of the pane background.
     */
    borderWidth?: number;
    /**
     * (Highcharts) The class name for this background.
     */
    className?: string;
    /**
     * (Highcharts) The inner radius of the pane background. Can be either
     * numeric (pixels) or a percentage string.
     */
    innerRadius?: (number|string);
    /**
     * (Highcharts) The outer radius of the circular pane background. Can be
     * either numeric (pixels) or a percentage string.
     */
    outerRadius?: (number|string);
    /**
     * (Highcharts) The shape of the pane background. When `solid`, the
     * background is circular. When `arc`, the background extends only from the
     * min to the max of the value axis.
     */
    shape?: PaneBackgroundShapeValue;
}
/**
 * (Highcharts) The pane serves as a container for axes and backgrounds for
 * circular gauges and polar charts.
 */
export interface PaneOptions {
    /**
     * (Highcharts) An array of background items for the pane.
     */
    background?: Array<PaneBackgroundOptions>;
    /**
     * (Highcharts) The center of a polar chart or angular gauge, given as an
     * array of [x, y] positions. Positions can be given as integers that
     * transform to pixels, or as percentages of the plot area size.
     */
    center?: Array<(string|number)>;
    /**
     * (Highcharts) The end angle of the polar X axis or gauge value axis, given
     * in degrees where 0 is north. Defaults to startAngle
     *
     * + 360.
     */
    endAngle?: number;
    /**
     * (Highcharts) The inner size of the pane, either as a number defining
     * pixels, or a percentage defining a percentage of the pane's size.
     */
    innerSize?: (number|string);
    /**
     * (Highcharts) The size of the pane, either as a number defining pixels, or
     * a percentage defining a percentage of the available plot area (the
     * smallest of the plot height or plot width).
     */
    size?: (number|string);
    /**
     * (Highcharts) The start angle of the polar X axis or gauge axis, given in
     * degrees where 0 is north. Defaults to 0.
     */
    startAngle?: number;
}
/**
 * Holds a pattern definition.
 */
export interface PatternObject {
    /**
     * Animation options for the image pattern loading.
     */
    animation?: (boolean|Partial<AnimationOptionsObject>);
    /**
     * Pattern options
     */
    pattern: PatternOptionsObject;
    /**
     * Optionally an index referencing which pattern to use. Highcharts adds 10
     * default patterns to the `Highcharts.patterns` array. Additional pattern
     * definitions can be pushed to this array if desired. This option is an
     * index into this array.
     */
    patternIndex?: number;
}
/**
 * Pattern options
 */
export interface PatternOptionsObject {
    /**
     * For automatically calculated width and height on images, it is possible
     * to set an aspect ratio. The image will be zoomed to fill the bounding
     * box, maintaining the aspect ratio defined.
     */
    aspectRatio: number;
    /**
     * Background color for the pattern if a `path` is set (not images).
     */
    backgroundColor: ColorString;
    /**
     * Pattern color, used as default path stroke.
     */
    color: ColorString;
    /**
     * Analogous to pattern.width.
     */
    height: number;
    /**
     * ID to assign to the pattern. This is automatically computed if not added,
     * and identical patterns are reused. To refer to an existing pattern for a
     * Highcharts color, use `color: "url(#pattern-id)"`.
     */
    id?: string;
    /**
     * URL to an image to use as the pattern.
     */
    image: string;
    /**
     * Opacity of the pattern as a float value from 0 to 1.
     */
    opacity: number;
    /**
     * Either an SVG path as string, or an object. As an object, supply the path
     * string in the `path.d` property. Other supported properties are standard
     * SVG attributes like `path.stroke` and `path.fill`. If a path is supplied
     * for the pattern, the `image` property is ignored.
     */
    path: (string|SVGAttributes);
    /**
     * SVG `patternTransform` to apply to the entire pattern.
     */
    patternTransform: string;
    /**
     * Width of the pattern. For images this is automatically set to the width
     * of the element bounding box if not supplied. For non-image patterns the
     * default is 32px. Note that automatic resizing of image patterns to fill a
     * bounding box dynamically is only supported for patterns with an
     * automatically calculated ID.
     */
    width: number;
    /**
     * Horizontal offset of the pattern. Defaults to 0.
     */
    x?: number;
    /**
     * Vertical offset of the pattern. Defaults to 0.
     */
    y?: number;
}
/**
 * (Highstock) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotAbandsAnimationOptions {
    defer?: number;
}
export interface PlotAbandsBottomLineOptions {
    styles?: PlotAbandsBottomLineStylesOptions;
}
export interface PlotAbandsBottomLineStylesOptions {
    /**
     * (Highstock) Pixel width of the line.
     */
    lineWidth?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotAbandsDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data labels,
 * appearing next to each data point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 */
export interface PlotAbandsDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|PlotAbandsDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotAbandsDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highstock) Acceleration bands (ABANDS). This series requires the `linkedTo`
 * option to be set and should be loaded after the
 * `stock/indicators/indicators.js`.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `abands` series are defined in plotOptions.abands.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotAbandsOptions {
    /**
     * (Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highstock) Allow this series' points to be selected by clicking on the
     * graphic (columns, point markers, pie slices, map areas etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highstock) Enable or disable the initial animation when a series is
     * displayed. The animation can also be set as a configuration object.
     * Please note that this option only applies to the initial animation of the
     * series itself. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotAbandsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highstock) For some series, there is a limit that shuts down initial
     * animation by default when the total number of points in the chart is too
     * high. For example, for a column chart and its derivatives, animation does
     * not run if there is more than 250 points totally. To disable this cap,
     * set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highstock) Sets the color blending in the boost module.
     */
    boostBlending?: OptionsBoostBlendingValue;
    /**
     * (Highstock) Set the point threshold for when a series should enter boost
     * mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    bottomLine?: PlotAbandsBottomLineOptions;
    /**
     * (Highstock) An additional class name to apply to the series' graphical
     * elements. This option does not replace default class names of the
     * graphical element.
     */
    className?: string;
    /**
     * (Highstock) Disable this option to allow series rendering in the whole
     * plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highstock) The main color of the series. In line type series it applies
     * to the line and the point markers unless otherwise specified. In bar type
     * series it applies to the bars unless a color is specified per point. The
     * default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Styled mode only. A specific color index to use for the
     * series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highstock) Whether to compare indicator to the main series values or
     * indicator values.
     */
    compareToMain?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highstock) You can set the cursor to "pointer" if you have click events
     * attached to the series, to signal to the user that the points and lines
     * can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highstock) A reserved subspace to store options and values for
     * customized functionality. Here you can add additional data for your own
     * event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highstock) Name of the dash style to use for the graph, or for some
     * series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data
     * labels, appearing next to each data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     */
    dataLabels?: (PlotAbandsDataLabelsOptions|Array<PlotAbandsDataLabelsOptions>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotAbandsDataSortingOptions);
    /**
     * (Highstock) A description of the series to add to the screen reader
     * information about the series.
     */
    description?: string;
    /**
     * (Highstock) Enable or disable the mouse tracking for a specific series.
     * This includes point tooltips and click events on graphs and points. For
     * large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highstock) General event handlers for the series items. These event
     * hooks can also be attached to the series at run time using the
     * `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highstock) Determines whether the series should look for the nearest
     * point in both dimensions or just the x-dimension when hovering the
     * series. Defaults to `'xy'` for scatter series and `'x'` for most other
     * series. If the data has duplicate x-values, it is recommended to set this
     * to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highstock) When set to `false` will prevent the series data from being
     * included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The main series ID that indicator will be
     * based on. Required for this indicator.
     */
    linkedTo?: string;
    /**
     * (Highstock) Options for the point markers of line-like series. Properties
     * like `fillColor`, `lineColor` and `lineWidth` define the visual
     * appearance of the markers. Other series types, like column series, don't
     * have markers, but have visual options on the series level instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) The name of the series as shown in the legend, tooltip etc.
     * If not set, it will be based on a technical indicator type and default
     * params.
     */
    name?: string;
    /**
     * (Highstock) The color for the parts of the graph or points that are below
     * the threshold. Note that `zones` takes precedence over the negative
     * color. Using `negativeColor` is equivalent to applying a zone with value
     * of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Opacity of a series parts: line, fill (e.g. area) and
     * dataLabels.
     */
    opacity?: number;
    /**
     * (Highstock) Paramters used in calculation of regression series' points.
     */
    params?: PlotAbandsParamsOptions;
    /**
     * (Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highstock) Same as accessibility.pointDescriptionFormatter, but for an
     * individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highstock) Whether to select the series initially. If `showCheckbox` is
     * true, the checkbox next to the series name in the legend will be checked
     * for a selected series.
     */
    selected?: boolean;
    /**
     * (Highstock) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) If true, a checkbox is displayed next to the legend item to
     * allow selecting the series. The state of the checkbox is determined by
     * the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highstock) Whether to display this particular series or series type in
     * the legend. Standalone series are shown in legend by default, and linked
     * series are not. Since v7.2.0 it is possible to show series that use
     * colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) If set to `true`, the accessibility module will skip past the
     * points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     */
    step?: OptionsStepValue;
    /**
     * (Highstock) Sticky tracking of mouse events. When true, the `mouseOut`
     * event on a series isn't triggered until the mouse moves over another
     * series, or out of the plot area. When false, the `mouseOut` event on a
     * series is triggered when the mouse leaves the area around the series'
     * graph or markers. This also implies the tooltip when not shared. When
     * `stickyTracking` is false and `tooltip.shared` is false, the tooltip will
     * be hidden when moving the mouse between series. Defaults to true for line
     * and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The threshold, also called zero level or base
     * level. For line type series this is only used in conjunction with
     * negativeColor.
     */
    threshold?: number;
    /**
     * (Highstock) A configuration object for the tooltip rendering of each
     * single series. Properties are inherited from tooltip, but only the
     * following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    topLine?: PlotAbandsTopLineOptions;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highstock) Paramters used in calculation of regression series' points.
 */
export interface PlotAbandsParamsOptions {
    /**
     * (Highstock) The algorithms factor value used to calculate bands.
     */
    factor?: number;
    /**
     * (Highstock) The point index which indicator calculations will base. For
     * example using OHLC data, index=2 means the indicator will be calculated
     * using Low values.
     */
    index?: number;
    /**
     * (Highstock) The base period for indicator calculations. This is the
     * number of data points which are taken into account for the indicator
     * calculations.
     */
    period?: number;
}
export interface PlotAbandsTopLineOptions {
    styles?: PlotAbandsTopLineStylesOptions;
}
export interface PlotAbandsTopLineStylesOptions {
    /**
     * (Highstock) Pixel width of the line.
     */
    lineWidth?: number;
}
/**
 * (Highstock) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotAdAnimationOptions {
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotAdDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data labels,
 * appearing next to each data point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 */
export interface PlotAdDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|PlotAdDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotAdDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highstock) Accumulation Distribution (AD). This series requires `linkedTo`
 * option to be set.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `ad` series are defined in plotOptions.ad.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotAdOptions {
    /**
     * (Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highstock) Allow this series' points to be selected by clicking on the
     * graphic (columns, point markers, pie slices, map areas etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highstock) Enable or disable the initial animation when a series is
     * displayed. The animation can also be set as a configuration object.
     * Please note that this option only applies to the initial animation of the
     * series itself. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotAdAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highstock) For some series, there is a limit that shuts down initial
     * animation by default when the total number of points in the chart is too
     * high. For example, for a column chart and its derivatives, animation does
     * not run if there is more than 250 points totally. To disable this cap,
     * set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highstock) Sets the color blending in the boost module.
     */
    boostBlending?: OptionsBoostBlendingValue;
    /**
     * (Highstock) Set the point threshold for when a series should enter boost
     * mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    /**
     * (Highstock) An additional class name to apply to the series' graphical
     * elements. This option does not replace default class names of the
     * graphical element.
     */
    className?: string;
    /**
     * (Highstock) Disable this option to allow series rendering in the whole
     * plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highstock) The main color of the series. In line type series it applies
     * to the line and the point markers unless otherwise specified. In bar type
     * series it applies to the bars unless a color is specified per point. The
     * default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Styled mode only. A specific color index to use for the
     * series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Compare the values of the series against the first non-null,
     * non- zero value in the visible range. The y axis will show percentage or
     * absolute change depending on whether `compare` is set to `"percent"` or
     * `"value"`. When this is applied to multiple series, it allows comparing
     * the development of the series against each other. Adds a `change` field
     * to every point object.
     */
    compare?: string;
    /**
     * (Highstock) When compare is `percent`, this option dictates whether to
     * use 0 or 100 as the base of comparison.
     */
    compareBase?: (0|100);
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highstock) Whether to compare indicator to the main series values or
     * indicator values.
     */
    compareToMain?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highstock) You can set the cursor to "pointer" if you have click events
     * attached to the series, to signal to the user that the points and lines
     * can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highstock) A reserved subspace to store options and values for
     * customized functionality. Here you can add additional data for your own
     * event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highstock) Name of the dash style to use for the graph, or for some
     * series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data
     * labels, appearing next to each data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     */
    dataLabels?: (PlotAdDataLabelsOptions|Array<PlotAdDataLabelsOptions>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotAdDataSortingOptions);
    /**
     * (Highstock) A description of the series to add to the screen reader
     * information about the series.
     */
    description?: string;
    /**
     * (Highstock) Enable or disable the mouse tracking for a specific series.
     * This includes point tooltips and click events on graphs and points. For
     * large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highstock) General event handlers for the series items. These event
     * hooks can also be attached to the series at run time using the
     * `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highstock) Determines whether the series should look for the nearest
     * point in both dimensions or just the x-dimension when hovering the
     * series. Defaults to `'xy'` for scatter series and `'x'` for most other
     * series. If the data has duplicate x-values, it is recommended to set this
     * to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highstock) When set to `false` will prevent the series data from being
     * included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The main series ID that indicator will be
     * based on. Required for this indicator.
     */
    linkedTo?: string;
    /**
     * (Highstock) Options for the point markers of line-like series. Properties
     * like `fillColor`, `lineColor` and `lineWidth` define the visual
     * appearance of the markers. Other series types, like column series, don't
     * have markers, but have visual options on the series level instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) The name of the series as shown in the legend, tooltip etc.
     * If not set, it will be based on a technical indicator type and default
     * params.
     */
    name?: string;
    /**
     * (Highstock) The color for the parts of the graph or points that are below
     * the threshold. Note that `zones` takes precedence over the negative
     * color. Using `negativeColor` is equivalent to applying a zone with value
     * of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Opacity of a series parts: line, fill (e.g. area) and
     * dataLabels.
     */
    opacity?: number;
    /**
     * (Highstock) Paramters used in calculation of regression series' points.
     */
    params?: PlotAdParamsOptions;
    /**
     * (Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highstock) Same as accessibility.pointDescriptionFormatter, but for an
     * individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highstock) Whether to select the series initially. If `showCheckbox` is
     * true, the checkbox next to the series name in the legend will be checked
     * for a selected series.
     */
    selected?: boolean;
    /**
     * (Highstock) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) If true, a checkbox is displayed next to the legend item to
     * allow selecting the series. The state of the checkbox is determined by
     * the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highstock) Whether to display this particular series or series type in
     * the legend. Standalone series are shown in legend by default, and linked
     * series are not. Since v7.2.0 it is possible to show series that use
     * colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) If set to `true`, the accessibility module will skip past the
     * points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     */
    step?: OptionsStepValue;
    /**
     * (Highstock) Sticky tracking of mouse events. When true, the `mouseOut`
     * event on a series isn't triggered until the mouse moves over another
     * series, or out of the plot area. When false, the `mouseOut` event on a
     * series is triggered when the mouse leaves the area around the series'
     * graph or markers. This also implies the tooltip when not shared. When
     * `stickyTracking` is false and `tooltip.shared` is false, the tooltip will
     * be hidden when moving the mouse between series. Defaults to true for line
     * and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The threshold, also called zero level or base
     * level. For line type series this is only used in conjunction with
     * negativeColor.
     */
    threshold?: number;
    /**
     * (Highstock) A configuration object for the tooltip rendering of each
     * single series. Properties are inherited from tooltip, but only the
     * following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highstock) Paramters used in calculation of regression series' points.
 */
export interface PlotAdParamsOptions {
    /**
     * (Highstock) The point index which indicator calculations will base. For
     * example using OHLC data, index=2 means the indicator will be calculated
     * using Low values.
     */
    index?: number;
    /**
     * (Highstock) The base period for indicator calculations. This is the
     * number of data points which are taken into account for the indicator
     * calculations.
     */
    period?: number;
    /**
     * (Highstock) The id of volume series which is mandatory. For example using
     * OHLC data, volumeSeriesID='volume' means the indicator will be calculated
     * using OHLC and volume values.
     */
    volumeSeriesID?: string;
}
/**
 * (Highstock) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotAoAnimationOptions {
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotAoDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data labels,
 * appearing next to each data point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 */
export interface PlotAoDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|PlotAoDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotAoDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highstock) Awesome Oscillator. This series requires the `linkedTo` option to
 * be set and should be loaded after the `stock/indicators/indicators.js`
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `ao` series are defined in plotOptions.ao.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotAoOptions {
    /**
     * (Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highstock) Allow this series' points to be selected by clicking on the
     * graphic (columns, point markers, pie slices, map areas etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highstock) Enable or disable the initial animation when a series is
     * displayed. The animation can also be set as a configuration object.
     * Please note that this option only applies to the initial animation of the
     * series itself. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotAoAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highstock) For some series, there is a limit that shuts down initial
     * animation by default when the total number of points in the chart is too
     * high. For example, for a column chart and its derivatives, animation does
     * not run if there is more than 250 points totally. To disable this cap,
     * set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highstock) Sets the color blending in the boost module.
     */
    boostBlending?: OptionsBoostBlendingValue;
    /**
     * (Highstock) Set the point threshold for when a series should enter boost
     * mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    /**
     * (Highstock) An additional class name to apply to the series' graphical
     * elements. This option does not replace default class names of the
     * graphical element.
     */
    className?: string;
    /**
     * (Highstock) Disable this option to allow series rendering in the whole
     * plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highstock) The main color of the series. In line type series it applies
     * to the line and the point markers unless otherwise specified. In bar type
     * series it applies to the bars unless a color is specified per point. The
     * default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Styled mode only. A specific color index to use for the
     * series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Compare the values of the series against the first non-null,
     * non- zero value in the visible range. The y axis will show percentage or
     * absolute change depending on whether `compare` is set to `"percent"` or
     * `"value"`. When this is applied to multiple series, it allows comparing
     * the development of the series against each other. Adds a `change` field
     * to every point object.
     */
    compare?: string;
    /**
     * (Highstock) When compare is `percent`, this option dictates whether to
     * use 0 or 100 as the base of comparison.
     */
    compareBase?: (0|100);
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highstock) Whether to compare indicator to the main series values or
     * indicator values.
     */
    compareToMain?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highstock) You can set the cursor to "pointer" if you have click events
     * attached to the series, to signal to the user that the points and lines
     * can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highstock) A reserved subspace to store options and values for
     * customized functionality. Here you can add additional data for your own
     * event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highstock) Name of the dash style to use for the graph, or for some
     * series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data
     * labels, appearing next to each data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     */
    dataLabels?: (PlotAoDataLabelsOptions|Array<PlotAoDataLabelsOptions>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotAoDataSortingOptions);
    /**
     * (Highstock) A description of the series to add to the screen reader
     * information about the series.
     */
    description?: string;
    /**
     * (Highstock) Enable or disable the mouse tracking for a specific series.
     * This includes point tooltips and click events on graphs and points. For
     * large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highstock) General event handlers for the series items. These event
     * hooks can also be attached to the series at run time using the
     * `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highstock) Determines whether the series should look for the nearest
     * point in both dimensions or just the x-dimension when hovering the
     * series. Defaults to `'xy'` for scatter series and `'x'` for most other
     * series. If the data has duplicate x-values, it is recommended to set this
     * to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highstock) Color of the Awesome oscillator series bar that is greater
     * than the previous one. Note that if a `color` is defined, the `color`
     * takes precedence and the `greaterBarColor` is ignored.
     */
    greaterBarColor?: (ColorString|GradientColorObject|PatternObject);
    groupPadding?: number;
    /**
     * (Highstock) When set to `false` will prevent the series data from being
     * included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The main series ID that indicator will be
     * based on. Required for this indicator.
     */
    linkedTo?: string;
    /**
     * (Highstock) Color of the Awesome oscillator series bar that is lower than
     * the previous one. Note that if a `color` is defined, the `color` takes
     * precedence and the `lowerBarColor` is ignored.
     */
    lowerBarColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Options for the point markers of line-like series. Properties
     * like `fillColor`, `lineColor` and `lineWidth` define the visual
     * appearance of the markers. Other series types, like column series, don't
     * have markers, but have visual options on the series level instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) The name of the series as shown in the legend, tooltip etc.
     * If not set, it will be based on a technical indicator type and default
     * params.
     */
    name?: string;
    /**
     * (Highstock) The color for the parts of the graph or points that are below
     * the threshold. Note that `zones` takes precedence over the negative
     * color. Using `negativeColor` is equivalent to applying a zone with value
     * of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Opacity of a series parts: line, fill (e.g. area) and
     * dataLabels.
     */
    opacity?: number;
    /**
     * (Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highstock) Same as accessibility.pointDescriptionFormatter, but for an
     * individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    pointPadding?: number;
    /**
     * (Highstock) Whether to select the series initially. If `showCheckbox` is
     * true, the checkbox next to the series name in the legend will be checked
     * for a selected series.
     */
    selected?: boolean;
    /**
     * (Highstock) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) If true, a checkbox is displayed next to the legend item to
     * allow selecting the series. The state of the checkbox is determined by
     * the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highstock) Whether to display this particular series or series type in
     * the legend. Standalone series are shown in legend by default, and linked
     * series are not. Since v7.2.0 it is possible to show series that use
     * colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) If set to `true`, the accessibility module will skip past the
     * points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     */
    step?: OptionsStepValue;
    /**
     * (Highstock) Sticky tracking of mouse events. When true, the `mouseOut`
     * event on a series isn't triggered until the mouse moves over another
     * series, or out of the plot area. When false, the `mouseOut` event on a
     * series is triggered when the mouse leaves the area around the series'
     * graph or markers. This also implies the tooltip when not shared. When
     * `stickyTracking` is false and `tooltip.shared` is false, the tooltip will
     * be hidden when moving the mouse between series. Defaults to true for line
     * and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The threshold, also called zero level or base
     * level. For line type series this is only used in conjunction with
     * negativeColor.
     */
    threshold?: number;
    /**
     * (Highstock) A configuration object for the tooltip rendering of each
     * single series. Properties are inherited from tooltip, but only the
     * following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highstock) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotApoAnimationOptions {
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotApoDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data labels,
 * appearing next to each data point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 */
export interface PlotApoDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|PlotApoDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotApoDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highstock) Absolute Price Oscillator. This series requires the `linkedTo`
 * option to be set and should be loaded after the
 * `stock/indicators/indicators.js` and `stock/indicators/ema.js`.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `apo` series are defined in plotOptions.apo.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotApoOptions {
    /**
     * (Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highstock) Allow this series' points to be selected by clicking on the
     * graphic (columns, point markers, pie slices, map areas etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highstock) Enable or disable the initial animation when a series is
     * displayed. The animation can also be set as a configuration object.
     * Please note that this option only applies to the initial animation of the
     * series itself. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotApoAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highstock) For some series, there is a limit that shuts down initial
     * animation by default when the total number of points in the chart is too
     * high. For example, for a column chart and its derivatives, animation does
     * not run if there is more than 250 points totally. To disable this cap,
     * set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highstock) Sets the color blending in the boost module.
     */
    boostBlending?: OptionsBoostBlendingValue;
    /**
     * (Highstock) Set the point threshold for when a series should enter boost
     * mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    /**
     * (Highstock) An additional class name to apply to the series' graphical
     * elements. This option does not replace default class names of the
     * graphical element.
     */
    className?: string;
    /**
     * (Highstock) Disable this option to allow series rendering in the whole
     * plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highstock) The main color of the series. In line type series it applies
     * to the line and the point markers unless otherwise specified. In bar type
     * series it applies to the bars unless a color is specified per point. The
     * default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Styled mode only. A specific color index to use for the
     * series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Compare the values of the series against the first non-null,
     * non- zero value in the visible range. The y axis will show percentage or
     * absolute change depending on whether `compare` is set to `"percent"` or
     * `"value"`. When this is applied to multiple series, it allows comparing
     * the development of the series against each other. Adds a `change` field
     * to every point object.
     */
    compare?: string;
    /**
     * (Highstock) When compare is `percent`, this option dictates whether to
     * use 0 or 100 as the base of comparison.
     */
    compareBase?: (0|100);
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highstock) Whether to compare indicator to the main series values or
     * indicator values.
     */
    compareToMain?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highstock) You can set the cursor to "pointer" if you have click events
     * attached to the series, to signal to the user that the points and lines
     * can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highstock) A reserved subspace to store options and values for
     * customized functionality. Here you can add additional data for your own
     * event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highstock) Name of the dash style to use for the graph, or for some
     * series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data
     * labels, appearing next to each data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     */
    dataLabels?: (PlotApoDataLabelsOptions|Array<PlotApoDataLabelsOptions>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotApoDataSortingOptions);
    /**
     * (Highstock) A description of the series to add to the screen reader
     * information about the series.
     */
    description?: string;
    /**
     * (Highstock) Enable or disable the mouse tracking for a specific series.
     * This includes point tooltips and click events on graphs and points. For
     * large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highstock) General event handlers for the series items. These event
     * hooks can also be attached to the series at run time using the
     * `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highstock) Determines whether the series should look for the nearest
     * point in both dimensions or just the x-dimension when hovering the
     * series. Defaults to `'xy'` for scatter series and `'x'` for most other
     * series. If the data has duplicate x-values, it is recommended to set this
     * to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highstock) When set to `false` will prevent the series data from being
     * included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The main series ID that indicator will be
     * based on. Required for this indicator.
     */
    linkedTo?: string;
    /**
     * (Highstock) Options for the point markers of line-like series. Properties
     * like `fillColor`, `lineColor` and `lineWidth` define the visual
     * appearance of the markers. Other series types, like column series, don't
     * have markers, but have visual options on the series level instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) The name of the series as shown in the legend, tooltip etc.
     * If not set, it will be based on a technical indicator type and default
     * params.
     */
    name?: string;
    /**
     * (Highstock) The color for the parts of the graph or points that are below
     * the threshold. Note that `zones` takes precedence over the negative
     * color. Using `negativeColor` is equivalent to applying a zone with value
     * of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Opacity of a series parts: line, fill (e.g. area) and
     * dataLabels.
     */
    opacity?: number;
    /**
     * (Highstock) Paramters used in calculation of Absolute Price Oscillator
     * series points.
     */
    params?: PlotApoParamsOptions;
    /**
     * (Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highstock) Same as accessibility.pointDescriptionFormatter, but for an
     * individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highstock) Whether to select the series initially. If `showCheckbox` is
     * true, the checkbox next to the series name in the legend will be checked
     * for a selected series.
     */
    selected?: boolean;
    /**
     * (Highstock) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) If true, a checkbox is displayed next to the legend item to
     * allow selecting the series. The state of the checkbox is determined by
     * the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highstock) Whether to display this particular series or series type in
     * the legend. Standalone series are shown in legend by default, and linked
     * series are not. Since v7.2.0 it is possible to show series that use
     * colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) If set to `true`, the accessibility module will skip past the
     * points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     */
    step?: OptionsStepValue;
    /**
     * (Highstock) Sticky tracking of mouse events. When true, the `mouseOut`
     * event on a series isn't triggered until the mouse moves over another
     * series, or out of the plot area. When false, the `mouseOut` event on a
     * series is triggered when the mouse leaves the area around the series'
     * graph or markers. This also implies the tooltip when not shared. When
     * `stickyTracking` is false and `tooltip.shared` is false, the tooltip will
     * be hidden when moving the mouse between series. Defaults to true for line
     * and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The threshold, also called zero level or base
     * level. For line type series this is only used in conjunction with
     * negativeColor.
     */
    threshold?: number;
    /**
     * (Highstock) A configuration object for the tooltip rendering of each
     * single series. Properties are inherited from tooltip, but only the
     * following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highstock) Paramters used in calculation of Absolute Price Oscillator series
 * points.
 */
export interface PlotApoParamsOptions {
    /**
     * (Highstock) The point index which indicator calculations will base. For
     * example using OHLC data, index=2 means the indicator will be calculated
     * using Low values.
     *
     * By default index value used to be set to 0. Since Highstock 7 by default
     * index is set to 3 which means that the ema indicator will be calculated
     * using Close values.
     */
    index?: number;
    /**
     * (Highstock) Periods for Absolute Price Oscillator calculations.
     */
    periods?: Array<number>;
}
/**
 * (Highcharts, Highstock) Enable or disable the initial animation when a series
 * is displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotAreaAnimationOptions {
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotAreaDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data labels,
 * appearing next to each data point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 */
export interface PlotAreaDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|PlotAreaDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotAreaDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highcharts, Highstock) Style options for the guide box. The guide box has
 * one state by default, the `default` state.
 */
export interface PlotAreaDragDropGuideBoxOptions {
    /**
     * (Highcharts, Highstock) Style options for the guide box default state.
     */
    default?: DragDropGuideBoxOptionsObject;
}
/**
 * (Highcharts, Highstock) The area series type.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `area` series are defined in plotOptions.area.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotAreaOptions {
    /**
     * (Highcharts, Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highmaps) Whether all areas of the map defined in `mapData` should be
     * rendered. If `true`, areas which don't correspond to a data point, are
     * rendered as `null` points. If `false`, those areas are skipped.
     */
    allAreas?: boolean;
    /**
     * (Highcharts, Highstock) Allow this series' points to be selected by
     * clicking on the graphic (columns, point markers, pie slices, map areas
     * etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highcharts, Highstock) Enable or disable the initial animation when a
     * series is displayed. The animation can also be set as a configuration
     * object. Please note that this option only applies to the initial
     * animation of the series itself. For other animations, see chart.animation
     * and the animation parameter under the API methods. The following
     * properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotAreaAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock) For some series, there is a limit that shuts down
     * initial animation by default when the total number of points in the chart
     * is too high. For example, for a column chart and its derivatives,
     * animation does not run if there is more than 250 points totally. To
     * disable this cap, set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highcharts, Highstock) Sets the color blending in the boost module.
     */
    boostBlending?: OptionsBoostBlendingValue;
    /**
     * (Highcharts, Highstock) Set the point threshold for when a series should
     * enter boost mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock) An additional class name to apply to the series'
     * graphical elements. This option does not replace default class names of
     * the graphical element.
     */
    className?: string;
    /**
     * (Highcharts, Highstock) Disable this option to allow series rendering in
     * the whole plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highcharts, Highstock) The main color of the series. In line type series
     * it applies to the line and the point markers unless otherwise specified.
     * In bar type series it applies to the bars unless a color is specified per
     * point. The default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) When using dual or multiple color axes,
     * this number defines which colorAxis the particular series is connected
     * to. It refers to either the axis id or the index of the axis in the
     * colorAxis array, with 0 being the first. Set this option to false to
     * prevent a series from connecting to the default color axis.
     *
     * Since v7.2.0 the option can also be an axis id or an axis index instead
     * of a boolean flag.
     */
    colorAxis?: (boolean|number|string);
    /**
     * (Highcharts, Highstock) Styled mode only. A specific color index to use
     * for the series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Compare the values of the series against the first non-null,
     * non- zero value in the visible range. The y axis will show percentage or
     * absolute change depending on whether `compare` is set to `"percent"` or
     * `"value"`. When this is applied to multiple series, it allows comparing
     * the development of the series against each other. Adds a `change` field
     * to every point object.
     */
    compare?: string;
    /**
     * (Highstock) When compare is `percent`, this option dictates whether to
     * use 0 or 100 as the base of comparison.
     */
    compareBase?: (0|100);
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highcharts, Highstock) You can set the cursor to "pointer" if you have
     * click events attached to the series, to signal to the user that the
     * points and lines can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highcharts, Highstock) A reserved subspace to store options and values
     * for customized functionality. Here you can add additional data for your
     * own event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highcharts, Highstock) Name of the dash style to use for the graph, or
     * for some series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data
     * labels, appearing next to each data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     */
    dataLabels?: (PlotAreaDataLabelsOptions|Array<PlotAreaDataLabelsOptions>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotAreaDataSortingOptions);
    /**
     * (Highcharts, Highstock) A description of the series to add to the screen
     * reader information about the series.
     */
    description?: string;
    /**
     * (Highcharts, Highstock) The draggable-points module allows points to be
     * moved around or modified in the chart. In addition to the options
     * mentioned under the `dragDrop` API structure, the module fires three
     * events, point.dragStart, point.drag and point.drop.
     */
    dragDrop?: SeriesDragDropOptionsObject;
    /**
     * (Highcharts, Highstock) Enable or disable the mouse tracking for a
     * specific series. This includes point tooltips and click events on graphs
     * and points. For large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highcharts, Highstock) General event handlers for the series items.
     * These event hooks can also be attached to the series at run time using
     * the `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highcharts, Highstock) Fill color or gradient for the area. When `null`,
     * the series' `color` is used with the series' `fillOpacity`.
     *
     * In styled mode, the fill color can be set with the `.highcharts-area`
     * class name.
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Fill opacity for the area. When you set an
     * explicit `fillColor`, the `fillOpacity` is not applied. Instead, you
     * should define the opacity in the `fillColor` with an rgba color
     * definition. The `fillOpacity` setting, also the default setting,
     * overrides the alpha component of the `color` setting.
     *
     * In styled mode, the fill opacity can be set with the `.highcharts-area`
     * class name.
     */
    fillOpacity?: number;
    /**
     * (Highcharts, Highstock) Determines whether the series should look for the
     * nearest point in both dimensions or just the x-dimension when hovering
     * the series. Defaults to `'xy'` for scatter series and `'x'` for most
     * other series. If the data has duplicate x-values, it is recommended to
     * set this to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highcharts, Highstock) When set to `false` will prevent the series data
     * from being included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highmaps) What property to join the `mapData` to the value data. For
     * example, if joinBy is "code", the mapData items with a specific code is
     * merged into the data with the same code. For maps loaded from GeoJSON,
     * the keys may be held in each point's `properties` object.
     *
     * The joinBy option can also be an array of two values, where the first
     * points to a key in the `mapData`, and the second points to another key in
     * the `data`.
     *
     * When joinBy is `null`, the map items are joined by their position in the
     * array, which performs much better in maps with many data points. This is
     * the recommended option if you are printing more than a thousand data
     * points and have a backend that can preprocess the data into a parallel
     * array of the mapData.
     */
    joinBy?: (string|Array<string>);
    /**
     * (Highcharts, Highstock) An array specifying which option maps to which
     * key in the data point array. This makes it convenient to work with
     * unstructured data arrays from different sources.
     */
    keys?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) A separate color for the graph line. By default
     * the line takes the `color` of the series, but the lineColor setting
     * allows setting a separate color for the line without altering the
     * `fillColor`.
     *
     * In styled mode, the line stroke can be set with the `.highcharts-graph`
     * class name.
     */
    lineColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The id of another series to link to.
     * Additionally, the value can be ":previous" to link to the previous
     * series. When two series are linked, only the first one appears in the
     * legend. Toggling the visibility of this also toggles the linked series.
     *
     * If master series uses data sorting and linked series does not have its
     * own sorting definition, the linked series will be sorted in the same
     * order as the master one.
     */
    linkedTo?: string;
    /**
     * (Highcharts, Highstock) Options for the point markers of line-like
     * series. Properties like `fillColor`, `lineColor` and `lineWidth` define
     * the visual appearance of the markers. Other series types, like column
     * series, don't have markers, but have visual options on the series level
     * instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) Options for the corresponding navigator series if
     * `showInNavigator` is `true` for this series. Available options are the
     * same as any series, documented at plotOptions and series.
     *
     * These options are merged with options in navigator.series, and will take
     * precedence if the same option is defined both places.
     */
    navigatorOptions?: PlotSeriesOptions;
    /**
     * (Highcharts, Highstock) The color for the parts of the graph or points
     * that are below the threshold. Note that `zones` takes precedence over the
     * negative color. Using `negativeColor` is equivalent to applying a zone
     * with value of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) A separate color for the negative part of the area.
     *
     * In styled mode, a negative color is set with the `.highcharts-negative`
     * class name.
     */
    negativeFillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Opacity of a series parts: line, fill (e.g. area)
     * and dataLabels.
     */
    opacity?: number;
    /**
     * (Highcharts, Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highcharts, Highstock) Same as accessibility.pointDescriptionFormatter,
     * but for an individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highcharts, Highstock, Gantt) If no x values are given for the points in
     * a series, `pointInterval` defines the interval of the x values. For
     * example, if a series contains one value every decade starting from year
     * 0, set `pointInterval` to `10`. In true `datetime` axes, the
     * `pointInterval` is set in milliseconds.
     *
     * It can be also be combined with `pointIntervalUnit` to draw irregular
     * time intervals.
     *
     * Please note that this options applies to the _series data_, not the
     * interval of the axis ticks, which is independent.
     */
    pointInterval?: number;
    /**
     * (Highcharts, Highstock, Gantt) On datetime series, this allows for
     * setting the pointInterval to irregular time units, `day`, `month` and
     * `year`. A day is usually the same as 24 hours, but `pointIntervalUnit`
     * also takes the DST crossover into consideration when dealing with local
     * time. Combine this option with `pointInterval` to draw weeks, quarters, 6
     * months, 10 years etc.
     *
     * Please note that this options applies to the _series data_, not the
     * interval of the axis ticks, which is independent.
     */
    pointIntervalUnit?: OptionsPointIntervalUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Possible values: `"on"`, `"between"`,
     * `number`.
     *
     * In a column chart, when pointPlacement is `"on"`, the point will not
     * create any padding of the X axis. In a polar column chart this means that
     * the first column points directly north. If the pointPlacement is
     * `"between"`, the columns will be laid out between ticks. This is useful
     * for example for visualising an amount between two points in time or in a
     * certain sector of a polar chart.
     *
     * Since Highcharts 3.0.2, the point placement can also be numeric, where 0
     * is on the axis value, -0.5 is between this value and the previous, and
     * 0.5 is between this value and the next. Unlike the textual options,
     * numeric point placement options won't affect axis padding.
     *
     * Note that pointPlacement needs a pointRange to work. For column series
     * this is computed, but for line-type series it needs to be set.
     *
     * For the `xrange` series type and gantt charts, if the Y axis is a
     * category axis, the `pointPlacement` applies to the Y axis rather than the
     * (typically datetime) X axis.
     *
     * Defaults to `undefined` in cartesian charts, `"between"` in polar charts.
     */
    pointPlacement?: (number|string);
    /**
     * (Highstock) The width of each point on the x axis. For example in a
     * column chart with one value each day, the pointRange would be 1 day (= 24
     * * 3600
     *
     * * 1000 milliseconds). This is normally computed automatically, but this
     * option can be used to override the automatic value.
     */
    pointRange?: number;
    /**
     * (Highcharts, Highstock, Gantt) If no x values are given for the points in
     * a series, pointStart defines on what value to start. For example, if a
     * series contains one yearly value starting from 1945, set pointStart to
     * 1945.
     */
    pointStart?: number;
    /**
     * (Highcharts, Highstock) Whether to select the series initially. If
     * `showCheckbox` is true, the checkbox next to the series name in the
     * legend will be checked for a selected series.
     */
    selected?: boolean;
    /**
     * (Highcharts, Highstock) Whether to apply a drop shadow to the graph line.
     * Since 2.3 the shadow can be an object configuration containing `color`,
     * `offsetX`, `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock) If true, a checkbox is displayed next to the
     * legend item to allow selecting the series. The state of the checkbox is
     * determined by the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highcharts, Highstock) Whether to display this particular series or
     * series type in the legend. Standalone series are shown in legend by
     * default, and linked series are not. Since v7.2.0 it is possible to show
     * series that use colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) Whether or not to show the series in the navigator. Takes
     * precedence over navigator.baseSeries if defined.
     */
    showInNavigator?: boolean;
    /**
     * (Highcharts, Highstock) If set to `true`, the accessibility module will
     * skip past the points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    /**
     * (Highcharts, Highstock) Whether to stack the values of each series on top
     * of each other. Possible values are `undefined` to disable, `"normal"` to
     * stack by value or `"percent"`.
     *
     * When stacking is enabled, data must be sorted in ascending X order.
     *
     * Some stacking options are related to specific series types. In the
     * streamgraph series type, the stacking option is set to `"stream"`. The
     * second one is `"overlap"`, which only applies to waterfall series.
     */
    stacking?: OptionsStackingValue;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     */
    step?: OptionsStepValue;
    /**
     * (Highcharts, Highstock) Sticky tracking of mouse events. When true, the
     * `mouseOut` event on a series isn't triggered until the mouse moves over
     * another series, or out of the plot area. When false, the `mouseOut` event
     * on a series is triggered when the mouse leaves the area around the
     * series' graph or markers. This also implies the tooltip when not shared.
     * When `stickyTracking` is false and `tooltip.shared` is false, the tooltip
     * will be hidden when moving the mouse between series. Defaults to true for
     * line and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The Y axis value to serve as the base for the
     * area, for distinguishing between values above and below a threshold. The
     * area between the graph and the threshold is filled.
     *
     * * If a number is given, the Y axis will scale to the threshold.
     *
     * * If `null`, the scaling behaves like a line series with fill between the
     * graph and the Y axis minimum.
     *
     * * If `Infinity` or `-Infinity`, the area between the graph and the
     * corresponding Y axis extreme is filled (since v6.1.0).
     */
    threshold?: (number|null);
    /**
     * (Highcharts, Highstock) A configuration object for the tooltip rendering
     * of each single series. Properties are inherited from tooltip, but only
     * the following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    /**
     * (Highcharts, Highstock) Whether the whole area or just the line should
     * respond to mouseover tooltips and other mouse or touch events.
     */
    trackByArea?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highcharts, Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highcharts, Highstock) Enable or disable the initial animation when a series
 * is displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotArearangeAnimationOptions {
    defer?: number;
}
/**
 * (Highcharts, Highstock) Enable or disable the initial animation when a series
 * is displayed for the `dataLabels`. The animation can also be set as a
 * configuration object. Please note that this option only applies to the
 * initial animation. For other animations, see chart.animation and the
 * animation parameter under the API methods. The following properties are
 * supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotArearangeDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock) The animation delay time in milliseconds. Set to
     * `0` renders dataLabel immediately. As `undefined` inherits defer time
     * from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotArearangeDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highcharts, Highstock) Style options for the guide box. The guide box has
 * one state by default, the `default` state.
 */
export interface PlotArearangeDragDropGuideBoxOptions {
    /**
     * (Highcharts, Highstock) Style options for the guide box default state.
     */
    default?: DragDropGuideBoxOptionsObject;
}
/**
 * (Highcharts, Highstock) The area range series is a carteseian series with
 * higher and lower values for each point along an X axis, where the area
 * between the values is shaded.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `arearange` series are defined in plotOptions.arearange.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotArearangeOptions {
    /**
     * (Highcharts, Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highmaps) Whether all areas of the map defined in `mapData` should be
     * rendered. If `true`, areas which don't correspond to a data point, are
     * rendered as `null` points. If `false`, those areas are skipped.
     */
    allAreas?: boolean;
    /**
     * (Highcharts, Highstock) Allow this series' points to be selected by
     * clicking on the graphic (columns, point markers, pie slices, map areas
     * etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highcharts, Highstock) Enable or disable the initial animation when a
     * series is displayed. The animation can also be set as a configuration
     * object. Please note that this option only applies to the initial
     * animation of the series itself. For other animations, see chart.animation
     * and the animation parameter under the API methods. The following
     * properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotArearangeAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock) For some series, there is a limit that shuts down
     * initial animation by default when the total number of points in the chart
     * is too high. For example, for a column chart and its derivatives,
     * animation does not run if there is more than 250 points totally. To
     * disable this cap, set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highcharts, Highstock) Sets the color blending in the boost module.
     */
    boostBlending?: OptionsBoostBlendingValue;
    /**
     * (Highcharts, Highstock) Set the point threshold for when a series should
     * enter boost mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock) An additional class name to apply to the series'
     * graphical elements. This option does not replace default class names of
     * the graphical element.
     */
    className?: string;
    /**
     * (Highcharts, Highstock) Disable this option to allow series rendering in
     * the whole plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highcharts, Highstock) The main color of the series. In line type series
     * it applies to the line and the point markers unless otherwise specified.
     * In bar type series it applies to the bars unless a color is specified per
     * point. The default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) When using dual or multiple color axes,
     * this number defines which colorAxis the particular series is connected
     * to. It refers to either the axis id or the index of the axis in the
     * colorAxis array, with 0 being the first. Set this option to false to
     * prevent a series from connecting to the default color axis.
     *
     * Since v7.2.0 the option can also be an axis id or an axis index instead
     * of a boolean flag.
     */
    colorAxis?: (boolean|number|string);
    /**
     * (Highcharts, Highstock) Styled mode only. A specific color index to use
     * for the series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Compare the values of the series against the first non-null,
     * non- zero value in the visible range. The y axis will show percentage or
     * absolute change depending on whether `compare` is set to `"percent"` or
     * `"value"`. When this is applied to multiple series, it allows comparing
     * the development of the series against each other. Adds a `change` field
     * to every point object.
     */
    compare?: string;
    /**
     * (Highstock) When compare is `percent`, this option dictates whether to
     * use 0 or 100 as the base of comparison.
     */
    compareBase?: (0|100);
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highcharts, Highstock) You can set the cursor to "pointer" if you have
     * click events attached to the series, to signal to the user that the
     * points and lines can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highcharts, Highstock) A reserved subspace to store options and values
     * for customized functionality. Here you can add additional data for your
     * own event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highcharts, Highstock) Name of the dash style to use for the graph, or
     * for some series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock) Extended data labels for range series types.
     * Range series data labels use no `x` and `y` options. Instead, they have
     * `xLow`, `xHigh`, `yLow` and `yHigh` options to allow the higher and lower
     * data label sets individually.
     */
    dataLabels?: (SeriesAreaRangeDataLabelsOptionsObject|Array<SeriesAreaRangeDataLabelsOptionsObject>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotArearangeDataSortingOptions);
    /**
     * (Highcharts, Highstock) A description of the series to add to the screen
     * reader information about the series.
     */
    description?: string;
    /**
     * (Highcharts, Highstock) The draggable-points module allows points to be
     * moved around or modified in the chart. In addition to the options
     * mentioned under the `dragDrop` API structure, the module fires three
     * events, point.dragStart, point.drag and point.drop.
     */
    dragDrop?: SeriesDragDropOptionsObject;
    /**
     * (Highcharts, Highstock) Enable or disable the mouse tracking for a
     * specific series. This includes point tooltips and click events on graphs
     * and points. For large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highcharts, Highstock) General event handlers for the series items.
     * These event hooks can also be attached to the series at run time using
     * the `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highcharts, Highstock) Fill color or gradient for the area. When `null`,
     * the series' `color` is used with the series' `fillOpacity`.
     *
     * In styled mode, the fill color can be set with the `.highcharts-area`
     * class name.
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Fill opacity for the area. When you set an
     * explicit `fillColor`, the `fillOpacity` is not applied. Instead, you
     * should define the opacity in the `fillColor` with an rgba color
     * definition. The `fillOpacity` setting, also the default setting,
     * overrides the alpha component of the `color` setting.
     *
     * In styled mode, the fill opacity can be set with the `.highcharts-area`
     * class name.
     */
    fillOpacity?: number;
    /**
     * (Highcharts, Highstock) Determines whether the series should look for the
     * nearest point in both dimensions or just the x-dimension when hovering
     * the series. Defaults to `'xy'` for scatter series and `'x'` for most
     * other series. If the data has duplicate x-values, it is recommended to
     * set this to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highcharts, Highstock) When set to `false` will prevent the series data
     * from being included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highmaps) What property to join the `mapData` to the value data. For
     * example, if joinBy is "code", the mapData items with a specific code is
     * merged into the data with the same code. For maps loaded from GeoJSON,
     * the keys may be held in each point's `properties` object.
     *
     * The joinBy option can also be an array of two values, where the first
     * points to a key in the `mapData`, and the second points to another key in
     * the `data`.
     *
     * When joinBy is `null`, the map items are joined by their position in the
     * array, which performs much better in maps with many data points. This is
     * the recommended option if you are printing more than a thousand data
     * points and have a backend that can preprocess the data into a parallel
     * array of the mapData.
     */
    joinBy?: (string|Array<string>);
    /**
     * (Highcharts, Highstock) An array specifying which option maps to which
     * key in the data point array. This makes it convenient to work with
     * unstructured data arrays from different sources.
     */
    keys?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) A separate color for the graph line. By default
     * the line takes the `color` of the series, but the lineColor setting
     * allows setting a separate color for the line without altering the
     * `fillColor`.
     *
     * In styled mode, the line stroke can be set with the `.highcharts-graph`
     * class name.
     */
    lineColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Pixel width of the arearange graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The id of another series to link to.
     * Additionally, the value can be ":previous" to link to the previous
     * series. When two series are linked, only the first one appears in the
     * legend. Toggling the visibility of this also toggles the linked series.
     *
     * If master series uses data sorting and linked series does not have its
     * own sorting definition, the linked series will be sorted in the same
     * order as the master one.
     */
    linkedTo?: string;
    /**
     * (Highcharts, Highstock) Options for the point markers of line-like
     * series. Properties like `fillColor`, `lineColor` and `lineWidth` define
     * the visual appearance of the markers. Other series types, like column
     * series, don't have markers, but have visual options on the series level
     * instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) Options for the corresponding navigator series if
     * `showInNavigator` is `true` for this series. Available options are the
     * same as any series, documented at plotOptions and series.
     *
     * These options are merged with options in navigator.series, and will take
     * precedence if the same option is defined both places.
     */
    navigatorOptions?: PlotSeriesOptions;
    /**
     * (Highcharts, Highstock) The color for the parts of the graph or points
     * that are below the threshold. Note that `zones` takes precedence over the
     * negative color. Using `negativeColor` is equivalent to applying a zone
     * with value of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) A separate color for the negative part of the area.
     *
     * In styled mode, a negative color is set with the `.highcharts-negative`
     * class name.
     */
    negativeFillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Opacity of a series parts: line, fill (e.g. area)
     * and dataLabels.
     */
    opacity?: number;
    /**
     * (Highcharts, Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highcharts, Highstock) Same as accessibility.pointDescriptionFormatter,
     * but for an individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highcharts, Highstock, Gantt) If no x values are given for the points in
     * a series, `pointInterval` defines the interval of the x values. For
     * example, if a series contains one value every decade starting from year
     * 0, set `pointInterval` to `10`. In true `datetime` axes, the
     * `pointInterval` is set in milliseconds.
     *
     * It can be also be combined with `pointIntervalUnit` to draw irregular
     * time intervals.
     *
     * Please note that this options applies to the _series data_, not the
     * interval of the axis ticks, which is independent.
     */
    pointInterval?: number;
    /**
     * (Highcharts, Highstock, Gantt) On datetime series, this allows for
     * setting the pointInterval to irregular time units, `day`, `month` and
     * `year`. A day is usually the same as 24 hours, but `pointIntervalUnit`
     * also takes the DST crossover into consideration when dealing with local
     * time. Combine this option with `pointInterval` to draw weeks, quarters, 6
     * months, 10 years etc.
     *
     * Please note that this options applies to the _series data_, not the
     * interval of the axis ticks, which is independent.
     */
    pointIntervalUnit?: OptionsPointIntervalUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Possible values: `"on"`, `"between"`,
     * `number`.
     *
     * In a column chart, when pointPlacement is `"on"`, the point will not
     * create any padding of the X axis. In a polar column chart this means that
     * the first column points directly north. If the pointPlacement is
     * `"between"`, the columns will be laid out between ticks. This is useful
     * for example for visualising an amount between two points in time or in a
     * certain sector of a polar chart.
     *
     * Since Highcharts 3.0.2, the point placement can also be numeric, where 0
     * is on the axis value, -0.5 is between this value and the previous, and
     * 0.5 is between this value and the next. Unlike the textual options,
     * numeric point placement options won't affect axis padding.
     *
     * Note that pointPlacement needs a pointRange to work. For column series
     * this is computed, but for line-type series it needs to be set.
     *
     * For the `xrange` series type and gantt charts, if the Y axis is a
     * category axis, the `pointPlacement` applies to the Y axis rather than the
     * (typically datetime) X axis.
     *
     * Defaults to `undefined` in cartesian charts, `"between"` in polar charts.
     */
    pointPlacement?: (number|string);
    /**
     * (Highstock) The width of each point on the x axis. For example in a
     * column chart with one value each day, the pointRange would be 1 day (= 24
     * * 3600
     *
     * * 1000 milliseconds). This is normally computed automatically, but this
     * option can be used to override the automatic value.
     */
    pointRange?: number;
    /**
     * (Highcharts, Highstock, Gantt) If no x values are given for the points in
     * a series, pointStart defines on what value to start. For example, if a
     * series contains one yearly value starting from 1945, set pointStart to
     * 1945.
     */
    pointStart?: number;
    /**
     * (Highcharts, Highstock) Whether to select the series initially. If
     * `showCheckbox` is true, the checkbox next to the series name in the
     * legend will be checked for a selected series.
     */
    selected?: boolean;
    /**
     * (Highcharts) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock) If true, a checkbox is displayed next to the
     * legend item to allow selecting the series. The state of the checkbox is
     * determined by the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highcharts, Highstock) Whether to display this particular series or
     * series type in the legend. Standalone series are shown in legend by
     * default, and linked series are not. Since v7.2.0 it is possible to show
     * series that use colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) Whether or not to show the series in the navigator. Takes
     * precedence over navigator.baseSeries if defined.
     */
    showInNavigator?: boolean;
    /**
     * (Highcharts, Highstock) If set to `true`, the accessibility module will
     * skip past the points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     */
    step?: OptionsStepValue;
    /**
     * (Highcharts, Highstock) Sticky tracking of mouse events. When true, the
     * `mouseOut` event on a series isn't triggered until the mouse moves over
     * another series, or out of the plot area. When false, the `mouseOut` event
     * on a series is triggered when the mouse leaves the area around the
     * series' graph or markers. This also implies the tooltip when not shared.
     * When `stickyTracking` is false and `tooltip.shared` is false, the tooltip
     * will be hidden when moving the mouse between series. Defaults to true for
     * line and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The Y axis value to serve as the base for the
     * area, for distinguishing between values above and below a threshold. The
     * area between the graph and the threshold is filled.
     *
     * * If a number is given, the Y axis will scale to the threshold.
     *
     * * If `null`, the scaling behaves like a line series with fill between the
     * graph and the Y axis minimum.
     *
     * * If `Infinity` or `-Infinity`, the area between the graph and the
     * corresponding Y axis extreme is filled (since v6.1.0).
     */
    threshold?: (number|null);
    /**
     * (Highcharts, Highstock) A configuration object for the tooltip rendering
     * of each single series. Properties are inherited from tooltip, but only
     * the following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    /**
     * (Highcharts, Highstock) Whether the whole area or just the line should
     * respond to mouseover tooltips and other mouse or touch events.
     */
    trackByArea?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highcharts, Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highcharts, Highstock) Enable or disable the initial animation when a series
 * is displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotAreasplineAnimationOptions {
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotAreasplineDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data labels,
 * appearing next to each data point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 */
export interface PlotAreasplineDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|PlotAreasplineDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotAreasplineDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highcharts, Highstock) Style options for the guide box. The guide box has
 * one state by default, the `default` state.
 */
export interface PlotAreasplineDragDropGuideBoxOptions {
    /**
     * (Highcharts, Highstock) Style options for the guide box default state.
     */
    default?: DragDropGuideBoxOptionsObject;
}
/**
 * (Highcharts, Highstock) The area spline series is an area series where the
 * graph between the points is smoothed into a spline.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `areaspline` series are defined in plotOptions.areaspline.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotAreasplineOptions {
    /**
     * (Highcharts, Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highmaps) Whether all areas of the map defined in `mapData` should be
     * rendered. If `true`, areas which don't correspond to a data point, are
     * rendered as `null` points. If `false`, those areas are skipped.
     */
    allAreas?: boolean;
    /**
     * (Highcharts, Highstock) Allow this series' points to be selected by
     * clicking on the graphic (columns, point markers, pie slices, map areas
     * etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highcharts, Highstock) Enable or disable the initial animation when a
     * series is displayed. The animation can also be set as a configuration
     * object. Please note that this option only applies to the initial
     * animation of the series itself. For other animations, see chart.animation
     * and the animation parameter under the API methods. The following
     * properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotAreasplineAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock) For some series, there is a limit that shuts down
     * initial animation by default when the total number of points in the chart
     * is too high. For example, for a column chart and its derivatives,
     * animation does not run if there is more than 250 points totally. To
     * disable this cap, set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock) An additional class name to apply to the series'
     * graphical elements. This option does not replace default class names of
     * the graphical element.
     */
    className?: string;
    /**
     * (Highcharts, Highstock) Disable this option to allow series rendering in
     * the whole plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highcharts, Highstock) The main color of the series. In line type series
     * it applies to the line and the point markers unless otherwise specified.
     * In bar type series it applies to the bars unless a color is specified per
     * point. The default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) When using dual or multiple color axes,
     * this number defines which colorAxis the particular series is connected
     * to. It refers to either the axis id or the index of the axis in the
     * colorAxis array, with 0 being the first. Set this option to false to
     * prevent a series from connecting to the default color axis.
     *
     * Since v7.2.0 the option can also be an axis id or an axis index instead
     * of a boolean flag.
     */
    colorAxis?: (boolean|number|string);
    /**
     * (Highcharts, Highstock) Styled mode only. A specific color index to use
     * for the series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Compare the values of the series against the first non-null,
     * non- zero value in the visible range. The y axis will show percentage or
     * absolute change depending on whether `compare` is set to `"percent"` or
     * `"value"`. When this is applied to multiple series, it allows comparing
     * the development of the series against each other. Adds a `change` field
     * to every point object.
     */
    compare?: string;
    /**
     * (Highstock) When compare is `percent`, this option dictates whether to
     * use 0 or 100 as the base of comparison.
     */
    compareBase?: (0|100);
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highcharts, Highstock) You can set the cursor to "pointer" if you have
     * click events attached to the series, to signal to the user that the
     * points and lines can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highcharts, Highstock) A reserved subspace to store options and values
     * for customized functionality. Here you can add additional data for your
     * own event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highcharts, Highstock) Name of the dash style to use for the graph, or
     * for some series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data
     * labels, appearing next to each data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     */
    dataLabels?: (PlotAreasplineDataLabelsOptions|Array<PlotAreasplineDataLabelsOptions>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotAreasplineDataSortingOptions);
    /**
     * (Highcharts, Highstock) A description of the series to add to the screen
     * reader information about the series.
     */
    description?: string;
    /**
     * (Highcharts, Highstock) The draggable-points module allows points to be
     * moved around or modified in the chart. In addition to the options
     * mentioned under the `dragDrop` API structure, the module fires three
     * events, point.dragStart, point.drag and point.drop.
     */
    dragDrop?: SeriesDragDropOptionsObject;
    /**
     * (Highcharts, Highstock) Enable or disable the mouse tracking for a
     * specific series. This includes point tooltips and click events on graphs
     * and points. For large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highcharts, Highstock) General event handlers for the series items.
     * These event hooks can also be attached to the series at run time using
     * the `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highcharts, Highstock) Fill color or gradient for the area. When `null`,
     * the series' `color` is used with the series' `fillOpacity`.
     *
     * In styled mode, the fill color can be set with the `.highcharts-area`
     * class name.
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Fill opacity for the area. When you set an
     * explicit `fillColor`, the `fillOpacity` is not applied. Instead, you
     * should define the opacity in the `fillColor` with an rgba color
     * definition. The `fillOpacity` setting, also the default setting,
     * overrides the alpha component of the `color` setting.
     *
     * In styled mode, the fill opacity can be set with the `.highcharts-area`
     * class name.
     */
    fillOpacity?: number;
    /**
     * (Highcharts, Highstock) Determines whether the series should look for the
     * nearest point in both dimensions or just the x-dimension when hovering
     * the series. Defaults to `'xy'` for scatter series and `'x'` for most
     * other series. If the data has duplicate x-values, it is recommended to
     * set this to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highcharts, Highstock) When set to `false` will prevent the series data
     * from being included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highmaps) What property to join the `mapData` to the value data. For
     * example, if joinBy is "code", the mapData items with a specific code is
     * merged into the data with the same code. For maps loaded from GeoJSON,
     * the keys may be held in each point's `properties` object.
     *
     * The joinBy option can also be an array of two values, where the first
     * points to a key in the `mapData`, and the second points to another key in
     * the `data`.
     *
     * When joinBy is `null`, the map items are joined by their position in the
     * array, which performs much better in maps with many data points. This is
     * the recommended option if you are printing more than a thousand data
     * points and have a backend that can preprocess the data into a parallel
     * array of the mapData.
     */
    joinBy?: (string|Array<string>);
    /**
     * (Highcharts, Highstock) An array specifying which option maps to which
     * key in the data point array. This makes it convenient to work with
     * unstructured data arrays from different sources.
     */
    keys?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) A separate color for the graph line. By default
     * the line takes the `color` of the series, but the lineColor setting
     * allows setting a separate color for the line without altering the
     * `fillColor`.
     *
     * In styled mode, the line stroke can be set with the `.highcharts-graph`
     * class name.
     */
    lineColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The id of another series to link to.
     * Additionally, the value can be ":previous" to link to the previous
     * series. When two series are linked, only the first one appears in the
     * legend. Toggling the visibility of this also toggles the linked series.
     *
     * If master series uses data sorting and linked series does not have its
     * own sorting definition, the linked series will be sorted in the same
     * order as the master one.
     */
    linkedTo?: string;
    /**
     * (Highcharts, Highstock) Options for the point markers of line-like
     * series. Properties like `fillColor`, `lineColor` and `lineWidth` define
     * the visual appearance of the markers. Other series types, like column
     * series, don't have markers, but have visual options on the series level
     * instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) Options for the corresponding navigator series if
     * `showInNavigator` is `true` for this series. Available options are the
     * same as any series, documented at plotOptions and series.
     *
     * These options are merged with options in navigator.series, and will take
     * precedence if the same option is defined both places.
     */
    navigatorOptions?: PlotSeriesOptions;
    /**
     * (Highcharts, Highstock) The color for the parts of the graph or points
     * that are below the threshold. Note that `zones` takes precedence over the
     * negative color. Using `negativeColor` is equivalent to applying a zone
     * with value of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) A separate color for the negative part of the area.
     *
     * In styled mode, a negative color is set with the `.highcharts-negative`
     * class name.
     */
    negativeFillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Opacity of a series parts: line, fill (e.g. area)
     * and dataLabels.
     */
    opacity?: number;
    /**
     * (Highcharts, Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highcharts, Highstock) Same as accessibility.pointDescriptionFormatter,
     * but for an individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highcharts, Highstock, Gantt) If no x values are given for the points in
     * a series, `pointInterval` defines the interval of the x values. For
     * example, if a series contains one value every decade starting from year
     * 0, set `pointInterval` to `10`. In true `datetime` axes, the
     * `pointInterval` is set in milliseconds.
     *
     * It can be also be combined with `pointIntervalUnit` to draw irregular
     * time intervals.
     *
     * Please note that this options applies to the _series data_, not the
     * interval of the axis ticks, which is independent.
     */
    pointInterval?: number;
    /**
     * (Highcharts, Highstock, Gantt) On datetime series, this allows for
     * setting the pointInterval to irregular time units, `day`, `month` and
     * `year`. A day is usually the same as 24 hours, but `pointIntervalUnit`
     * also takes the DST crossover into consideration when dealing with local
     * time. Combine this option with `pointInterval` to draw weeks, quarters, 6
     * months, 10 years etc.
     *
     * Please note that this options applies to the _series data_, not the
     * interval of the axis ticks, which is independent.
     */
    pointIntervalUnit?: OptionsPointIntervalUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Possible values: `"on"`, `"between"`,
     * `number`.
     *
     * In a column chart, when pointPlacement is `"on"`, the point will not
     * create any padding of the X axis. In a polar column chart this means that
     * the first column points directly north. If the pointPlacement is
     * `"between"`, the columns will be laid out between ticks. This is useful
     * for example for visualising an amount between two points in time or in a
     * certain sector of a polar chart.
     *
     * Since Highcharts 3.0.2, the point placement can also be numeric, where 0
     * is on the axis value, -0.5 is between this value and the previous, and
     * 0.5 is between this value and the next. Unlike the textual options,
     * numeric point placement options won't affect axis padding.
     *
     * Note that pointPlacement needs a pointRange to work. For column series
     * this is computed, but for line-type series it needs to be set.
     *
     * For the `xrange` series type and gantt charts, if the Y axis is a
     * category axis, the `pointPlacement` applies to the Y axis rather than the
     * (typically datetime) X axis.
     *
     * Defaults to `undefined` in cartesian charts, `"between"` in polar charts.
     */
    pointPlacement?: (number|string);
    /**
     * (Highstock) The width of each point on the x axis. For example in a
     * column chart with one value each day, the pointRange would be 1 day (= 24
     * * 3600
     *
     * * 1000 milliseconds). This is normally computed automatically, but this
     * option can be used to override the automatic value.
     */
    pointRange?: number;
    /**
     * (Highcharts, Highstock, Gantt) If no x values are given for the points in
     * a series, pointStart defines on what value to start. For example, if a
     * series contains one yearly value starting from 1945, set pointStart to
     * 1945.
     */
    pointStart?: number;
    /**
     * (Highcharts, Highstock) Whether to select the series initially. If
     * `showCheckbox` is true, the checkbox next to the series name in the
     * legend will be checked for a selected series.
     */
    selected?: boolean;
    /**
     * (Highcharts, Highstock) Whether to apply a drop shadow to the graph line.
     * Since 2.3 the shadow can be an object configuration containing `color`,
     * `offsetX`, `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock) If true, a checkbox is displayed next to the
     * legend item to allow selecting the series. The state of the checkbox is
     * determined by the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highcharts, Highstock) Whether to display this particular series or
     * series type in the legend. Standalone series are shown in legend by
     * default, and linked series are not. Since v7.2.0 it is possible to show
     * series that use colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) Whether or not to show the series in the navigator. Takes
     * precedence over navigator.baseSeries if defined.
     */
    showInNavigator?: boolean;
    /**
     * (Highcharts, Highstock) If set to `true`, the accessibility module will
     * skip past the points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    /**
     * (Highcharts, Highstock) Whether to stack the values of each series on top
     * of each other. Possible values are `undefined` to disable, `"normal"` to
     * stack by value or `"percent"`.
     *
     * When stacking is enabled, data must be sorted in ascending X order.
     *
     * Some stacking options are related to specific series types. In the
     * streamgraph series type, the stacking option is set to `"stream"`. The
     * second one is `"overlap"`, which only applies to waterfall series.
     */
    stacking?: OptionsStackingValue;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Sticky tracking of mouse events. When true, the
     * `mouseOut` event on a series isn't triggered until the mouse moves over
     * another series, or out of the plot area. When false, the `mouseOut` event
     * on a series is triggered when the mouse leaves the area around the
     * series' graph or markers. This also implies the tooltip when not shared.
     * When `stickyTracking` is false and `tooltip.shared` is false, the tooltip
     * will be hidden when moving the mouse between series. Defaults to true for
     * line and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The Y axis value to serve as the base for the
     * area, for distinguishing between values above and below a threshold. The
     * area between the graph and the threshold is filled.
     *
     * * If a number is given, the Y axis will scale to the threshold.
     *
     * * If `null`, the scaling behaves like a line series with fill between the
     * graph and the Y axis minimum.
     *
     * * If `Infinity` or `-Infinity`, the area between the graph and the
     * corresponding Y axis extreme is filled (since v6.1.0).
     */
    threshold?: (number|null);
    /**
     * (Highcharts, Highstock) A configuration object for the tooltip rendering
     * of each single series. Properties are inherited from tooltip, but only
     * the following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    /**
     * (Highcharts, Highstock) Whether the whole area or just the line should
     * respond to mouseover tooltips and other mouse or touch events.
     */
    trackByArea?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highcharts, Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highcharts, Highstock) Enable or disable the initial animation when a series
 * is displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotAreasplinerangeAnimationOptions {
    defer?: number;
}
/**
 * (Highcharts, Highstock) Enable or disable the initial animation when a series
 * is displayed for the `dataLabels`. The animation can also be set as a
 * configuration object. Please note that this option only applies to the
 * initial animation. For other animations, see chart.animation and the
 * animation parameter under the API methods. The following properties are
 * supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotAreasplinerangeDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock) The animation delay time in milliseconds. Set to
     * `0` renders dataLabel immediately. As `undefined` inherits defer time
     * from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotAreasplinerangeDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highcharts, Highstock) Style options for the guide box. The guide box has
 * one state by default, the `default` state.
 */
export interface PlotAreasplinerangeDragDropGuideBoxOptions {
    /**
     * (Highcharts, Highstock) Style options for the guide box default state.
     */
    default?: DragDropGuideBoxOptionsObject;
}
/**
 * (Highcharts, Highstock) The area spline range is a cartesian series type with
 * higher and lower Y values along an X axis. The area inside the range is
 * colored, and the graph outlining the area is a smoothed spline.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `areasplinerange` series are defined in
 * plotOptions.areasplinerange.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotAreasplinerangeOptions {
    /**
     * (Highcharts, Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highmaps) Whether all areas of the map defined in `mapData` should be
     * rendered. If `true`, areas which don't correspond to a data point, are
     * rendered as `null` points. If `false`, those areas are skipped.
     */
    allAreas?: boolean;
    /**
     * (Highcharts, Highstock) Allow this series' points to be selected by
     * clicking on the graphic (columns, point markers, pie slices, map areas
     * etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highcharts, Highstock) Enable or disable the initial animation when a
     * series is displayed. The animation can also be set as a configuration
     * object. Please note that this option only applies to the initial
     * animation of the series itself. For other animations, see chart.animation
     * and the animation parameter under the API methods. The following
     * properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotAreasplinerangeAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock) For some series, there is a limit that shuts down
     * initial animation by default when the total number of points in the chart
     * is too high. For example, for a column chart and its derivatives,
     * animation does not run if there is more than 250 points totally. To
     * disable this cap, set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock) An additional class name to apply to the series'
     * graphical elements. This option does not replace default class names of
     * the graphical element.
     */
    className?: string;
    /**
     * (Highcharts, Highstock) Disable this option to allow series rendering in
     * the whole plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highcharts, Highstock) The main color of the series. In line type series
     * it applies to the line and the point markers unless otherwise specified.
     * In bar type series it applies to the bars unless a color is specified per
     * point. The default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) When using dual or multiple color axes,
     * this number defines which colorAxis the particular series is connected
     * to. It refers to either the axis id or the index of the axis in the
     * colorAxis array, with 0 being the first. Set this option to false to
     * prevent a series from connecting to the default color axis.
     *
     * Since v7.2.0 the option can also be an axis id or an axis index instead
     * of a boolean flag.
     */
    colorAxis?: (boolean|number|string);
    /**
     * (Highcharts, Highstock) Styled mode only. A specific color index to use
     * for the series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Compare the values of the series against the first non-null,
     * non- zero value in the visible range. The y axis will show percentage or
     * absolute change depending on whether `compare` is set to `"percent"` or
     * `"value"`. When this is applied to multiple series, it allows comparing
     * the development of the series against each other. Adds a `change` field
     * to every point object.
     */
    compare?: string;
    /**
     * (Highstock) When compare is `percent`, this option dictates whether to
     * use 0 or 100 as the base of comparison.
     */
    compareBase?: (0|100);
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highcharts, Highstock) You can set the cursor to "pointer" if you have
     * click events attached to the series, to signal to the user that the
     * points and lines can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highcharts, Highstock) A reserved subspace to store options and values
     * for customized functionality. Here you can add additional data for your
     * own event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highcharts, Highstock) Name of the dash style to use for the graph, or
     * for some series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock) Extended data labels for range series types.
     * Range series data labels use no `x` and `y` options. Instead, they have
     * `xLow`, `xHigh`, `yLow` and `yHigh` options to allow the higher and lower
     * data label sets individually.
     */
    dataLabels?: (SeriesAreaRangeDataLabelsOptionsObject|Array<SeriesAreaRangeDataLabelsOptionsObject>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotAreasplinerangeDataSortingOptions);
    /**
     * (Highcharts, Highstock) A description of the series to add to the screen
     * reader information about the series.
     */
    description?: string;
    /**
     * (Highcharts, Highstock) The draggable-points module allows points to be
     * moved around or modified in the chart. In addition to the options
     * mentioned under the `dragDrop` API structure, the module fires three
     * events, point.dragStart, point.drag and point.drop.
     */
    dragDrop?: SeriesDragDropOptionsObject;
    /**
     * (Highcharts, Highstock) Enable or disable the mouse tracking for a
     * specific series. This includes point tooltips and click events on graphs
     * and points. For large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highcharts, Highstock) General event handlers for the series items.
     * These event hooks can also be attached to the series at run time using
     * the `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highcharts, Highstock) Fill color or gradient for the area. When `null`,
     * the series' `color` is used with the series' `fillOpacity`.
     *
     * In styled mode, the fill color can be set with the `.highcharts-area`
     * class name.
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Fill opacity for the area. When you set an
     * explicit `fillColor`, the `fillOpacity` is not applied. Instead, you
     * should define the opacity in the `fillColor` with an rgba color
     * definition. The `fillOpacity` setting, also the default setting,
     * overrides the alpha component of the `color` setting.
     *
     * In styled mode, the fill opacity can be set with the `.highcharts-area`
     * class name.
     */
    fillOpacity?: number;
    /**
     * (Highcharts, Highstock) Determines whether the series should look for the
     * nearest point in both dimensions or just the x-dimension when hovering
     * the series. Defaults to `'xy'` for scatter series and `'x'` for most
     * other series. If the data has duplicate x-values, it is recommended to
     * set this to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highcharts, Highstock) When set to `false` will prevent the series data
     * from being included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highmaps) What property to join the `mapData` to the value data. For
     * example, if joinBy is "code", the mapData items with a specific code is
     * merged into the data with the same code. For maps loaded from GeoJSON,
     * the keys may be held in each point's `properties` object.
     *
     * The joinBy option can also be an array of two values, where the first
     * points to a key in the `mapData`, and the second points to another key in
     * the `data`.
     *
     * When joinBy is `null`, the map items are joined by their position in the
     * array, which performs much better in maps with many data points. This is
     * the recommended option if you are printing more than a thousand data
     * points and have a backend that can preprocess the data into a parallel
     * array of the mapData.
     */
    joinBy?: (string|Array<string>);
    /**
     * (Highcharts, Highstock) An array specifying which option maps to which
     * key in the data point array. This makes it convenient to work with
     * unstructured data arrays from different sources.
     */
    keys?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) A separate color for the graph line. By default
     * the line takes the `color` of the series, but the lineColor setting
     * allows setting a separate color for the line without altering the
     * `fillColor`.
     *
     * In styled mode, the line stroke can be set with the `.highcharts-graph`
     * class name.
     */
    lineColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Pixel width of the arearange graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The id of another series to link to.
     * Additionally, the value can be ":previous" to link to the previous
     * series. When two series are linked, only the first one appears in the
     * legend. Toggling the visibility of this also toggles the linked series.
     *
     * If master series uses data sorting and linked series does not have its
     * own sorting definition, the linked series will be sorted in the same
     * order as the master one.
     */
    linkedTo?: string;
    /**
     * (Highcharts, Highstock) Options for the point markers of line-like
     * series. Properties like `fillColor`, `lineColor` and `lineWidth` define
     * the visual appearance of the markers. Other series types, like column
     * series, don't have markers, but have visual options on the series level
     * instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) Options for the corresponding navigator series if
     * `showInNavigator` is `true` for this series. Available options are the
     * same as any series, documented at plotOptions and series.
     *
     * These options are merged with options in navigator.series, and will take
     * precedence if the same option is defined both places.
     */
    navigatorOptions?: PlotSeriesOptions;
    /**
     * (Highcharts, Highstock) The color for the parts of the graph or points
     * that are below the threshold. Note that `zones` takes precedence over the
     * negative color. Using `negativeColor` is equivalent to applying a zone
     * with value of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) A separate color for the negative part of the area.
     *
     * In styled mode, a negative color is set with the `.highcharts-negative`
     * class name.
     */
    negativeFillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Opacity of a series parts: line, fill (e.g. area)
     * and dataLabels.
     */
    opacity?: number;
    /**
     * (Highcharts, Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highcharts, Highstock) Same as accessibility.pointDescriptionFormatter,
     * but for an individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highcharts, Highstock, Gantt) If no x values are given for the points in
     * a series, `pointInterval` defines the interval of the x values. For
     * example, if a series contains one value every decade starting from year
     * 0, set `pointInterval` to `10`. In true `datetime` axes, the
     * `pointInterval` is set in milliseconds.
     *
     * It can be also be combined with `pointIntervalUnit` to draw irregular
     * time intervals.
     *
     * Please note that this options applies to the _series data_, not the
     * interval of the axis ticks, which is independent.
     */
    pointInterval?: number;
    /**
     * (Highcharts, Highstock, Gantt) On datetime series, this allows for
     * setting the pointInterval to irregular time units, `day`, `month` and
     * `year`. A day is usually the same as 24 hours, but `pointIntervalUnit`
     * also takes the DST crossover into consideration when dealing with local
     * time. Combine this option with `pointInterval` to draw weeks, quarters, 6
     * months, 10 years etc.
     *
     * Please note that this options applies to the _series data_, not the
     * interval of the axis ticks, which is independent.
     */
    pointIntervalUnit?: OptionsPointIntervalUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Possible values: `"on"`, `"between"`,
     * `number`.
     *
     * In a column chart, when pointPlacement is `"on"`, the point will not
     * create any padding of the X axis. In a polar column chart this means that
     * the first column points directly north. If the pointPlacement is
     * `"between"`, the columns will be laid out between ticks. This is useful
     * for example for visualising an amount between two points in time or in a
     * certain sector of a polar chart.
     *
     * Since Highcharts 3.0.2, the point placement can also be numeric, where 0
     * is on the axis value, -0.5 is between this value and the previous, and
     * 0.5 is between this value and the next. Unlike the textual options,
     * numeric point placement options won't affect axis padding.
     *
     * Note that pointPlacement needs a pointRange to work. For column series
     * this is computed, but for line-type series it needs to be set.
     *
     * For the `xrange` series type and gantt charts, if the Y axis is a
     * category axis, the `pointPlacement` applies to the Y axis rather than the
     * (typically datetime) X axis.
     *
     * Defaults to `undefined` in cartesian charts, `"between"` in polar charts.
     */
    pointPlacement?: (number|string);
    /**
     * (Highstock) The width of each point on the x axis. For example in a
     * column chart with one value each day, the pointRange would be 1 day (= 24
     * * 3600
     *
     * * 1000 milliseconds). This is normally computed automatically, but this
     * option can be used to override the automatic value.
     */
    pointRange?: number;
    /**
     * (Highcharts, Highstock, Gantt) If no x values are given for the points in
     * a series, pointStart defines on what value to start. For example, if a
     * series contains one yearly value starting from 1945, set pointStart to
     * 1945.
     */
    pointStart?: number;
    /**
     * (Highcharts, Highstock) Whether to select the series initially. If
     * `showCheckbox` is true, the checkbox next to the series name in the
     * legend will be checked for a selected series.
     */
    selected?: boolean;
    /**
     * (Highcharts) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock) If true, a checkbox is displayed next to the
     * legend item to allow selecting the series. The state of the checkbox is
     * determined by the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highcharts, Highstock) Whether to display this particular series or
     * series type in the legend. Standalone series are shown in legend by
     * default, and linked series are not. Since v7.2.0 it is possible to show
     * series that use colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) Whether or not to show the series in the navigator. Takes
     * precedence over navigator.baseSeries if defined.
     */
    showInNavigator?: boolean;
    /**
     * (Highcharts, Highstock) If set to `true`, the accessibility module will
     * skip past the points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Sticky tracking of mouse events. When true, the
     * `mouseOut` event on a series isn't triggered until the mouse moves over
     * another series, or out of the plot area. When false, the `mouseOut` event
     * on a series is triggered when the mouse leaves the area around the
     * series' graph or markers. This also implies the tooltip when not shared.
     * When `stickyTracking` is false and `tooltip.shared` is false, the tooltip
     * will be hidden when moving the mouse between series. Defaults to true for
     * line and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The Y axis value to serve as the base for the
     * area, for distinguishing between values above and below a threshold. The
     * area between the graph and the threshold is filled.
     *
     * * If a number is given, the Y axis will scale to the threshold.
     *
     * * If `null`, the scaling behaves like a line series with fill between the
     * graph and the Y axis minimum.
     *
     * * If `Infinity` or `-Infinity`, the area between the graph and the
     * corresponding Y axis extreme is filled (since v6.1.0).
     */
    threshold?: (number|null);
    /**
     * (Highcharts, Highstock) A configuration object for the tooltip rendering
     * of each single series. Properties are inherited from tooltip, but only
     * the following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    /**
     * (Highcharts, Highstock) Whether the whole area or just the line should
     * respond to mouseover tooltips and other mouse or touch events.
     */
    trackByArea?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highcharts, Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highstock) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotAroonAnimationOptions {
    defer?: number;
}
/**
 * (Highstock) aroonDown line options.
 */
export interface PlotAroonAroonDownOptions {
    /**
     * (Highstock) Styles for an aroonDown line.
     */
    styles?: PlotAroonAroonDownStylesOptions;
}
/**
 * (Highstock) Styles for an aroonDown line.
 */
export interface PlotAroonAroonDownStylesOptions {
    /**
     * (Highstock) Color of the line. If not set, it's inherited from
     * plotOptions.aroon.color.
     */
    lineColor?: ColorString;
    /**
     * (Highstock) Pixel width of the line.
     */
    lineWidth?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotAroonDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data labels,
 * appearing next to each data point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 */
export interface PlotAroonDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|PlotAroonDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotAroonDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highstock) Aroon. This series requires the `linkedTo` option to be set and
 * should be loaded after the `stock/indicators/indicators.js`.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `aroon` series are defined in plotOptions.aroon.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotAroonOptions {
    /**
     * (Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highstock) Allow this series' points to be selected by clicking on the
     * graphic (columns, point markers, pie slices, map areas etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highstock) Enable or disable the initial animation when a series is
     * displayed. The animation can also be set as a configuration object.
     * Please note that this option only applies to the initial animation of the
     * series itself. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotAroonAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highstock) For some series, there is a limit that shuts down initial
     * animation by default when the total number of points in the chart is too
     * high. For example, for a column chart and its derivatives, animation does
     * not run if there is more than 250 points totally. To disable this cap,
     * set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highstock) aroonDown line options.
     */
    aroonDown?: PlotAroonAroonDownOptions;
    /**
     * (Highstock) Sets the color blending in the boost module.
     */
    boostBlending?: OptionsBoostBlendingValue;
    /**
     * (Highstock) Set the point threshold for when a series should enter boost
     * mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    /**
     * (Highstock) An additional class name to apply to the series' graphical
     * elements. This option does not replace default class names of the
     * graphical element.
     */
    className?: string;
    /**
     * (Highstock) Disable this option to allow series rendering in the whole
     * plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highstock) The main color of the series. In line type series it applies
     * to the line and the point markers unless otherwise specified. In bar type
     * series it applies to the bars unless a color is specified per point. The
     * default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Styled mode only. A specific color index to use for the
     * series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highstock) Whether to compare indicator to the main series values or
     * indicator values.
     */
    compareToMain?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highstock) You can set the cursor to "pointer" if you have click events
     * attached to the series, to signal to the user that the points and lines
     * can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highstock) A reserved subspace to store options and values for
     * customized functionality. Here you can add additional data for your own
     * event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highstock) Name of the dash style to use for the graph, or for some
     * series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data
     * labels, appearing next to each data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     */
    dataLabels?: (PlotAroonDataLabelsOptions|Array<PlotAroonDataLabelsOptions>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotAroonDataSortingOptions);
    /**
     * (Highstock) A description of the series to add to the screen reader
     * information about the series.
     */
    description?: string;
    /**
     * (Highstock) Enable or disable the mouse tracking for a specific series.
     * This includes point tooltips and click events on graphs and points. For
     * large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highstock) General event handlers for the series items. These event
     * hooks can also be attached to the series at run time using the
     * `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highstock) Determines whether the series should look for the nearest
     * point in both dimensions or just the x-dimension when hovering the
     * series. Defaults to `'xy'` for scatter series and `'x'` for most other
     * series. If the data has duplicate x-values, it is recommended to set this
     * to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highstock) When set to `false` will prevent the series data from being
     * included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The main series ID that indicator will be
     * based on. Required for this indicator.
     */
    linkedTo?: string;
    /**
     * (Highstock) Options for the point markers of line-like series. Properties
     * like `fillColor`, `lineColor` and `lineWidth` define the visual
     * appearance of the markers. Other series types, like column series, don't
     * have markers, but have visual options on the series level instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) The name of the series as shown in the legend, tooltip etc.
     * If not set, it will be based on a technical indicator type and default
     * params.
     */
    name?: string;
    /**
     * (Highstock) The color for the parts of the graph or points that are below
     * the threshold. Note that `zones` takes precedence over the negative
     * color. Using `negativeColor` is equivalent to applying a zone with value
     * of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Opacity of a series parts: line, fill (e.g. area) and
     * dataLabels.
     */
    opacity?: number;
    /**
     * (Highstock) Paramters used in calculation of aroon series points.
     */
    params?: PlotAroonParamsOptions;
    /**
     * (Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highstock) Same as accessibility.pointDescriptionFormatter, but for an
     * individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highstock) Whether to select the series initially. If `showCheckbox` is
     * true, the checkbox next to the series name in the legend will be checked
     * for a selected series.
     */
    selected?: boolean;
    /**
     * (Highstock) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) If true, a checkbox is displayed next to the legend item to
     * allow selecting the series. The state of the checkbox is determined by
     * the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highstock) Whether to display this particular series or series type in
     * the legend. Standalone series are shown in legend by default, and linked
     * series are not. Since v7.2.0 it is possible to show series that use
     * colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) If set to `true`, the accessibility module will skip past the
     * points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     */
    step?: OptionsStepValue;
    /**
     * (Highstock) Sticky tracking of mouse events. When true, the `mouseOut`
     * event on a series isn't triggered until the mouse moves over another
     * series, or out of the plot area. When false, the `mouseOut` event on a
     * series is triggered when the mouse leaves the area around the series'
     * graph or markers. This also implies the tooltip when not shared. When
     * `stickyTracking` is false and `tooltip.shared` is false, the tooltip will
     * be hidden when moving the mouse between series. Defaults to true for line
     * and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The threshold, also called zero level or base
     * level. For line type series this is only used in conjunction with
     * negativeColor.
     */
    threshold?: number;
    /**
     * (Highstock) A configuration object for the tooltip rendering of each
     * single series. Properties are inherited from tooltip, but only the
     * following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highstock) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotAroonoscillatorAnimationOptions {
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotAroonoscillatorDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data labels,
 * appearing next to each data point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 */
export interface PlotAroonoscillatorDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|PlotAroonoscillatorDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotAroonoscillatorDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highstock) Aroon Oscillator. This series requires the `linkedTo` option to
 * be set and should be loaded after the `stock/indicators/indicators.js` and
 * `stock/indicators/aroon.js`.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `aroonoscillator` series are defined in
 * plotOptions.aroonoscillator.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotAroonoscillatorOptions {
    /**
     * (Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highstock) Allow this series' points to be selected by clicking on the
     * graphic (columns, point markers, pie slices, map areas etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highstock) Enable or disable the initial animation when a series is
     * displayed. The animation can also be set as a configuration object.
     * Please note that this option only applies to the initial animation of the
     * series itself. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotAroonoscillatorAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highstock) For some series, there is a limit that shuts down initial
     * animation by default when the total number of points in the chart is too
     * high. For example, for a column chart and its derivatives, animation does
     * not run if there is more than 250 points totally. To disable this cap,
     * set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highstock) Sets the color blending in the boost module.
     */
    boostBlending?: OptionsBoostBlendingValue;
    /**
     * (Highstock) Set the point threshold for when a series should enter boost
     * mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    /**
     * (Highstock) An additional class name to apply to the series' graphical
     * elements. This option does not replace default class names of the
     * graphical element.
     */
    className?: string;
    /**
     * (Highstock) Disable this option to allow series rendering in the whole
     * plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highstock) The main color of the series. In line type series it applies
     * to the line and the point markers unless otherwise specified. In bar type
     * series it applies to the bars unless a color is specified per point. The
     * default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Styled mode only. A specific color index to use for the
     * series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highstock) Whether to compare indicator to the main series values or
     * indicator values.
     */
    compareToMain?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highstock) You can set the cursor to "pointer" if you have click events
     * attached to the series, to signal to the user that the points and lines
     * can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highstock) A reserved subspace to store options and values for
     * customized functionality. Here you can add additional data for your own
     * event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highstock) Name of the dash style to use for the graph, or for some
     * series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data
     * labels, appearing next to each data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     */
    dataLabels?: (PlotAroonoscillatorDataLabelsOptions|Array<PlotAroonoscillatorDataLabelsOptions>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotAroonoscillatorDataSortingOptions);
    /**
     * (Highstock) A description of the series to add to the screen reader
     * information about the series.
     */
    description?: string;
    /**
     * (Highstock) Enable or disable the mouse tracking for a specific series.
     * This includes point tooltips and click events on graphs and points. For
     * large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highstock) General event handlers for the series items. These event
     * hooks can also be attached to the series at run time using the
     * `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highstock) Determines whether the series should look for the nearest
     * point in both dimensions or just the x-dimension when hovering the
     * series. Defaults to `'xy'` for scatter series and `'x'` for most other
     * series. If the data has duplicate x-values, it is recommended to set this
     * to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highstock) When set to `false` will prevent the series data from being
     * included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The main series ID that indicator will be
     * based on. Required for this indicator.
     */
    linkedTo?: string;
    /**
     * (Highstock) Options for the point markers of line-like series. Properties
     * like `fillColor`, `lineColor` and `lineWidth` define the visual
     * appearance of the markers. Other series types, like column series, don't
     * have markers, but have visual options on the series level instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) The name of the series as shown in the legend, tooltip etc.
     * If not set, it will be based on a technical indicator type and default
     * params.
     */
    name?: string;
    /**
     * (Highstock) The color for the parts of the graph or points that are below
     * the threshold. Note that `zones` takes precedence over the negative
     * color. Using `negativeColor` is equivalent to applying a zone with value
     * of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Opacity of a series parts: line, fill (e.g. area) and
     * dataLabels.
     */
    opacity?: number;
    /**
     * (Highstock) Paramters used in calculation of aroon oscillator series
     * points.
     */
    params?: PlotAroonoscillatorParamsOptions;
    /**
     * (Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highstock) Same as accessibility.pointDescriptionFormatter, but for an
     * individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highstock) Whether to select the series initially. If `showCheckbox` is
     * true, the checkbox next to the series name in the legend will be checked
     * for a selected series.
     */
    selected?: boolean;
    /**
     * (Highstock) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) If true, a checkbox is displayed next to the legend item to
     * allow selecting the series. The state of the checkbox is determined by
     * the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highstock) Whether to display this particular series or series type in
     * the legend. Standalone series are shown in legend by default, and linked
     * series are not. Since v7.2.0 it is possible to show series that use
     * colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) If set to `true`, the accessibility module will skip past the
     * points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     */
    step?: OptionsStepValue;
    /**
     * (Highstock) Sticky tracking of mouse events. When true, the `mouseOut`
     * event on a series isn't triggered until the mouse moves over another
     * series, or out of the plot area. When false, the `mouseOut` event on a
     * series is triggered when the mouse leaves the area around the series'
     * graph or markers. This also implies the tooltip when not shared. When
     * `stickyTracking` is false and `tooltip.shared` is false, the tooltip will
     * be hidden when moving the mouse between series. Defaults to true for line
     * and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The threshold, also called zero level or base
     * level. For line type series this is only used in conjunction with
     * negativeColor.
     */
    threshold?: number;
    /**
     * (Highstock) A configuration object for the tooltip rendering of each
     * single series. Properties are inherited from tooltip, but only the
     * following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highstock) Paramters used in calculation of aroon oscillator series points.
 */
export interface PlotAroonoscillatorParamsOptions {
    /**
     * (Highstock) Period for Aroon Oscillator
     */
    period?: number;
}
/**
 * (Highstock) Paramters used in calculation of aroon series points.
 */
export interface PlotAroonParamsOptions {
    /**
     * (Highstock) Period for Aroon indicator
     */
    period?: number;
}
/**
 * (Highstock) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotAtrAnimationOptions {
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotAtrDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data labels,
 * appearing next to each data point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 */
export interface PlotAtrDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: (AlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|PlotAtrDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: (VerticalAlignValue|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotAtrDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highstock) Average true range indicator (ATR). This series requires
 * `linkedTo` option to be set.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `atr` series are defined in plotOptions.atr.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotAtrOptions {
    /**
     * (Highstock) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highstock) Allow this series' points to be selected by clicking on the
     * graphic (columns, point markers, pie slices, map areas etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highstock) Enable or disable the initial animation when a series is
     * displayed. The animation can also be set as a configuration object.
     * Please note that this option only applies to the initial animation of the
     * series itself. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotAtrAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highstock) For some series, there is a limit that shuts down initial
     * animation by default when the total number of points in the chart is too
     * high. For example, for a column chart and its derivatives, animation does
     * not run if there is more than 250 points totally. To disable this cap,
     * set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highstock) Sets the color blending in the boost module.
     */
    boostBlending?: OptionsBoostBlendingValue;
    /**
     * (Highstock) Set the point threshold for when a series should enter boost
     * mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     */
    borderWidth?: number;
    /**
     * (Highstock) An additional class name to apply to the series' graphical
     * elements. This option does not replace default class names of the
     * graphical element.
     */
    className?: string;
    /**
     * (Highstock) Disable this option to allow series rendering in the whole
     * plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highstock) The main color of the series. In line type series it applies
     * to the line and the point markers unless otherwise specified. In bar type
     * series it applies to the bars unless a color is specified per point. The
     * default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Styled mode only. A specific color index to use for the
     * series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highstock) Compare the values of the series against the first non-null,
     * non- zero value in the visible range. The y axis will show percentage or
     * absolute change depending on whether `compare` is set to `"percent"` or
     * `"value"`. When this is applied to multiple series, it allows comparing
     * the development of the series against each other. Adds a `change` field
     * to every point object.
     */
    compare?: string;
    /**
     * (Highstock) When compare is `percent`, this option dictates whether to
     * use 0 or 100 as the base of comparison.
     */
    compareBase?: (0|100);
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Highstock) Whether to compare indicator to the main series values or
     * indicator values.
     */
    compareToMain?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highstock) You can set the cursor to "pointer" if you have click events
     * attached to the series, to signal to the user that the points and lines
     * can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highstock) A reserved subspace to store options and values for
     * customized functionality. Here you can add additional data for your own
     * event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highstock) Name of the dash style to use for the graph, or for some
     * series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data
     * labels, appearing next to each data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     */
    dataLabels?: (PlotAtrDataLabelsOptions|Array<PlotAtrDataLabelsOptions>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotAtrDataSortingOptions);
    /**
     * (Highstock) A description of the series to add to the screen reader
     * information about the series.
     */
    description?: string;
    /**
     * (Highstock) Enable or disable the mouse tracking for a specific series.
     * This includes point tooltips and click events on graphs and points. For
     * large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highstock) General event handlers for the series items. These event
     * hooks can also be attached to the series at run time using the
     * `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highstock) Determines whether the series should look for the nearest
     * point in both dimensions or just the x-dimension when hovering the
     * series. Defaults to `'xy'` for scatter series and `'x'` for most other
     * series. If the data has duplicate x-values, it is recommended to set this
     * to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `"relative"` (default), a gap size of 5 means that
     * if the distance between two points is greater than 5 times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `"value"`, the gap is based on absolute axis
     * values, which on a datetime axis is milliseconds. This also applies to
     * the navigator series that inherits gap options from the base series.
     */
    gapUnit?: OptionsGapUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highstock) When set to `false` will prevent the series data from being
     * included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     */
    linecap?: SeriesLinecapValue;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The main series ID that indicator will be
     * based on. Required for this indicator.
     */
    linkedTo?: string;
    /**
     * (Highstock) Options for the point markers of line-like series. Properties
     * like `fillColor`, `lineColor` and `lineWidth` define the visual
     * appearance of the markers. Other series types, like column series, don't
     * have markers, but have visual options on the series level instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     */
    marker?: PointMarkerOptionsObject;
    /**
     * (Highstock) The name of the series as shown in the legend, tooltip etc.
     * If not set, it will be based on a technical indicator type and default
     * params.
     */
    name?: string;
    /**
     * (Highstock) The color for the parts of the graph or points that are below
     * the threshold. Note that `zones` takes precedence over the negative
     * color. Using `negativeColor` is equivalent to applying a zone with value
     * of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Opacity of a series parts: line, fill (e.g. area) and
     * dataLabels.
     */
    opacity?: number;
    /**
     * (Highstock) Paramters used in calculation of regression series' points.
     */
    params?: PlotAtrParamsOptions;
    /**
     * (Highstock) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highstock) Same as accessibility.pointDescriptionFormatter, but for an
     * individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highstock) Whether to select the series initially. If `showCheckbox` is
     * true, the checkbox next to the series name in the legend will be checked
     * for a selected series.
     */
    selected?: boolean;
    /**
     * (Highstock) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) If true, a checkbox is displayed next to the legend item to
     * allow selecting the series. The state of the checkbox is determined by
     * the `selected` option.
     */
    showCheckbox?: boolean;
    /**
     * (Highstock) Whether to display this particular series or series type in
     * the legend. Standalone series are shown in legend by default, and linked
     * series are not. Since v7.2.0 it is possible to show series that use
     * colorAxis by setting this option to `true`.
     */
    showInLegend?: boolean;
    /**
     * (Highstock) If set to `true`, the accessibility module will skip past the
     * points in this series for keyboard navigation.
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     */
    softThreshold?: boolean;
    states?: SeriesStatesOptionsObject;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     */
    step?: OptionsStepValue;
    /**
     * (Highstock) Sticky tracking of mouse events. When true, the `mouseOut`
     * event on a series isn't triggered until the mouse moves over another
     * series, or out of the plot area. When false, the `mouseOut` event on a
     * series is triggered when the mouse leaves the area around the series'
     * graph or markers. This also implies the tooltip when not shared. When
     * `stickyTracking` is false and `tooltip.shared` is false, the tooltip will
     * be hidden when moving the mouse between series. Defaults to true for line
     * and area type series, but to false for columns, pies etc.
     *
     * **Note:** The boost module will force this option because of technical
     * limitations.
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The threshold, also called zero level or base
     * level. For line type series this is only used in conjunction with
     * negativeColor.
     */
    threshold?: number;
    /**
     * (Highstock) A configuration object for the tooltip rendering of each
     * single series. Properties are inherited from tooltip, but only the
     * following properties can be defined on a series level.
     */
    tooltip?: SeriesTooltipOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * Note: In boost mode turbo threshold is forced. Only array of numbers or
     * two dimensional arrays are allowed.
     */
    turboThreshold?: number;
    /**
     * (Highstock) Set the initial visibility of the series.
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     */
    zones?: Array<SeriesZonesOptionsObject>;
}
/**
 * (Highstock) Paramters used in calculation of regression series' points.
 */
export interface PlotAtrParamsOptions {
    /**
     * (Highstock) The point index which indicator calculations will base. For
     * example using OHLC data, index=2 means the indicator will be calculated
     * using Low values.
     */
    index?: number;
    /**
     * (Highstock) The base period for indicator calculations. This is the
     * number of data points which are taken into account for the indicator
     * calculations.
     */
    period?: number;
}
/**
 * (Highcharts) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 */
export interface PlotBarAnimationOptions {
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
 * animation when a series is displayed for the `dataLabels`. The animation can
 * also be set as a configuration object. Please note that this option only
 * applies to the initial animation. For other animations, see chart.animation
 * and the animation parameter under the API methods. The following properties
 * are supported:
 *
 * - `defer`: The animation delay time in milliseconds.
 */
export interface PlotBarDataLabelsAnimationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The animation delay time in
     * milliseconds. Set to `0` renders dataLabel immediately. As `undefined`
     * inherits defer time from the series.animation.defer.
     */
    defer?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data labels,
 * appearing next to each data point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 */
export interface PlotBarDataLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the data label
     * compared to the point. If `right`, the right side of the label should be
     * touching the point. For points with an extent, like columns, the
     * alignments also dictates how to align it inside the box, as given with
     * the inside option. Can be one of `left`, `center` or `right`.
     */
    align?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow data labels to
     * overlap. To make the labels less sensitive for overlapping, the
     * dataLabels.padding can be set to 0.
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the initial
     * animation when a series is displayed for the `dataLabels`. The animation
     * can also be set as a configuration object. Please note that this option
     * only applies to the initial animation. For other animations, see
     * chart.animation and the animation parameter under the API methods. The
     * following properties are supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     */
    animation?: (boolean|PlotBarDataLabelsAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the data label.
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the data
     * label. Defaults to `undefined`.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the data label.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the data label.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for the data label.
     * Particularly in styled mode, this can be used to give each series' or
     * point's data label unique styling. In addition to this option, a default
     * color class name is added so that we can give the labels a contrast text
     * shadow.
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text color for the data
     * labels. Defaults to `undefined`. For certain series types, like column or
     * map, the data labels can be drawn inside the points. In this case the
     * data label will be drawn with maximum contrast by default. Additionally,
     * it will be given a `text-outline` style with the opposite color, to
     * further increase the contrast. This can be overridden by setting the
     * `text-outline` style to `none` in the `dataLabels.style` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide data labels that
     * are outside the plot area. By default, the data label is moved inside the
     * plot area according to the overflow option.
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished. Setting to
     * `false` renders the data label immediately. If set to `true` inherits the
     * defer time set in plotOptions.series.animation.
     */
    defer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the data
     * labels.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A declarative filter to control
     * of which data labels to display. The declarative filter is designed for
     * use when callback functions are not available, like when the chart
     * options require a pure JSON structure or for use with graphical editors.
     * For programmatic control, use the `formatter` instead, and return
     * `undefined` to disable a single data label.
     */
    filter?: DataLabelsFilterOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label. Available variables are the same as for `formatter`.
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the data label. Note that if a `format` is defined, the format
     * takes precedence and the formatter is ignored.
     */
    formatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) For points with an extent, like
     * columns or map areas, whether to align the data label inside the box or
     * to the actual value point. Defaults to `false` in most cases, `true` in
     * stacked columns.
     */
    inside?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Format for points with the value
     * of null. Works analogously to format. `nullFormat` can be applied only to
     * series which support displaying null points.
     */
    nullFormat?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function
     * that defines formatting for points with the value of null. Works
     * analogously to formatter. `nullPointFormatter` can be applied only to
     * series which support displaying null points.
     */
    nullFormatter?: DataLabelsFormatterCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle data labels that
     * flow outside the plot area. The default is `"justify"`, which aligns them
     * inside the plot area. For columns and bars, this means it will be moved
     * inside the bar. To display data labels outside the plot area, set `crop`
     * to `false` and `overflow` to `"allow"`.
     */
    overflow?: DataLabelsOverflowValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the `borderWidth` or
     * the `backgroundColor` is set, this is the padding within the box.
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Aligns data labels relative to
     * points. If `center` alignment is not possible, it defaults to `right`.
     */
    position?: AlignValue;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text rotation in degrees. Note
     * that due to a more complex structure, backgrounds, borders and padding
     * will be lost on a rotated data label.
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. Works
     * best with `borderWidth` or `backgroundColor`. Since 2.3 the shadow can be
     * an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the label. The
     * default `color` setting is `"contrast"`, which is a pseudo color that
     * Highcharts picks up and applies the maximum contrast to the underlying
     * point item, for example the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for a label text which
     * should follow marker's shape. Border and background are disabled for a
     * label that follows a path.
     *
     * **Note:** Only SVG-based renderer supports this option. Setting `useHTML`
     * to true will disable this option.
     */
    textPath?: DataLabelsTextPathOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the labels.
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of a data
     * label. Can be one of `top`, `middle` or `bottom`. The default value
     * depends on the data, for instance in a column chart, the label is above
     * positive values and below negative values.
     */
    verticalAlign?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point in pixels.
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point in pixels.
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the data labels.
     * The default Z index puts it above the series. Use a Z index of 2 to
     * display it behind the series.
     */
    z?: number;
}
/**
 * (Highcharts, Highstock) Options for the series data sorting.
 */
export interface PlotBarDataSortingOptions {
    /**
     * (Highcharts, Highstock) Enable or disable data sorting for the series.
     * Use xAxis.reversed to change the sorting order.
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Whether to allow matching points by name in an
     * update. If this option is disabled, points will be matched by order.
     */
    matchByName?: boolean;
    /**
     * (Highcharts, Highstock) Determines what data value should be used to sort
     * by.
     */
    sortKey?: string;
}
/**
 * (Highcharts) Style options for the guide box. The guide box has one state by
 * default, the `default` state.
 */
export interface PlotBarDragDropGuideBoxOptions {
    /**
     * (Highcharts) Style options for the guide box default state.
     */
    default?: DragDropGuideBoxOptionsObject;
}
/**
 * (Highcharts) A bar series is a special type of column series where the
 * columns are horizontal.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `bar` series are defined in plotOptions.bar.
 *
 * 3. Options for one single series are given in the series instance array. (see
 * online documentation for example)
 */
export interface PlotBarOptions {
    /**
     * (Highcharts) Accessibility options for a series.
     */
    accessibility?: SeriesAccessibilityOptionsObject;
    /**
     * (Highmaps) Whether all areas of the map defined in `mapData` should be
     * rendered. If `true`, areas which don't correspond to a data point, are
     * rendered as `null` points. If `false`, those areas are skipped.
     */
    allAreas?: boolean;
    /**
     * (Highcharts) Allow this series' points to be selected by clicking on the
     * graphic (columns, point markers, pie slices, map areas etc).
     *
     * The selected points can be handled by point select and unselect events,
     * or collectively by the getSelectedPoints function.
     *
     * And alternative way of selecting points is through dragging.
     */
    allowPointSelect?: boolean;
    /**
     * (Highcharts) Enable or disable the initial animation when a series is
     * displayed. The animation can also be set as a configuration object.
     * Please note that this option only applies to the initial animation of the
     * series itself. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `defer`: The animation delay time in milliseconds.
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     */
    animation?: (boolean|PlotBarAnimationOptions|Partial<AnimationOptionsObject>);
    /**
     * (Highcharts) For some series, there is a limit that shuts down initial
     * animation by default when the total number of points in the chart is too
     * high. For example, for a column chart and its derivatives, animation does
     * not run if there is more than 250 points totally. To disable this cap,
     * set `animationLimit` to `Infinity`.
     */
    animationLimit?: number;
    /**
     * (Highcharts) Sets the color blending in the boost module.
     */
    boostBlending?: OptionsBoostBlendingValue;
    /**
     * (Highcharts) Set the point threshold for when a series should enter boost
     * mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     */
    boostThreshold?: number;
    /**
     * (Highcharts, Highstock, Gantt) The color of the border surrounding each
     * column or bar.
     *
     * In styled mode, the border stroke can be set with the `.highcharts-point`
     * rule.
     */
    borderColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Gantt) The corner radius of the border
     * surrounding each column or bar.
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Gantt) The width of the border surrounding each
     * column or bar. Defaults to `1` when there is room for a border, but to
     * `0` when the columns are so dense that a border would cover the next
     * column.
     *
     * In styled mode, the stroke width can be set with the `.highcharts-point`
     * rule.
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) When `true`, the columns will center in
     * the category, ignoring null or missing points. When `false`, space will
     * be reserved for null or missing points.
     */
    centerInCategory?: boolean;
    /**
     * (Highcharts) An additional class name to apply to the series' graphical
     * elements. This option does not replace default class names of the
     * graphical element.
     */
    className?: string;
    /**
     * (Highcharts) Disable this option to allow series rendering in the whole
     * plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     */
    clip?: boolean;
    /**
     * (Highcharts) The main color of the series. In line type series it applies
     * to the line and the point markers unless otherwise specified. In bar type
     * series it applies to the bars unless a color is specified per point. The
     * default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) When using dual or multiple color axes,
     * this number defines which colorAxis the particular series is connected
     * to. It refers to either the axis id or the index of the axis in the
     * colorAxis array, with 0 being the first. Set this option to false to
     * prevent a series from connecting to the default color axis.
     *
     * Since v7.2.0 the option can also be an axis id or an axis index instead
     * of a boolean flag.
     */
    colorAxis?: (boolean|number|string);
    /**
     * (Highcharts, Highstock, Gantt) When using automatic point colors pulled
     * from the global colors or series-specific plotOptions.column.colors
     * collections, this option determines whether the chart should receive one
     * color per series or one color per point.
     *
     * In styled mode, the `colors` or `series.colors` arrays are not supported,
     * and instead this option gives the points individual color class names on
     * the form `highcharts-color-{n}`.
     */
    colorByPoint?: boolean;
    /**
     * (Highcharts) Styled mode only. A specific color index to use for the
     * series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     */
    colorIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Determines what data value should be
     * used to calculate point color if `colorAxis` is used. Requires to set
     * `min` and `max` if some custom point property is used or if approximation
     * for data grouping is set to `'sum'`.
     */
    colorKey?: string;
    /**
     * (Highcharts, Highstock, Gantt) A series specific or series type specific
     * color set to apply instead of the global colors when colorByPoint is
     * true.
     */
    colors?: Array<(ColorString|GradientColorObject|PatternObject)>;
    /**
     * (Highstock) Compare the values of the series against the first non-null,
     * non- zero value in the visible range. The y axis will show percentage or
     * absolute change depending on whether `compare` is set to `"percent"` or
     * `"value"`. When this is applied to multiple series, it allows comparing
     * the development of the series against each other. Adds a `change` field
     * to every point object.
     */
    compare?: string;
    /**
     * (Highstock) When compare is `percent`, this option dictates whether to
     * use 0 or 100 as the base of comparison.
     */
    compareBase?: (0|100);
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point **before**
     * the range.
     *
     * In other words, this flag determines if first point within the visible
     * range will have 0% (`compareStart=true`) or should have been already
     * calculated according to the previous point (`compareStart=false`).
     */
    compareStart?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     */
    connectors?: SeriesConnectorsOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) When true, each point or column edge is
     * rounded to its nearest pixel in order to render sharp on screen. In some
     * cases, when there are a lot of densely packed columns, this leads to
     * visible difference in column widths or distance between columns. In these
     * cases, setting `crisp` to `false` may look better, even though each
     * column is rendered blurry.
     */
    crisp?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) When the series contains less points than
     * the crop threshold, all points are drawn, event if the points fall
     * outside the visible plot area at the current zoom. The advantage of
     * drawing all points (including markers and columns), is that animation is
     * performed on updates. On the other hand, when the series contains more
     * points than the crop threshold, the series data is cropped to only
     * contain points that fall within the plot area. The advantage of cropping
     * away invisible points is to increase performance on large series.
     */
    cropThreshold?: number;
    /**
     * (Highcharts) You can set the cursor to "pointer" if you have click events
     * attached to the series, to signal to the user that the points and lines
     * can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     */
    cursor?: (string|CursorValue);
    /**
     * (Highcharts) A reserved subspace to store options and values for
     * customized functionality. Here you can add additional data for your own
     * event callbacks and formatter callbacks.
     */
    custom?: Dictionary<any>;
    /**
     * (Highcharts) Name of the dash style to use for the graph, or for some
     * series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     */
    dashStyle?: DashStyleValue;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     */
    dataGrouping?: DataGroupingOptionsObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the series data
     * labels, appearing next to each data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     */
    dataLabels?: (PlotBarDataLabelsOptions|Array<PlotBarDataLabelsOptions>);
    /**
     * (Highcharts, Highstock) Options for the series data sorting.
     */
    dataSorting?: (DataSortingOptionsObject|PlotBarDataSortingOptions);
    /**
     * (Highcharts) Depth of the columns in a 3D column chart.
     */
    depth?: number;
    /**
     * (Highcharts) A description of the series to add to the screen reader
     * information about the series.
     */
    description?: string;
    /**
     * (Highcharts) The draggable-points module allows points to be moved around
     * or modified in the chart. In addition to the options mentioned under the
     * `dragDrop` API structure, the module fires three events, point.dragStart,
     * point.drag and point.drop.
     */
    dragDrop?: SeriesDragDropOptionsObject;
    /**
     * (Highcharts) 3D columns only. The color of the edges. Similar to
     * `borderColor`, except it defaults to the same color as the column.
     */
    edgeColor?: ColorString;
    /**
     * (Highcharts) 3D columns only. The width of the colored edges.
     */
    edgeWidth?: number;
    /**
     * (Highcharts) Enable or disable the mouse tracking for a specific series.
     * This includes point tooltips and click events on graphs and points. For
     * large datasets it improves performance.
     */
    enableMouseTracking?: boolean;
    /**
     * (Highcharts) General event handlers for the series items. These event
     * hooks can also be attached to the series at run time using the
     * `Highcharts.addEvent` function.
     */
    events?: SeriesEventsOptionsObject;
    /**
     * (Highcharts) Determines whether the series should look for the nearest
     * point in both dimensions or just the x-dimension when hovering the
     * series. Defaults to `'xy'` for scatter series and `'x'` for most other
     * series. If the data has duplicate x-values, it is recommended to set this
     * to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     */
    findNearestPointBy?: OptionsFindNearestPointByValue;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to group non-stacked columns or to
     * let them render independent of each other. Non-grouped columns will be
     * laid out individually and overlap each other.
     */
    grouping?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Padding between each value groups, in x
     * axis units.
     */
    groupPadding?: number;
    /**
     * (Highcharts) The spacing between columns on the Z Axis in a 3D chart.
     */
    groupZPadding?: number;
    /**
     * (Highcharts) When set to `false` will prevent the series data from being
     * included in any form of data export.
     *
     * Since version 6.0.0 until 7.1.0 the option was existing undocumented as
     * `includeInCSVExport`.
     */
    includeInDataExport?: boolean;
    /**
     * (Highmaps) What property to join the `mapData` to the value data. For
     * example, if joinBy is "code", the mapData items with a specific code is
     * merged into the data with the same code. For maps loaded from GeoJSON,
     * the keys may be held in each point's `properties` object.
     *
     * The joinBy option can also be an array of two values, where the first
     * points to a key in the `mapData`, and the second points to another key in
     * the `data`.
     *
     * When joinBy is `null`, the map items are joined by their position in the
     * array, which performs much better in maps with many data points. This is
     * the recommended option if you are printing more than a thousand data
     * points and have a backend that can preprocess the data into a parallel
     * array of the mapData.
     */
    joinBy?: (string|Array<string>);
    /**
     * (Highcharts) An array specifying which option maps to which key in the
     * data point array. This makes it convenient to work with unstructured data
     * arrays from different sources.
     */
    keys?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     */
    label?: SeriesLabelOptionsObject;
    /**
     * (Highstock) The line marks the last price from all points.
     */
    lastPrice?: SeriesLastPriceOptionsObject;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     */
    lastVisiblePrice?: SeriesLastVisiblePriceOptionsObject;
    /**
     * (Highcharts, Highstock, Gantt) The id of another series to link to.
     * Additionally, the value can be ":previous" to link to the previous
     * series. When two series are linked, only the first one appears in the
     * legend. Toggling the visibility of this also toggles the linked series.
     *
     * If master series uses data sorting and linked series does not have its
     * own sorting definition, the linked series will be sorted in the same
     * order as the master one.
     */
    linkedTo?: string;
    /**
     * (Highcharts, Highstock, Gantt) The maximum allowed pixel width for a
     * column, translated to the height of a bar in a bar chart. This prevents
     * the columns from becoming too wide when there is a small number of points
     * in the chart.
     */
    maxPointWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The minimal height for a column or width
     * for a bar. By default, 0 values are not shown. To visualize a 0 (or close
     * to zero) point, set the minimal point length to a pixel value like 3\. In
     * stacked column charts, minPointLength might not be respected for tightly
     * packed values.
     */
    minPointLength?: number;
    /**
     * (Highstock) Options for the corresponding navigator series if
     * `showInNavigator` is `true` for this series. Available options are the
     * same as any series, documented at plotOptions and series.
     *
     * These options are merged with options in navigator.series, and will take
     * precedence if the same option is defined both places.
     */
    navigatorOptions?: PlotSeriesOptions;
    /**
     * (Highcharts) The color for the parts of the graph or points that are
     * below the threshold. Note that `zones` takes precedence over the negative
     * color. Using `negativeColor` is equivalent to applying a zone with value
     * of 0.
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) Opacity of a series parts: line, fill (e.g. area) and
     * dataLabels.
     */
    opacity?: number;
    /**
     * (Highcharts) Properties for each single point.
     */
    point?: PlotSeriesPointOptions;
    /**
     * (Highcharts) Same as accessibility.pointDescriptionFormatter, but for an
     * individual series. Overrides the chart wide configuration.
     */
    pointDescriptionFormatter?: Function;
    /**
     * (Highcharts, Highstock, Gantt) If no x values are given for the points in
     * a series, `pointInterval` defines the interval of the x values. For
     * example, if a series contains one value every decade starting from year
     * 0, set `pointInterval` to `10`. In true `datetime` axes, the
     * `pointInterval` is set in milliseconds.
     *
     * It can be also be combined with `pointIntervalUnit` to draw irregular
     * time intervals.
     *
     * Please note that this options applies to the _series data_, not the
     * interval of the axis ticks, which is independent.
     */
    pointInterval?: number;
    /**
     * (Highcharts, Highstock, Gantt) On datetime series, this allows for
     * setting the pointInterval to irregular time units, `day`, `month` and
     * `year`. A day is usually the same as 24 hours, but `pointIntervalUnit`
     * also takes the DST crossover into consideration when dealing with local
     * time. Combine this option with `pointInterval` to draw weeks, quarters, 6
     * months, 10 years etc.
     *
     * Please note that this options applies to the _series data_, not the
     * interval of the axis ticks, which is independent.
     */
    pointIntervalUnit?: OptionsPointIntervalUnitValue;
    /**
     * (Highcharts, Highstock, Gantt) Padding between each column or bar, in x
     * axis units.
     */
    pointPadding?: number;
    /**
     * (Highcharts, Highstock, Gantt) Possible values: `"on"`, `"between"`,
     * `number`.
     *
     * In a column chart, when pointPlacement is `"on"`, the point will not
     * create any padding of the X axis. In a polar column chart this means that
     * the first column points directly north. If the pointPlacement is
     * `"between"`, the columns will be laid out between ticks. This is useful
     * for example for visualising an amount between two points in time or in a
     * certain sector of a polar chart.
     *
     * Since Highcharts 3.0.2, the point placement can also be numeric, where 0
     * is on the axis value, -0.5 is between this value and the previous, and
     * 0.5 is between this value and the next. Unlike the textual options,
     * numeric point placement options won't affect axis padding.
     *
     * Note that pointPlacement needs a pointRange to work. For column series
     * this is computed, but for line-type series it needs to be set.
     *
     * For the `xrange` series type and gantt charts, if the Y axis is a
     * category axis, the `pointPlacement` applies to the Y axis rather than the
     * (typically datetime) X ax